///
/// @file
/// @details An entity within the LudumDare35 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "artificial_fighter_entity.h"
#include "arena_entity.h"

//--------------------------------------------------------------------------------------------------------------------//

const int kPowerAttackChance(40); //0 to 100 for percentage.

LudumDare35::ArtificialFighterEntity::ArtificialFighterEntity(const tbMath::Vector2& spawnPosition, const FighterShape& fighterShape, const FighterState& spawnState) :
	FighterEntity(spawnPosition, fighterShape),
	mState(spawnState),
	mPowerAttack(false),
	mTargetArea(static_cast<FighterTarget>(tbMath::RandomInt() % kMaximumTargets))
{
	ChangeToState(spawnState);
	AddEntityType("Enemy");
}

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::ArtificialFighterEntity::~ArtificialFighterEntity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnAdded(void)
{
	FighterEntity::OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnRemoved(void)
{
	FighterEntity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnSimulate(void)
{
	if (true == IsAlive())
	{
		//TODO: TIM: TurtleBrains: Make a FindNearestEntityByType()
		std::list<Entity*> heroes = GetEntityManager()->GetEntitiesByType("Hero");

		tb_error_if(heroes.size() != 1, "Expected to have 1 and only 1 hero.");

		FighterEntity* targetHero = dynamic_cast<FighterEntity*>(heroes.front());
		tb_error_if(nullptr == targetHero, "Expected to always have a hero.");

		tbMath::Vector2 heroPosition(targetHero->GetPosition());
		tbMath::Vector2 towardHero(heroPosition - GetPosition());
		const float toHeroDistance(towardHero.Normalize());

		const float targetAwayDistance(32.0f);
		tbMath::Vector2 heroTargetPosition(heroPosition);
		switch (mTargetArea)
		{
		case kTargetCenter: break;
		case kTargetForward: heroTargetPosition += targetHero->GetForwardDirection() * targetAwayDistance; break;
		case kTargetBackward: heroTargetPosition -= targetHero->GetForwardDirection() * targetAwayDistance; break;
		case kTargetLeft: heroTargetPosition -= targetHero->GetRightDirection() * targetAwayDistance; break;
		case kTargetRight: heroTargetPosition += targetHero->GetRightDirection() * targetAwayDistance; break;
		};

		mStateTimer.IncrementStep();

		mIsWalkingForward = false;
		mIsWalkingBackward = false;
		mIsStrafingLeft = false;
		mIsStrafingRight = false;
		mIsAttacking = false;
		mAttemptShapeShift = false;

		if (mState == kStateSpawn)
		{
			mIsLockedOnTarget = false;

			if (false == IsSpawning())
			{
				ChangeToState(kStateWandering);
			}
		}
		else if (mState == kStateWandering)
		{
			mIsLockedOnTarget = true;
			mTargetPosition = mWanderingPosition;

			tbMath::Vector2 towardWandering(mWanderingPosition - GetPosition());
			const float toWanderingDistance(towardWandering.Normalize());

			if (toWanderingDistance > 20.0f)
			{
				mIsWalkingForward = true;
			}
			else
			{
				ChangeToState(kStateWandering);
			}

			if (toHeroDistance < 250.0f)
			{	//Pick a target and lock on
				ChangeToState(kStateMove);
			}
		}
		else if (mState == kStateMove)
		{
			mIsLockedOnTarget = true;
			//mTargetPosition = heroPosition;
			mTargetPosition = heroTargetPosition;

			if (toHeroDistance > 80.0f)
			{
				mIsWalkingForward = true;
			}
			else if (toHeroDistance < 60.0f)
			{
				mIsWalkingBackward = true;
			}
			else
			{
				if (mStateTimer > 300)
				{
					ChangeToState(kStateAttack);
				}
			}
		}
		else if (mState == kStateAttack)
		{
			if (false == targetHero->IsAlive())
			{
				ChangeToState(kStateIdle);
			}
			else
			{
				mIsLockedOnTarget = !mPowerAttack;
				mTargetPosition = heroPosition;

				if (mStateTimer > 1000)
				{
					ChangeToState(kStateIdle);
				}
				else if (mStateTimer == 350)
				{
					if (true == mPowerAttack)
					{
						ArtificialHackMoveForward(350, 250.0f);
						mIsAttacking = true;
					}
				}
				else if (mStateTimer > 300)
				{
					if (true == mPowerAttack)
					{
						mIsAttacking = true;
					}
				}
				else if (mStateTimer > 250)
				{
					mIsAttacking = true;
				}
				else if (true == mPowerAttack)
				{
					if (mStateTimer == 100)
					{
						ArtificialHackMoveBackward(150, 25.0f);
					}

					mIsAttacking = true;
					PrepareAttack(mPowerAttack);
				}
				else
				{
					PrepareAttack(mPowerAttack);
				}
			}
		}
		else if (mState == kStateIdle)
		{
			mIsLockedOnTarget = false;

			if (toHeroDistance < 250.0f)
			{	//Pick a target and lock on
				ChangeToState(kStateMove);
			}
		}
	}

	FighterEntity::OnSimulate();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnUpdate(const float deltaTime)
{
	FighterEntity::OnUpdate(deltaTime);
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnRender(void) const
{
	FighterEntity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	FighterEntity::OnCollideWith(otherEntity);
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::ChangeToState(const FighterState& state)
{
	mStateTimer = tbGame::GameTimer(0);
	mState = state;

	switch (state)
	{
	case kStateSpawn: {
		ArtificialHackSetSpawnTimer(1000);
		break; }

	case kStateIdle: {
		mIsLockedOnTarget = false;
		break; }

	case kStateWandering: {
		mWanderingPosition = ArenaEntity::RandomPositionInArena();
		break; }

	case kStateMove: {
		mIsLockedOnTarget = true;
		break; }

	case kStateAttack: {
		mPowerAttack = (tbMath::RandomInt() % 100 >= (100 - kPowerAttackChance)) ? true : false;

		//mPowerAttack = false;
		//switch (GetFighterShape())
		//{
		//case kFighterFire: mPowerAttack = true; break;
		//case kFighterWater: mPowerAttack = false; break;
		//case kFighterEarth: mPowerAttack = (tbMath::RandomInt() % 100 >= 50) ? true : false; break;
		//}

		if (true == mPowerAttack) { printf("POWER ATTACK!\n"); }
		break; }
	};
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::ArtificialFighterEntity::SeparateArtificialFighters(tbGame::EntityManager& entityManager)
{
	const std::list<Entity*> enemies = entityManager.GetEntitiesByType("Enemy");

	const float kSafeRadius(68.0f);

	for (auto outerItr = enemies.begin(), outerEnd = enemies.end(); outerItr != outerEnd; ++outerItr)
	{
		FighterEntity* outerEntity = dynamic_cast<FighterEntity*>(*outerItr);
		tb_error_if(nullptr == outerEntity, "This was unexpected!");
		
		if (false == outerEntity->IsAlive())
		{
			continue;
		}

		tbMath::Vector2 outerPosition = outerEntity->GetPosition();

		for (auto innerItr = enemies.begin(), innerEnd = enemies.end(); innerItr != innerEnd; ++innerItr)
		{
			FighterEntity* innerEntity = dynamic_cast<FighterEntity*>(*innerItr);
			tb_error_if(nullptr == innerEntity, "This was unexpected!");

			tbMath::Vector2 innerPosition = innerEntity->GetPosition();

			if (innerEntity == outerEntity || false == innerEntity->IsAlive())
			{
				continue;
			}

			tbMath::Vector2 directionTowardsOuter(outerPosition - innerPosition);
			const float distanceBetween(directionTowardsOuter.Normalize());
			if (distanceBetween < kSafeRadius)
			{
				float displacement = (kSafeRadius - distanceBetween);
				//TODO: TIM: TurtleBrains: Make a MovePosition? or ChangePosition()
				//Only doing 25% each because this will technically run for both entities twice, see innerItr, not (j = i + 1)
				innerEntity->SetPosition(innerEntity->GetPosition() - directionTowardsOuter * (displacement * 0.25f));
				outerEntity->SetPosition(outerEntity->GetPosition() + directionTowardsOuter * (displacement * 0.25f));
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//
