///
/// @file
/// @details This scene is the gameplay scene for the LudumDare35 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "gameplay_scene.h"
#include "scene_manager.h"
#include "score_display.h"
#include "game_results.h"

#include "arena_entity.h"
#include "shifter_entity.h"
#include "player_fighter_entity.h"
#include "artificial_fighter_entity.h"

#include "turtle_brains/graphics/unstable/tbu_gif_exporter.h"

const tbGame::GameTimer kMaximumFadeInTime(500);

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::GameplayScene::GameplayScene(void) :
	mBlackQuad(tbGraphics::Color::kBlack),
	mWavesSpawned(0),
	mSpawnedThisWave(0),
	mWaitingToSpawnWave(false)
{
	//tbGraphics::Sprite* backdrop(new tbGraphics::Sprite("data/backdrop.png"));
	//backdrop->SetColor(tbGraphics::Color(0x80FFFFFF));
	//AddGraphic(backdrop);

	//TODO: TIM: TurtleBrains: Make it easier to fade alpha from 0 to 1.
	mBlackQuad.SetScroll(0.0f);
	mBlackQuad.SetDepth(1.0f);
	AddGraphic(mBlackQuad);
}

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::GameplayScene::~GameplayScene(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

bool LudumDare35::GameplayScene::IsPlayerAlive(void)
{
	std::list<tbGame::Entity*> heroes = GetEntitiesByType("Hero");
	tb_error_if(heroes.size() != 1, "Expected to have 1 and only 1 hero.");

	FighterEntity* targetHero = dynamic_cast<FighterEntity*>(heroes.front());
	tb_error_if(nullptr == targetHero, "Expected to always have a hero.");

	return targetHero->IsAlive();
}

//--------------------------------------------------------------------------------------------------------------------//

int LudumDare35::GameplayScene::CountEnemiesAlive(void)
{
	std::list<tbGame::Entity*> enemies = GetEntitiesByType("Enemy");

	int aliveEnemies(0);
	for (tbGame::Entity* entity : enemies)
	{
		FighterEntity* fighter(dynamic_cast<FighterEntity*>(entity));
		if (fighter->IsAlive())
		{
			++aliveEnemies;
		}
	}

	return aliveEnemies;
}

//--------------------------------------------------------------------------------------------------------------------//

int LudumDare35::GameplayScene::CountEnemiesAlive(LudumDare35::FighterShape& fighterShape)
{
	return 0;
}

//--------------------------------------------------------------------------------------------------------------------//

bool LudumDare35::GameplayScene::CanSpawnEnemy(void) const
{
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::HandleSpawnLogic(void)
{	//Already checked for Player is alive.
	mMinimumSpawnDelayTimer.DecrementStep();

	bool skipSpawning = false;
	
	if (mWavesSpawned < 4)
	{
		if (CountEnemiesAlive() > 0)
		{
			skipSpawning = true;
			mWaitingToSpawnWave = false;
		}
		else if (false == mWaitingToSpawnWave)
		{
			mMinimumSpawnDelayTimer = 1000;
			mWaitingToSpawnWave = true;
		}
	}
	
	if (false == mMinimumSpawnDelayTimer.IsZero())
	{
		skipSpawning = true;
	}
	
	if (true == skipSpawning)
	{
		return;
	}

	///////////

	if (mWavesSpawned < 4)
	{
		SpawnWaveLogic();
	}
	else
	{
		if (mSpawnedThisWave >= 3)
		{
			SpawnWaveLogic();
		}

		mMinimumSpawnDelayTimer = 1000;

		int maximumEnemies(3);
		int difficulty(mWavesSpawned / 2);
		if (difficulty > 4) { maximumEnemies = 4; }
		if (difficulty > 6) { maximumEnemies = 5; }
		if (difficulty > 9) { maximumEnemies = 6; }

		printf("Spawning Enemy Wave %i,  diff: %i,   maxE: %i,  sp: %i\n", mWavesSpawned, difficulty, maximumEnemies, mSpawnedThisWave);

		if (CountEnemiesAlive() < maximumEnemies)
		{
			SpawnEnemy(difficulty);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

int LudumDare35::GameplayScene::SpawnWaveLogic(void)
{
	++mWavesSpawned;
	mSpawnedThisWave = 0;

	if (1 == mWavesSpawned)
	{
		SpawnEnemy(kFighterFire, ArenaEntity::GetDoorPositionInArena(kDirectionNorth));
		SpawnEnemy(kFighterFire, ArenaEntity::GetDoorPositionInArena(kDirectionNorth) + tbMath::Vector2(100.0f, 0.0f));
		SpawnEnemy(kFighterFire, ArenaEntity::GetDoorPositionInArena(kDirectionNorth) + tbMath::Vector2(-100.0f, 0.0f));
	}
	if (2 == mWavesSpawned)
	{
		SpawnEnemy(kFighterEarth, ArenaEntity::GetDoorPositionInArena(kDirectionSouth));
		SpawnEnemy(kFighterEarth, ArenaEntity::GetDoorPositionInArena(kDirectionSouth) + tbMath::Vector2(100.0f, 0.0f));
		SpawnEnemy(kFighterEarth, ArenaEntity::GetDoorPositionInArena(kDirectionSouth) + tbMath::Vector2(-100.0f, 0.0f));
	}
	if (3 == mWavesSpawned)
	{
		SpawnEnemy(kFighterWater, ArenaEntity::GetDoorPositionInArena(kDirectionEast));
		SpawnEnemy(kFighterWater, ArenaEntity::GetDoorPositionInArena(kDirectionWest));
		SpawnEnemy(kFighterWater);
	}

	return 0;
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::SpawnEnemy(const int difficulty)
{
	std::list<tbGame::Entity*> heroes = GetEntitiesByType("Hero");
	tb_error_if(heroes.size() != 1, "Expected to have 1 and only 1 hero.");
	FighterEntity* targetHero = dynamic_cast<FighterEntity*>(heroes.front());
	tb_error_if(nullptr == targetHero, "Expected to always have a hero.");

	int randomValue(tbMath::RandomInt() % 100);
	FighterShape playerShape(targetHero->GetFighterShape());

	int easyChance = 70;
	int mediumChance = 20;

	if (2 == difficulty) { easyChance = 50;	mediumChance = 30; }
	else if (3 == difficulty) { easyChance = 30;   mediumChance = 50; }
	else if (4 == difficulty) { easyChance = 20;   mediumChance = 50; }
	else if (5 == difficulty) { easyChance = 20;   mediumChance = 40; }
	else if (6 == difficulty) { easyChance = 10;   mediumChance = 40; }
	else if (7 == difficulty) { easyChance = 10;   mediumChance = 30; }
	else if (8 == difficulty) { easyChance = 5;    mediumChance = 30; }
	else if (9 == difficulty) { easyChance = 5;    mediumChance = 20; }
	else if (10 == difficulty) { easyChance = 5;   mediumChance = 10; }
	else if (difficulty > 10) { easyChance = 0;    mediumChance = 0; }

	//Spawn Based on the diffulty above.
	if (randomValue > (100 - easyChance))
	{
		SpawnEnemy(kFighterTable[playerShape][0]);
	}
	else if (randomValue > (100 - (easyChance + mediumChance)))
	{
		SpawnEnemy(kFighterTable[playerShape][1]);
	}
	else //hard
	{
		SpawnEnemy(kFighterTable[playerShape][2]);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::SpawnEnemy(const FighterShape& fighterShape)
{
	SpawnEnemy(fighterShape, ArenaEntity::RandomPositionInArena());
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::SpawnEnemy(const FighterShape& fighterShape, const tbMath::Vector2& spawnPosition)
{
	++mSpawnedThisWave;
	AddEntity(new ArtificialFighterEntity(spawnPosition, fighterShape));
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::SpawnRandomEnemy(void)
{
	switch (tbMath::RandomInt() % 3)
	{
	case kFighterFire: SpawnEnemy(kFighterFire); break;
	case kFighterWater: SpawnEnemy(kFighterWater); break;
	case kFighterEarth: SpawnEnemy(kFighterEarth); break;
	};
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::SpawnOpposingEnemy(void)
{
	std::list<tbGame::Entity*> heroes = GetEntitiesByType("Hero");
	tb_error_if(heroes.size() != 1, "Expected to have 1 and only 1 hero.");

	FighterEntity* targetHero = dynamic_cast<FighterEntity*>(heroes.front());
	tb_error_if(nullptr == targetHero, "Expected to always have a hero.");

	switch (targetHero->GetFighterShape())
	{
	case kFighterFire: SpawnEnemy(kFighterWater); break;
	case kFighterWater: SpawnEnemy(kFighterEarth); break;
	case kFighterEarth: SpawnEnemy(kFighterFire); break;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::OnSimulate(void)
{
	tbGame::GameScene::OnSimulate();

	LudumDare35::ArtificialFighterEntity::SeparateArtificialFighters(*this);

	if (false == mEntryFadeInTimer.IsZero())
	{
		if (true == mEntryFadeInTimer.DecrementStep())
		{
			mBlackQuad.SetVisible(false);
		}
		else
		{
			float p = mEntryFadeInTimer.GetPercentageOf(kMaximumFadeInTime);
			tbGraphics::Color fadeColor;
			fadeColor.SetColor(1.0f - p, tbGraphics::Color(0xFF000000), tbGraphics::Color(0x00000000));
			mBlackQuad.SetColor(fadeColor);
			mBlackQuad.SetVisible(true);
		}

		return;
	}

	if (false == mDeathFadeOutTimer.IsZero())
	{
		mDeathFadeOutTimer.DecrementStep();
		
		if (mDeathFadeOutTimer > 2000)
		{

		}
		else if (mDeathFadeOutTimer > 1000)
		{
			float p = static_cast<float>(mDeathFadeOutTimer.GetElapsedTime() - 1000) / 1000.0f;
			tbGraphics::Color fadeColor;
			fadeColor.SetColor(p, tbGraphics::Color(0xFF000000), tbGraphics::Color(0x00000000));
			mBlackQuad.SetColor(fadeColor);
			mBlackQuad.SetVisible(true);
		}
		else if (mDeathFadeOutTimer > 100)
		{
			mBlackQuad.SetColor(tbGraphics::Color::kBlack);
			mBlackQuad.SetVisible(true);
		}
		else if (mDeathFadeOutTimer.IsZero())
		{
			SceneManager::ChangeToScene(kGameplayScene);
		}

		return;
	}

	if (false == IsPlayerAlive())
	{
		mDeathFadeOutTimer = 3000;
	}
	else
	{
		HandleSpawnLogic();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::OnUpdate(const float deltaTime)
{
	tbGame::GameScene::OnUpdate(deltaTime);

	if (true == tbApplication::Input::IsKeyReleased(tbApplication::tbKeyEscape))
	{	//This should be the very last check to happen as game ends immediately.
		theSceneManager->QuitGame();
	}

	tbGraphics::Unstable::GifPerformCapture(deltaTime, tbApplication::Input::IsKeyPressed(tbApplication::tbKeyG));
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::OnRender(void) const
{
	tbGame::GameScene::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::OnOpen(void)
{
	tbGame::GameScene::OnOpen();

	GameResults::ResetGameResults(false);

	mBlackQuad.SetColor(tbGraphics::Color::kBlack);
	mBlackQuad.SetVisible(true);
	mEntryFadeInTimer = kMaximumFadeInTime;
	mDeathFadeOutTimer = tbGame::GameTimer::kZero;

	AddEntity(new ArenaEntity());
	AddEntity(new LudumDare35::ScoreDisplay());

	AddEntity(new PlayerFighterEntity(ArenaEntity::GetDoorPositionInArena(kDirectionSouth)));

	AddEntity(new ArtificialFighterEntity(ArenaEntity::GetCenterPosition(), kFighterSquare, kStateIdle));
	AddEntity(new ArtificialFighterEntity(ArenaEntity::GetDoorPositionInArena(kDirectionWest), kFighterCircle, kStateIdle));
	AddEntity(new ArtificialFighterEntity(ArenaEntity::GetDoorPositionInArena(kDirectionEast), kFighterTriangle, kStateIdle));
	mMinimumSpawnDelayTimer = tbGame::GameTimer::kZero;

	static auto controller = tbAudio::theAudioManager.PlayEvent("audio_events", "music");

	mWavesSpawned = 0;
	mSpawnedThisWave = 0;
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::GameplayScene::OnClose(void)
{
	RemoveEntities();

	tbGame::GameScene::OnClose();
}

//--------------------------------------------------------------------------------------------------------------------//
