///
/// @file
/// @details An entity within the LudumDare35 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "pickup_entity.h"
#include "fighter_entity.h"
#include "game_results.h"
#include "arena_entity.h"
#include "score_entity.h"

const int kHealthDropChance(30); //0 to 100 % chance of drop.
const tbGame::GameTimer kMaximumLifeTime(3000);
const tbGame::GameTimer kMaximumFlyingTime(750);
const tbGame::GameTimer kMaximumSpawnTime(2000);

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::PickupEntity::PickupEntity(const tbMath::Vector2& spawnPosition, bool isFlying) :
	tbGame::Entity("PickupEntity"),
	mSprite("data/pickup_heart.png"),
	mSpawningTimer(kMaximumSpawnTime),
	mLifeTimer(0),
	mIsHealthPickup(true),
	mPickupAmount(25),
	mEndOfLifeFlash(false),
	mFlyingTimer((true == isFlying) ? kMaximumFlyingTime : 0),
	mStartFlyingPosition(spawnPosition)
{
	RandomizePickup();

	SetPosition(spawnPosition);
	AddGraphic(mSprite);

	AddBoundingCircle(32.0f);
	SetCollidable(false);
}

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::PickupEntity::~PickupEntity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::RandomizePickup(void)
{
	mIsHealthPickup = (tbMath::RandomInt() % 100 >= (100 - kHealthDropChance)) ? true : false;

	if (true == mIsHealthPickup)
	{
		mPickupAmount = 25;
		mSprite = tbGraphics::Sprite("data/pickup_heart.png");
	}
	else
	{
		mPickupAmount = 250;
		mSprite = tbGraphics::Sprite("data/pickup_coin.png");
	}

	mFlyingTimer = kMaximumFlyingTime;
	mFinalFlyingPosition = ArenaEntity::RandomPositionInArena();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnPickup(FighterEntity& fighter)
{
	if (true == mIsHealthPickup)
	{
		fighter.GetHealthBar().Heal(mPickupAmount);
		tbAudio::theAudioManager.PlayEvent("audio_events", "pickup_health");
	}
	else
	{
		tbAudio::theAudioManager.PlayEvent("audio_events", "pickup_coin");
		GameResults::IncrementCurrentScore(mPickupAmount);

		const tbMath::Vector2 randomPosition = ArenaEntity::RandomPositionNear(GetPosition(), 64.0f);
		GetEntityManager()->AddEntity(new ScoreEntity(randomPosition, mPickupAmount));
	}

	mSprite.SetVisible(false);
	SetCollidable(false);
	GetEntityManager()->RemoveEntity(this);
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnAdded(void)
{
	tbGame::Entity::OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnRemoved(void)
{
	tbGame::Entity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnSimulate(void)
{
	tbGame::Entity::OnSimulate();

	if (false == mFlyingTimer.IsZero())
	{
		mFlyingTimer.DecrementStep();

		tbMath::Vector2 towardFinal(mFinalFlyingPosition - mStartFlyingPosition);
		//tbMath::Vector2 center(mStartFlyingPosition + (towardFinal * 0.5f));

		float percentage(1.0f - mFlyingTimer.GetPercentageOf(kMaximumFlyingTime));
		SetPosition(mStartFlyingPosition + (towardFinal * percentage));

		tbGraphics::Color spriteColor(0x80FFFFFF);
		mSprite.SetColor(spriteColor);

		return;
	}

	if (false == mSpawningTimer.IsZero())
	{
		tbGraphics::Color spriteColor(0x80FFFFFF);
		mSprite.SetColor(spriteColor);

		if (true == mSpawningTimer.DecrementStep())
		{	//TODO: TIM: TurtleBrains: This is probably okay to be implicit!
			mSprite.SetColor(tbGraphics::Color(0xFFFFFFFF));
			SetCollidable(true);
		}
	}

	if (true == CanPickup())
	{
		mLifeTimer.IncrementStep();
		if (mLifeTimer > (kMaximumLifeTime.GetElapsedTime() - 1500))
		{
			if (mLifeTimer.GetElapsedTime() % 100)
			{
				mEndOfLifeFlash = !mEndOfLifeFlash;

				if (mEndOfLifeFlash)
				{
					mSprite.SetVisible(false);
				}
				else
				{
					mSprite.SetVisible(true);
				}
			}

			if (mLifeTimer > kMaximumLifeTime)
			{
				GetEntityManager()->RemoveEntity(this);
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnUpdate(const float deltaTime)
{
	tbGame::Entity::OnUpdate(deltaTime);
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnRender(void) const
{
	tbGame::Entity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::PickupEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	tbGame::Entity::OnCollideWith(otherEntity);
}

//--------------------------------------------------------------------------------------------------------------------//
