
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_linux

#include "../tbi_realtime_system_application.h"
#include "../../tb_realtime_application.h"
#include "../../../core/tb_defines.h"
#include "../../../core/tb_error.h"
#include "../../../core/tb_opengl.h"
#include "../../tb_application_handler_interface.h"
#include "../../tb_application_window.h"
#include "../tbi_system_application_input.h"

#include "../../../graphics/implementation/tbi_renderer.h"

#include <X11/X.h>
#include <X11/Xlib.h>
#include <GL/glx.h>

#include <map>

//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	tbApplication::ApplicationHandlerInterface* tbiApplicationHandler(nullptr);

	//void FixWorkingDirectoryAsNeeded(void);
	
	typedef std::map<KeySym, tbApplication::Key> KeyTable;
	KeyTable tbiVirtualCodesToKey;

	void SetupKeyTable(void);
	void ProcessXEvent(XEvent& event);
	
	//TODO: TIM: This is now shared on Windows and Linux, it should be refactored to a more common base.
	enum GLContextType { kContextCore, kContextCompatible, kContextES, };
	struct GLVersion
	{
		GLContextType mType;
		int mMajor;
		int mMinor;
		
		GLVersion(int major, int minor, const GLContextType& type) :
			mType(type),
			mMajor(major),
			mMinor(minor)
		{
		}
	};
	
	bool FindBestFBC(GLXFBConfig* bestFrameBufferConfiguration, const int* visualAttributes);
	bool TryGLContext(const GLVersion& glVersion, const GLXFBConfig& bestFrameBufferConfiguration);
	
	bool tbiIsRunning(false);
	Display* tbiDisplay(nullptr);
	Window tbiWindow;
	int tbiScreen(0);
	
	GLXContext tbiGraphicsContext;
	XVisualInfo* tbiVisualInfo;

};  /* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetupKeyTable(void)
{
	tbiVirtualCodesToKey.clear();

	tbiVirtualCodesToKey[XK_0] = tbApplication::tbKey0;
	tbiVirtualCodesToKey[XK_1] = tbApplication::tbKey1;
	tbiVirtualCodesToKey[XK_2] = tbApplication::tbKey2;
	tbiVirtualCodesToKey[XK_3] = tbApplication::tbKey3;
	tbiVirtualCodesToKey[XK_4] = tbApplication::tbKey4;
	tbiVirtualCodesToKey[XK_5] = tbApplication::tbKey5;
	tbiVirtualCodesToKey[XK_6] = tbApplication::tbKey6;
	tbiVirtualCodesToKey[XK_7] = tbApplication::tbKey7;
	tbiVirtualCodesToKey[XK_8] = tbApplication::tbKey8;
	tbiVirtualCodesToKey[XK_9] = tbApplication::tbKey9;

	tbiVirtualCodesToKey[XK_a] = tbApplication::tbKeyA;
	tbiVirtualCodesToKey[XK_b] = tbApplication::tbKeyB;
	tbiVirtualCodesToKey[XK_c] = tbApplication::tbKeyC;
	tbiVirtualCodesToKey[XK_d] = tbApplication::tbKeyD;
	tbiVirtualCodesToKey[XK_e] = tbApplication::tbKeyE;
	tbiVirtualCodesToKey[XK_f] = tbApplication::tbKeyF;
	tbiVirtualCodesToKey[XK_g] = tbApplication::tbKeyG;
	tbiVirtualCodesToKey[XK_h] = tbApplication::tbKeyH;
	tbiVirtualCodesToKey[XK_i] = tbApplication::tbKeyI;
	tbiVirtualCodesToKey[XK_j] = tbApplication::tbKeyJ;
	tbiVirtualCodesToKey[XK_k] = tbApplication::tbKeyK;
	tbiVirtualCodesToKey[XK_l] = tbApplication::tbKeyL;
	tbiVirtualCodesToKey[XK_m] = tbApplication::tbKeyM;
	tbiVirtualCodesToKey[XK_n] = tbApplication::tbKeyN;
	tbiVirtualCodesToKey[XK_o] = tbApplication::tbKeyO;
	tbiVirtualCodesToKey[XK_p] = tbApplication::tbKeyP;
	tbiVirtualCodesToKey[XK_q] = tbApplication::tbKeyQ;
	tbiVirtualCodesToKey[XK_r] = tbApplication::tbKeyR;
	tbiVirtualCodesToKey[XK_s] = tbApplication::tbKeyS;
	tbiVirtualCodesToKey[XK_t] = tbApplication::tbKeyT;
	tbiVirtualCodesToKey[XK_u] = tbApplication::tbKeyU;
	tbiVirtualCodesToKey[XK_v] = tbApplication::tbKeyV;
	tbiVirtualCodesToKey[XK_w] = tbApplication::tbKeyW;
	tbiVirtualCodesToKey[XK_x] = tbApplication::tbKeyX;
	tbiVirtualCodesToKey[XK_y] = tbApplication::tbKeyY;
	tbiVirtualCodesToKey[XK_z] = tbApplication::tbKeyZ;

	tbiVirtualCodesToKey[XK_space] = tbApplication::tbKeySpace;
	tbiVirtualCodesToKey[XK_Escape] = tbApplication::tbKeyEscape;
	tbiVirtualCodesToKey[XK_Return] = tbApplication::tbKeyEnter;
	tbiVirtualCodesToKey[XK_Up] = tbApplication::tbKeyUp;
	tbiVirtualCodesToKey[XK_Down] = tbApplication::tbKeyDown;
	tbiVirtualCodesToKey[XK_Left] = tbApplication::tbKeyLeft;
	tbiVirtualCodesToKey[XK_Right] = tbApplication::tbKeyRight;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetMousePosition(int mouseX, int mouseY)
{
	tb_error_if(nullptr == tbiDisplay, "tbExternalError: Cannot SetMousePosition without a running application.")
	XWarpPointer(tbiDisplay, None, tbiWindow, 0, 0, 0, 0, mouseX, mouseY);
}

//--------------------------------------------------------------------------------------------------------------------//

// - (void)applicationWillTerminate:(NSNotification *)notification
// {
// //	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
// //	tbImplementation::Renderer::CleanupBasicRenderer();
// //	tbImplementation::tbiApplicationHandler->OnWindowClose();
// }

// //--------------------------------------------------------------------------------------------------------------------//

// - (void)windowWillClose:(NSNotification *)notification
// {
// 	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
// 	tbImplementation::Renderer::CleanupBasicRenderer();
// 	tbImplementation::tbiApplicationHandler->OnWindowClose();

// 	mWindow = nil;
// }

// //--------------------------------------------------------------------------------------------------------------------//

// - (void)applicationDidBecomeActive:(NSNotification *)notification
// //- (void)windowDidBecomeKey:(NSNotification *)notification
// {
// 	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
// 	tbImplementation::tbiApplicationHandler->OnBecomeActive();
// }

// //--------------------------------------------------------------------------------------------------------------------//

// - (void)applicationDidResignActive:(NSNotification *)notification
// //- (void)windowDidResignKey:(NSNotification *)notification
// {
// 	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
// 	tbImplementation::tbiApplicationHandler->OnBecomeInactive();
// }

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::FindBestFBC(GLXFBConfig* bestFrameBufferConfiguration, const int* visualAttributes)
{
	int bestFBC = -1;
	int worstFBC = -1;
	int bestSamples = -1;
	int worstSamples = 999;
	
	int glxMajor(0);
	int glxMinor(0);
	if (false == glXQueryVersion(tbiDisplay, &glxMajor, &glxMinor))
	{
		tbi_log_warning("Unable to query gl version with glXQueryVersion.");
		return false;
	}
	
	if (glxMajor < 1 || (glxMajor == 1 && glxMinor < 3))
	{
		tbi_log_warning("FrameBuffer Configurations were added in GLX version 1.3.");
		return false;
	}
	
	int frameBufferCount(0);
	GLXFBConfig* frameBufferConfigs = glXChooseFBConfig(tbiDisplay, DefaultScreen(tbiDisplay), visualAttributes, &frameBufferCount);
	
	for (int i = 0; i < frameBufferCount; ++i)
	{
		XVisualInfo* visualInfo = glXGetVisualFromFBConfig(tbiDisplay, frameBufferConfigs[i]);
		if (nullptr != visualInfo)
		{
			int sampleBuffers(0);
			glXGetFBConfigAttrib(tbiDisplay, frameBufferConfigs[i], GLX_SAMPLE_BUFFERS, &sampleBuffers);
			
			int samples(0);
			glXGetFBConfigAttrib(tbiDisplay, frameBufferConfigs[i], GLX_SAMPLES, &samples);
			
			if (bestFBC < 0 || sampleBuffers && samples > bestSamples)
			{
				bestFBC = i;
				bestSamples = samples;
			}
			if (worstFBC < 0 || !sampleBuffers || samples < worstSamples)
			{
				worstFBC = i;
				worstSamples = samples;
			}
		}
		XFree(visualInfo);
	}
	
	*bestFrameBufferConfiguration = frameBufferConfigs[bestFBC];
	XFree(frameBufferConfigs);
	
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

typedef GLXContext (*glXCreateContextAttribsARBProc)(Display*, GLXFBConfig, GLXContext, Bool, const int*);

bool tbImplementation::TryGLContext(const GLVersion& glVersion, const GLXFBConfig& bestFrameBufferConfiguration)
{
//	const char* glxExtensions = glXQueryExtensionsString(tbiDisplay, DefaultScreen(tbiDisplay));
//	glXCreateContextAttribsARBProc glXCreateContextAttribsARB = (glXCreateContextAttribsARBProc)glXGetProcAddressARB(reinterpret_cast<const GLubyte*>("glXCreateContextAttribsARB"));
	
	int contextProfileBit(GLX_CONTEXT_CORE_PROFILE_BIT_ARB);
	if (kContextCompatible == glVersion.mType)
	{
		contextProfileBit = GLX_CONTEXT_COMPATIBILITY_PROFILE_BIT_ARB;
	}
	else if (kContextES == glVersion.mType)
	{
		tb_error("tbInternalError: This context is unsupported at the present time.");
	}
	
	int contextAttributes[] = {
		GLX_CONTEXT_PROFILE_MASK_ARB, contextProfileBit,
		GLX_CONTEXT_MAJOR_VERSION_ARB, glVersion.mMajor,
		GLX_CONTEXT_MINOR_VERSION_ARB, glVersion.mMinor,
		GLX_NONE
	};
	
	tb_error_if(!GLX_ARB_create_context, "tbInternalError: Was unable to confirm GLX_ARB_create_context extension for OpenGL 3.2 context.");
	tb_error_if(!GLX_ARB_create_context_profile, "tbInternalError: Was unable to confirm GLX_ARB_create_context_profile extension for OpenGL 3.2 context.");
	
	if (nullptr != glXCreateContextAttribsARB)
	{
		GLXContext newContext = glXCreateContextAttribsARB(tbiDisplay, bestFrameBufferConfiguration, 0, True, contextAttributes);
		if (newContext)
		{
			glXMakeCurrent(tbiDisplay, tbiWindow, newContext);
			glXDestroyContext(tbiDisplay, tbiGraphicsContext);
			tbiGraphicsContext = newContext;
			
			return true;
		}
	}
	else
	{
		tbi_log_warning("tbInternalError: Was unable to fund/use glXCreateContextAttribsARB function.");
	}
	
	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::ProcessXEvent(XEvent& event)
{
	switch (event.type)
	{
	case KeyPress: {
		XComposeStatus composeStatus;
		
		const size_t kBufferSize(32);
		char buffer[kBufferSize];
		KeySym keySymbol;
		XLookupString(&event.xkey, buffer, kBufferSize, &keySymbol, &composeStatus);

		KeyTable::const_iterator keyIter = tbiVirtualCodesToKey.find(keySymbol);
		if (keyIter != tbiVirtualCodesToKey.end())
		{
			tbImplementation::OnPressKey(keyIter->second);
		}
		
		break; }
		
	case KeyRelease: {
		XComposeStatus composeStatus;
		
		const size_t kBufferSize(32);
		char buffer[kBufferSize];
		KeySym keySymbol;
		XLookupString(&event.xkey, buffer, kBufferSize, &keySymbol, &composeStatus);

		KeyTable::const_iterator keyIter = tbiVirtualCodesToKey.find(keySymbol);
		if (keyIter != tbiVirtualCodesToKey.end())
		{
			tbImplementation::OnReleaseKey(keyIter->second);
		}
		
		break; }
		
	case MotionNotify: {
		tbImplementation::UpdateMousePosition(event.xmotion.x, event.xmotion.y);
		break; }
	
	case ButtonPress: {
		if (Button1 == event.xbutton.button)
		{
			tbImplementation::OnPressKey(tbApplication::tbMouseLeft);
		}
		else if (Button2 == event.xbutton.button)
		{
			tbImplementation::OnPressKey(tbApplication::tbMouseMiddle);
		}
		else if (Button3 == event.xbutton.button)
		{
			tbImplementation::OnPressKey(tbApplication::tbMouseRight);
		}
		break; }
	
	case ButtonRelease: {
		if (Button1 == event.xbutton.button)
		{
			tbImplementation::OnReleaseKey(tbApplication::tbMouseLeft);
		}
		else if (Button2 == event.xbutton.button)
		{
			tbImplementation::OnReleaseKey(tbApplication::tbMouseMiddle);
		}
		else if (Button3 == event.xbutton.button)
		{
			tbImplementation::OnReleaseKey(tbApplication::tbMouseRight);
		}
		break; }
	};
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OpenRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(nullptr != tbiApplicationHandler, "tbExternalError: TurtleBrains currently supports only a single application at a time.");
	tb_error_if(nullptr != tbiDisplay, "tbInternalError: Display should have been nullptr if no application is created.");
	//tb_error_if(nullptr != tbiWindow, "tbInternalError: Window should have been nullptr if no application is created.");

	tbiApplicationHandler = &applicationHandler;

	//Create and setup default window properties but allow the programmer to change them if desired.
	tbApplication::WindowProperties properties;
	tbImplementation::tbiApplicationHandler->CollectWindowProperties(properties);

	//Resources used to get OpenGL Window Created:
	//www.rosettacode.org/wiki/Window_creation/X11
	//www.opengl.org/wiki/Programming_OpenGL_in_Linux:_GLX_and_Xlib
	//www.opengl.org/wiki/Tutorial:_OpenGL_3.0_Context_Creation_(GLX)
	tbiDisplay = XOpenDisplay(NULL);
	if (nullptr == tbiDisplay)
	{
		tb_error("tbInternalError: Could not open display with XOpenDisplay()");
		return;
	}
	
	int attributes[] =
	{
		GLX_RGBA,
		GLX_DOUBLEBUFFER,
		GLX_DEPTH_SIZE, 24,
		None
	};
	
	tbiVisualInfo = glXChooseVisual(tbiDisplay, 0, attributes);
	
	if (nullptr == tbiVisualInfo)
	{	//It may be possible to choose a different glXChooseVisual call with different depth buffer etc.
		tb_error("tbInternalError: No appropriate visual information found.");
		return;
	}
	
	Window desktopWindow(DefaultRootWindow(tbiDisplay));
	Colormap colorMap = XCreateColormap(tbiDisplay, desktopWindow, tbiVisualInfo->visual, AllocNone);
	XSetWindowAttributes windowAttributes;
	windowAttributes.colormap = colorMap;
	windowAttributes.event_mask = KeyPressMask | KeyReleaseMask | ButtonPressMask | ButtonReleaseMask | PointerMotionMask;
	
	//TODO: TIM: Double check reference documentation on all the following...
	tbiScreen = DefaultScreen(tbiDisplay);
	tbiWindow = XCreateWindow(tbiDisplay, desktopWindow, properties.mWindowPositionX, properties.mWindowPositionY,
		properties.mWindowWidth, properties.mWindowHeight, 0, tbiVisualInfo->depth, InputOutput, tbiVisualInfo->visual,
		CWColormap | CWEventMask, &windowAttributes);
	
	if (!tbiWindow)
	{
		tb_error("tbInternalError: Failed to create window with XCreateWindow()");
		return;
	}

	XMapWindow(tbiDisplay, tbiWindow);
	XStoreName(tbiDisplay, tbiWindow, "TurtleBrains Project");

	//Window is now created, lets create the OpenGL context!
	tbiGraphicsContext = glXCreateContext(tbiDisplay, tbiVisualInfo, NULL, GL_TRUE);
	if (!tbiGraphicsContext)
	{
		tb_error("tbInternalError: Failed to create OpenGL Context");
		return;
	}
	
	glXMakeCurrent(tbiDisplay, tbiWindow, tbiGraphicsContext);
	glewExperimental = 0;
	GLenum glewCode = glewInit();
	if (GLEW_OK != glewCode)
	{
		tb_error("tbInternalError: Could not initialize glew. Error code: %s", tb_string(glewCode).c_str());
		return;
	}
	
	bool setupContext(false);
	
#if !defined(tb_legacy_gl_forced)
	GLXFBConfig bestFrameBufferConfiguration;
	FindBestFBC(&bestFrameBufferConfiguration, attributes);

	std::vector<GLVersion> versions;
	versions.push_back(GLVersion(3, 2, kContextCore));
	versions.push_back(GLVersion(3, 2, kContextCompatible));
	
	for (auto itr = versions.begin(), itrEnd = versions.end(); itr != itrEnd && false == setupContext; ++itr)
	{
		if (true == TryGLContext(*itr, bestFrameBufferConfiguration))
		{
			setupContext = true;
		}
	}
#endif /* tb_legacy_gl_forced */

	if (false == setupContext)
	{
#if defined(tb_legacy_gl_support)
		tbi_log_warning("tbWarning: Unable to setup an OpenGL 3.2 Context, using Legacy OpenGL which may not be fully supported.\n");
		tbImplementation::Renderer::SetLegacyRenderer(true);
		setupContext = true;
#else
		tb_error("tbInternalError: Failed to Create OpenGL 3.2 Context.");
#endif
	}

	glewCode = glewInit();
	if (GLEW_OK != glewCode)
	{
		tb_error("tbInternalError: Could not reinitialize glew. Error code: %s", tb_string(glewCode).c_str());
		return;
	}

	SetupKeyTable();
	tbImplementation::Renderer::InitializeBasicRenderer();
	
	tbiApplicationHandler->OnWindowOpen();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::RunRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: ApplicationHandler attempted to change between Open and Run.");
	tb_error_if(nullptr == tbiDisplay, "tbInternalError: Expected valid display pointer.");
	 
	tbiIsRunning = true;
	 
	XEvent event;
	while (true == tbiIsRunning)
	{
		tbImplementation::StartNewFrame();
		 
		while (0 != XPending(tbiDisplay))
		{
			XNextEvent(tbiDisplay, &event);
			ProcessXEvent(event);
		}
		
		if (true == tbiIsRunning)
		{
			tbiApplicationHandler->OnRealtimeUpdate();
			
			if (true == tbiIsRunning)
			{	//It may have changed during Update, should refactor this a bit for safer shutdown experience.
				glXSwapBuffers(tbiDisplay, tbiWindow);
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CloseRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface &applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: ApplicationHandler attempted to change between Open and Close.");
	tbiApplicationHandler->OnWindowClose();
	
	tbImplementation::Renderer::CleanupBasicRenderer();
	 
	glXMakeCurrent(tbiDisplay, tbiWindow, NULL);
	glXDestroyContext(tbiDisplay, tbiGraphicsContext);
	tbiGraphicsContext = nullptr;
	XDestroyWindow(tbiDisplay, tbiWindow);
	XCloseDisplay(tbiDisplay);
	tbiDisplay = nullptr;
	tbiIsRunning = false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowTitle(const tbCore::tbString& windowTitle)
{
	tb_error_if(nullptr == tbiApplicationHandler, "tbExternalError: This cannot be called until after the Window/Application is opened.");
	tb_error_if(nullptr == tbiDisplay, "tbInternalError: Window is open without a display?");

	XStoreName(tbiDisplay, tbiWindow, tbCore::ToStdString(windowTitle).c_str());
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_linux */
