///
/// @file
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tbi_system_application_dialog.h"
#include "../../core/tb_error.h"

namespace tbImplementation
{
#if defined(tb_windows)
	//Making an assumption that winapi libraries are ready to use in "static" space...
	const int kControlBaseWidth(280);
//	const int kControlBaseHeight(HIWORD(GetDialogBaseUnits()) + (HIWORD(GetDialogBaseUnits()) / 2));
	const int kControlBaseHeight(25);
	const int kControlPaddingX(kControlBaseWidth  / 20);	//5% controlwidth
	const int kControlPaddingY(10);
#elif defined(tb_macosx)
	const int kControlBaseWidth(280);
	const int kControlBaseHeight(25);
	const int kControlPaddingX(kControlBaseWidth  / 20);	//5% controlwidth
	const int kControlPaddingY(10);
#elif defined(tb_linux)
	const int kControlBaseWidth(280);
	const int kControlBaseHeight(25);
	const int kControlPaddingX(kControlBaseWidth  / 20);	//5% controlwidth
	const int kControlPaddingY(10);	
#else
	#error This configuration is currently unsupported by TurtleBrains.
#endif /* tb_configuration */

	const size_t kMaximumDisplayedDropdownItems(10);
	const size_t kMaximumAvailableDropdownItems(64);

	std::set<tbApplication::DialogControlIdentifier> tbiControlsSafeForDuplication;

	struct CommonDialogInformation
	{
		std::vector<DialogControl> mControls;
		tbApplication::DialogBucketType mBucketType;
		int mNumberOfVisibleSlots;
		bool mAllowVerticalScrolling;

		CommonDialogInformation(void) :
			mBucketType(tbApplication::kSingleBucket),
			mNumberOfVisibleSlots(5),
			mAllowVerticalScrolling(false)
		{
		}
	};

	typedef std::map<tbApplication::DialogIdentifier, CommonDialogInformation> DialogTable;
	DialogTable tbiControlTable;

	const tbCore::tbString kValueDelimiter(tbCore::ToString(":"));

};	/* namespace tbImplementation */


//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::ResizeDialog(const tbApplication::DialogIdentifier& dialogIdentifier,
	const tbApplication::DialogBucketType& bucketType, const int numberOfVisibleSlots, const bool allowVerticalScrolling)
{	//This will have no effect once OpenDialog has been called.
	CommonDialogInformation& dialogInformation = tbiControlTable[dialogIdentifier];
	dialogInformation.mBucketType = bucketType;
	dialogInformation.mNumberOfVisibleSlots = numberOfVisibleSlots;
	dialogInformation.mAllowVerticalScrolling = allowVerticalScrolling;
}

//--------------------------------------------------------------------------------------------------------------------//

tbApplication::DialogBucketType tbImplementation::GetDialogBucketType(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Dialog not found for identifier: %d", dialogIdentifier);
	return dialogItr->second.mBucketType;
}

//--------------------------------------------------------------------------------------------------------------------//

int tbImplementation::GetDialogNumberOfVisibleSlots(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Dialog not found for identifier: %d", dialogIdentifier);
	return dialogItr->second.mNumberOfVisibleSlots;
}

//--------------------------------------------------------------------------------------------------------------------//

int tbImplementation::GetDialogNumberOfFilledSlots(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Dialog not found for identifier: %d", dialogIdentifier);

	int filledSlots(0);

	for (std::vector<DialogControl>::iterator controlItr = dialogItr->second.mControls.begin(), 
		controlEnd = dialogItr->second.mControls.end(); controlItr != controlEnd; ++controlItr)
	{
		if (controlItr->mSlotInBucket >= filledSlots)
		{
			filledSlots = controlItr->mSlotInBucket + 1;
		}
	}

	return filledSlots;
}

//--------------------------------------------------------------------------------------------------------------------//

int tbImplementation::GetDialogNumberOfFilledSlots(const tbApplication::DialogIdentifier& dialogIdentifier, const tbApplication::DialogControlGuide& forGuide)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Dialog not found for identifier: %d", dialogIdentifier);

	int filledSlots(0);

	for (std::vector<DialogControl>::iterator controlItr = dialogItr->second.mControls.begin(), 
		controlEnd = dialogItr->second.mControls.end(); controlItr != controlEnd; ++controlItr)
	{
		if (controlItr->mGuide == forGuide && controlItr->mSlotInBucket >= filledSlots)
		{
			filledSlots = controlItr->mSlotInBucket + 1;
		}
	}

	return filledSlots;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::IsControlSafeForDuplication(const tbApplication::DialogControlIdentifier& controlIdentifier)
{
	return (tbiControlsSafeForDuplication.find(controlIdentifier) != tbiControlsSafeForDuplication.end()) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetControlSafeForDuplication(const tbApplication::DialogControlIdentifier& controlIdentifier)
{
	tbiControlsSafeForDuplication.insert(controlIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::DialogControl* tbImplementation::CreateDialogControl(const tbApplication::DialogIdentifier& dialogIdentifier,
	const tbApplication::DialogControlIdentifier& controlIdentifier, const tbImplementation::DialogControlType& controlType,
	const tbCore::tbString& controlValue, const tbApplication::DialogControlGuide& guide)
{
	if (false == IsControlSafeForDuplication(controlIdentifier))
	{	//Make sure not to duplicate controls that the programmer does not want duplicated.
		const bool hasControl((nullptr == GetDialogControl(dialogIdentifier, controlIdentifier)) ? false : true);
		tb_error_if(true == hasControl, "tbExternalError: controlIdentifier(%d) already exists and is not safe to duplicate.", controlIdentifier);
	}

	DialogControl newControl;
	newControl.mIdentifier = controlIdentifier;
	newControl.mType = controlType;
	newControl.mValue = controlValue;
	newControl.mLeftEdgeOffset = 0;
	newControl.mWidthPercentage = 100;
	newControl.mGuide = guide;
	newControl.mSlotInBucket = GetDialogNumberOfFilledSlots(dialogIdentifier, newControl.mGuide);
	newControl.mIsVisible = true;
	newControl.mIsEnabled = true;
	tbiControlTable[dialogIdentifier].mControls.push_back(newControl);
	return &tbiControlTable[dialogIdentifier].mControls.back();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::DestroyDialogControls(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	if (dialogItr != tbiControlTable.end())
	{
		dialogItr->second.mControls.clear();
		tbiControlTable.erase(dialogItr);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::DialogControl* tbImplementation::GetDialogControl(const tbApplication::DialogIdentifier& dialogIdentifier,
	const tbApplication::DialogControlIdentifier& controlIdentifier)
{
	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);

	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	if (dialogItr == tbiControlTable.end())
	{
		return nullptr;
	}

	for (std::vector<DialogControl>::iterator controlItr = dialogItr->second.mControls.begin(), 
		controlEnd = dialogItr->second.mControls.end(); controlItr != controlEnd; ++controlItr)
	{
		if (controlIdentifier == controlItr->mIdentifier)
		{
			return &(*controlItr);
		}
	}

	return nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::DialogControl* tbImplementation::GetDialogControlByIndex(const tbApplication::DialogIdentifier& dialogIdentifier,
	const size_t& controlIndex)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Trying to count controls for dialog that does not exist: %d", dialogIdentifier);
	return &dialogItr->second.mControls[controlIndex];
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbImplementation::NumberOfControlsForDialog(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	DialogTable::iterator dialogItr = tbiControlTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiControlTable.end(), "tbInternalError: Trying to count controls for dialog that does not exist: %d", dialogIdentifier);
	return dialogItr->second.mControls.size();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::GetDialogArea(AreaRectangle& visibleRectangle, int& filledHeight, const tbApplication::DialogIdentifier& dialogIdentifier)
{
	const tbApplication::DialogBucketType bucketType(GetDialogBucketType(dialogIdentifier));
	const int numberOfFilledSlots(GetDialogNumberOfFilledSlots(dialogIdentifier));
	const int numberOfVisibleSlots(GetDialogNumberOfVisibleSlots(dialogIdentifier));

	//	const bool isScrolling = (numberOfVisibleSlots < numberOfFilledSlots) ? true : false;
	visibleRectangle.mWidth = (kControlPaddingX) + ((kControlBaseWidth + kControlPaddingX) * (bucketType + 1));
	visibleRectangle.mHeight = (kControlPaddingY * 2) + ((kControlBaseHeight + kControlPaddingY) * numberOfVisibleSlots);
	filledHeight = (kControlPaddingY * 2) + ((kControlBaseHeight + kControlPaddingY) * numberOfFilledSlots);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::GetDialogControlArea(AreaRectangle& rectangle, const DialogControl& dialogControl)
{
	const int kControlWidth(kControlBaseWidth);
	const int kControlHeight(kControlBaseHeight);

	rectangle.mTop = kControlPaddingY + ((kControlHeight + kControlPaddingY) * dialogControl.mSlotInBucket);
	const int leftEdgeOffset = static_cast<int>(static_cast<float>(kControlWidth) * static_cast<float>(dialogControl.mLeftEdgeOffset) / 100.0f);
	rectangle.mLeft = kControlPaddingX + ((kControlWidth + kControlPaddingX) * dialogControl.mGuide) + leftEdgeOffset;
	rectangle.mWidth = static_cast<int>(static_cast<float>(kControlWidth) * static_cast<float>(dialogControl.mWidthPercentage) / 100.0f);
	rectangle.mHeight = kControlBaseHeight;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbImplementation::CheckboxToString(bool isChecked, const tbCore::tbString& name)
{
	tb_error_if(true == name.empty(), "tbExternalError: Expected controlName to be a valid, non-empty string.");
	return tbCore::ToString((true == isChecked) ? "1" : "0") + name;
}

//--------------------------------------------------------------------------------------------------------------------//
void tbImplementation::StringToCheckbox(const tbCore::tbString& input, bool& isChecked, tbCore::tbString& name)
{
	tb_error_if(input.length() < 2, "tbInternalError: Expected input length to contain a char and a string.");
	isChecked = ('0' == input[0]) ? false : true;
	name = &input[1];
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbImplementation::StringToCheckboxName(const tbCore::tbString& input)
{
	bool throwAway = false;
	tbCore::tbString output;
	StringToCheckbox(input, throwAway, output);
	return output;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbImplementation::DropdownToString(int selectedIndex, const std::vector<tbCore::tbString>& values)
{	//TODO: TIM: Change the input type of controlValues to be any std::container type...
	tbCore::tbString output = tbCore::ToString(selectedIndex);

	for (size_t index = 0; index < values.size(); ++index)
	{
		tb_error_if(true == values[index].empty(), "tbExternalError: Expected each value to be a valid, non empty string.");
		tb_error_if(tbCore::tbString::npos != values[index].find(tbImplementation::kValueDelimiter), "tbInternalError with input of dropdown control values, don't use : in values.");
		output += tbImplementation::kValueDelimiter;
		output += values[index];
	}

	return output;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::StringToDropdown(const tbCore::tbString& input, int& selectedIndex, std::vector<tbCore::tbString>& values)
{
	values.clear();

	const size_t delimiterLength = kValueDelimiter.size();
	size_t delimiterIndex = input.find_first_of(kValueDelimiter);
	tb_error_if(delimiterIndex == tbCore::tbString::npos, "tbInternalError: Expected to find delimiter to separate selectedIndex.");
	tbCore::tbString selectedIndexString(input.substr(0, delimiterIndex));
	selectedIndex = tbCore::FromString<int>(selectedIndexString);

	size_t previousDelimiterIndex = delimiterIndex + delimiterLength;
	delimiterIndex = input.find_first_of(kValueDelimiter, previousDelimiterIndex);
	tb_error_if(delimiterIndex == tbCore::tbString::npos, "tbInternalError: Expected to find delimiter to separate first dropdown option.");
	while (tbCore::tbString::npos != delimiterIndex)
	{
		values.push_back(input.substr(previousDelimiterIndex, delimiterIndex - previousDelimiterIndex));
		tb_error_if(values.size() > kMaximumAvailableDropdownItems, "tbInternalError: Expected dropdown size to be less than maximum available items: %d", kMaximumAvailableDropdownItems);
		previousDelimiterIndex = delimiterIndex + delimiterLength;
		delimiterIndex = input.find_first_of(kValueDelimiter, previousDelimiterIndex);
	}
	//Don't forget to add the final value!
	values.push_back(input.substr(previousDelimiterIndex, tbCore::tbString::npos));
}

//--------------------------------------------------------------------------------------------------------------------//
