///
/// @file
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tbi_system_application_menu.h"
#include "../../core/tb_error.h"

#include <map>
#include <set>

namespace tbImplementation
{

	typedef std::map<tbApplication::MenuIdentifier, MenuItem> MenuTable;
	MenuTable tbiMenuTable;

	std::set<tbApplication::MenuItemIdentifier> tbiMenuItemsSafeForDuplication;

};	/* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::IsMenuItemSafeForDuplication(const tbApplication::MenuItemIdentifier& itemIdentifier)
{
	return (tbiMenuItemsSafeForDuplication.find(itemIdentifier) != tbiMenuItemsSafeForDuplication.end()) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetMenuItemSafeForDuplication(const tbApplication::MenuItemIdentifier& itemIdentifier)
{
	tbiMenuItemsSafeForDuplication.insert(itemIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::MenuItem& tbImplementation::CreateMenuItem(const tbApplication::MenuIdentifier& menuIdentifier,
	const tbApplication::MenuItemIdentifier& itemIdentifier, const tbCore::tbString& displayName,
	const tbApplication::MenuItemIdentifier& parentIdentifier, bool isEnabled, bool isChecked, bool isVisible)
{
	MenuTable::iterator menuItr = tbiMenuTable.find(menuIdentifier);
	if (menuItr == tbiMenuTable.end())
	{
		MenuItem item;
		menuItr = tbiMenuTable.insert(MenuTable::value_type(menuIdentifier, item)).first;
	}

	tb_error_if(menuItr == tbiMenuTable.end(), "tbInternalError: Was unable to find or create a new Menu for identifier: %d", menuIdentifier);

if (false == IsMenuItemSafeForDuplication(itemIdentifier))
	{	//Make sure not to duplicate menu items that the programmer does not want duplicated.
		const bool hasMenuItem((nullptr == GetMenuItem(menuIdentifier, itemIdentifier)) ? false : true);
		tb_error_if(true == hasMenuItem, "tbExternalError: itemIdentifier(%d) already exists in menu(%d) and is not safe to duplicate.", itemIdentifier, menuIdentifier);
	}

	MenuItem newMenuItem;
	newMenuItem.mItemIdentifier = itemIdentifier;
	newMenuItem.mDisplayName = displayName;
	newMenuItem.mIsEnabled = isEnabled;
	newMenuItem.mIsChecked = isChecked;
	newMenuItem.mIsVisible = isVisible;

	if (tbApplication::kInvalidMenuItemIdentifier != parentIdentifier)
	{
		MenuItem& parentItem = GetMenuItemReference(menuIdentifier, parentIdentifier);
		parentItem.mSubItems.push_back(newMenuItem);
		return parentItem.mSubItems.back();
	}

	menuItr->second.mSubItems.push_back(newMenuItem);
	return menuItr->second.mSubItems.back();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::DestroyMenuItem(const tbApplication::MenuIdentifier& menuIdentifier)
{
	tb_error_if(true, "tbInternalError: Not yet implemented.");
}

tbImplementation::MenuItem& tbImplementation::GetMenuFor(const tbApplication::MenuIdentifier& menuIdentifier)
{
	MenuTable::iterator menuItr = tbiMenuTable.find(menuIdentifier);
	tb_error_if(menuItr == tbiMenuTable.end(), "tbInternalError: Trying to access item for a menu that does not exist: %d", menuIdentifier);
	return menuItr->second;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::MenuItem* FindMenuItemWithin(tbImplementation::MenuItem& parentItem, const tbApplication::MenuItemIdentifier& itemIdentifier)
{
	for (size_t childIndex = 0; childIndex < parentItem.mSubItems.size(); ++childIndex)
	{
		if (itemIdentifier == parentItem.mSubItems[childIndex].mItemIdentifier)
		{
			return &parentItem.mSubItems[childIndex];
		}
	}

	tbImplementation::MenuItem* resultItem(nullptr);
	for (size_t childIndex = 0; childIndex < parentItem.mSubItems.size(); ++childIndex)
	{
		resultItem = FindMenuItemWithin(parentItem.mSubItems[childIndex], itemIdentifier);
		if (nullptr != resultItem)
		{
			return resultItem;
		}
	}

	return nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::MenuItem* tbImplementation::GetMenuItem(const tbApplication::MenuIdentifier& menuIdentifier,
	const tbApplication::MenuItemIdentifier& itemIdentifier)
{
	//If itemIdentifier is invalid, or safe for duplication it can not be searched for.
	tb_error_if(tbApplication::kInvalidMenuItemIdentifier == itemIdentifier, "tbExternalError: Cannot search for a menuItem with invalid identifier.");
	tb_error_if(true == IsMenuItemSafeForDuplication(itemIdentifier), "tbExternalError: Cannot access or modify menuItem(%d) because it is safe for duplication.", itemIdentifier);

	MenuTable::iterator menuItr = tbiMenuTable.find(menuIdentifier);
	tb_error_if(menuItr == tbiMenuTable.end(), "+tbExternalError: Menu(%d) does not exist, trying to access item(%d).", menuIdentifier, itemIdentifier);
	return FindMenuItemWithin(menuItr->second, itemIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::MenuItem& tbImplementation::GetMenuItemReference(const tbApplication::MenuIdentifier& menuIdentifier,
	const tbApplication::MenuItemIdentifier& itemIdentifier)
{
	MenuItem* result = GetMenuItem(menuIdentifier, itemIdentifier);
	tb_error_if(nullptr == result, "tbExternalError: Menu(%d) does not have an item with identifier: %d", menuIdentifier, itemIdentifier);
	return *result;
}

//--------------------------------------------------------------------------------------------------------------------//
//
//tbImplementation::MenuItem& tbImplementation::GetMenuItemByIndex(const tbApplication::MenuIdentifier& menuIdentifier, const size_t& itemIndex)
//{
//	tb_error_if(true, "tbInternalError: Not yet implemented.");
//
//	MenuTable::iterator menuItr = tbiMenuTable.find(menuIdentifier);
//	tb_error_if(menuItr == tbiMenuTable.end(), "tbInternalError: Trying to access item for a menu that does not exist: %d", menuIdentifier);
//	tb_error_if(itemIndex >= menuItr->second.mMenuItems.size(), "tbInternalError: Trying to access out-of-range index(%d) on menu: %d", itemIndex, menuIdentifier);
//	return menuItr->second.mSubItems[itemIndex];
//}
//
////--------------------------------------------------------------------------------------------------------------------//
//
//size_t tbImplementation::NumberOfItemsForMenu(const tbApplication::MenuIdentifier& menuIdentifier)
//{
//	tb_error_if(true, "tbInternalError: Not yet implemented.");
//	return 0;
//}

//--------------------------------------------------------------------------------------------------------------------//

//--------------------------------------------------------------------------------------------------------------------//
