///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved --> 
///-----------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_windows

#include "../tbi_system_application_dialog.h"
#include "../../tb_application_handler_interface.h"
#include "../../../core/tb_error.h"

#define WIN32_LEAN_AND_MEAN
#include <Windows.h>
#include <CommCtrl.h>

//--------------------------------------------------------------------------------------------------------------------//

namespace tbiWindows
{	//Definition for this is actually in tbi_windows_application_dialog.cpp currently.
	typedef std::basic_string<TCHAR> WindowsString;
	WindowsString ToWindowsString(const tbCore::tbString& input);
};

//--------------------------------------------------------------------------------------------------------------------//

tbiWindows::WindowsString tbiWindows::ToWindowsString(const tbCore::tbString& input)
{
#ifdef UNICODE
		return tbCore::ToWideString(input);
#else
		return tbCore::ToStdString(input);
#endif
	}

//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	extern HWND tbiWindowHandle;		//This exists in tbi_realtime_windows_application.cpp
	extern tbApplication::ApplicationHandlerInterface* tbiApplicationHandler;		//This exists in tbi_realtime_windows_application.cpp

	struct DialogWindowInformation
	{
		tbApplication::DialogIdentifier mDialogIdentifier;
		HWND mDialogHandle;
	};

	typedef std::map<tbApplication::DialogIdentifier, DialogWindowInformation> DialogTable;
	DialogTable tbiDialogTable;

	DialogWindowInformation* FindDialogInformationByWindow(HWND dialogHandle);

	HWND CreateDialogWindow(const tbApplication::DialogIdentifier& dialogIdentifier, HWND parentWindow, HINSTANCE instanceHandle);
	HWND CreateDialogControl(HWND dialogHandle, HINSTANCE instanceHandle, const DialogControl& dialogControl);

	///
	///	This function is called from the message pump in RealtimeApplication which will attempt to process
	///	the message as a DialogMessage for each of the active DialogBoxes (for modeless dialogs), it is
	///	required to get the Tab to work between controls.
	///
	bool HandleAsDialogMessage(MSG& message);

	///
	///	The typical dialog procedure for handling messages regarding the controls on the dialog and
	///	sending them back to the programmer to handle as desired.
	///
	LRESULT CALLBACK DialogProcedure(HWND dialogHandle, UINT message, WPARAM wParam, LPARAM lParam);

};	/* namespace tbImplementation */


//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OpenDialog(const tbApplication::DialogIdentifier& dialogIdentifier,
	tbApplication::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(0 == tbiWindowHandle, "tbInternalError: OpenDialog cannot be called without a valid, running application.");

	DialogTable::iterator dialogItr = tbiDialogTable.find(dialogIdentifier);
	tb_error_if(dialogItr != tbiDialogTable.end(), "tbExternalError: OpenDialog can't open existing dialog: %d", dialogIdentifier);

	HINSTANCE instanceHandle = (HINSTANCE)GetWindowLongPtr(tbiWindowHandle, GWLP_HINSTANCE);

	DialogWindowInformation& dialogInformation = tbiDialogTable[dialogIdentifier];
	dialogInformation.mDialogIdentifier = dialogIdentifier;
	dialogInformation.mDialogHandle = CreateDialogWindow(dialogIdentifier, tbiWindowHandle, instanceHandle);

	const size_t numberOfControls = NumberOfControlsForDialog(dialogIdentifier);
	for (size_t controlIndex = 0; controlIndex < numberOfControls; ++controlIndex)
	{
		const DialogControl* dialogControl = GetDialogControlByIndex(dialogIdentifier, controlIndex);
		HWND controlHandle = CreateDialogControl(dialogInformation.mDialogHandle, instanceHandle, *dialogControl);
		EnableWindow(controlHandle, (true == dialogControl->mIsEnabled) ? TRUE : FALSE);
	}

	ShowWindow(dialogInformation.mDialogHandle, SW_SHOW);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CloseDialog(const tbApplication::DialogIdentifier& dialogIdentifier,
	tbApplication::ApplicationHandlerInterface& applicationHandler)
{
	DialogTable::iterator dialogItr = tbiDialogTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiDialogTable.end(), "tbInternalError: CloseDialog found dialog did not exist: %d", dialogIdentifier);

	DialogWindowInformation& dialogInformation = dialogItr->second;
	DestroyWindow(dialogInformation.mDialogHandle);
	dialogInformation.mDialogHandle = NULL;

	tbiDialogTable.erase(dialogItr);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::DialogWindowInformation* tbImplementation::FindDialogInformationByWindow(HWND dialogHandle)
{
	for (DialogTable::iterator dialogItr = tbiDialogTable.begin(), dialogEnd = tbiDialogTable.end(); dialogItr != dialogEnd; ++dialogItr)
	{
		if (dialogItr->second.mDialogHandle == dialogHandle)
		{
			return &dialogItr->second;
		}
	}

	//tb_error_if(true, "tbInternalError: Was unable to find dialog information by window.");
	//static DialogWindowInformation removeWarning;
	//return removeWarning;

	return nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

HWND tbImplementation::CreateDialogWindow(const tbApplication::DialogIdentifier& dialogIdentifier, HWND parentWindow, HINSTANCE instanceHandle)
{
	const TCHAR kDefaultDialogClass[] = TEXT("TurtleBrainsDialog");
	const TCHAR kDefaultDialogTitle[] = TEXT("TurtleBrains Dialog");

	WNDCLASS windowClass;
	windowClass.style = CS_VREDRAW | CS_HREDRAW ;
	windowClass.lpfnWndProc = (WNDPROC)tbImplementation::DialogProcedure;
	windowClass.cbClsExtra = 0;
	windowClass.cbWndExtra = 0;
	windowClass.hInstance = NULL;
	windowClass.hIcon = LoadIcon(NULL, IDI_APPLICATION);
	windowClass.hCursor = LoadCursor(NULL, IDC_ARROW);
	windowClass.hbrBackground = (HBRUSH)COLOR_WINDOW;
	windowClass.lpszMenuName = NULL;
	windowClass.lpszClassName = kDefaultDialogClass;
	RegisterClass(&windowClass);

	//const tbApplication::DialogBucketType bucketType(GetDialogBucketType(dialogIdentifier));
	//const int numberOfFilledSlots(GetDialogNumberOfFilledSlots(dialogIdentifier));
	//const int numberOfVisibleSlots(GetDialogNumberOfVisibleSlots(dialogIdentifier));

	//const int isScrolling = (numberOfVisibleSlots < numberOfFilledSlots) ? WS_VSCROLL : 0;
	//const int dialogWidth = (kWindowsControlPaddingX) + ((kWindowsControlBaseWidth + kWindowsControlPaddingX) * (bucketType + 1));
	//const int dialogVisibleHeight = (kWindowsControlPaddingY * 2) + ((kWindowsControlBaseHeight + kWindowsControlPaddingY) * numberOfVisibleSlots);
	//const int dialogFilledHeight = (kWindowsControlPaddingY * 2) + ((kWindowsControlBaseHeight + kWindowsControlPaddingY) * numberOfFilledSlots);

	AreaRectangle dialogArea;
	int filledHeight(0);
	GetDialogArea(dialogArea, filledHeight, dialogIdentifier);
	const int isScrolling((filledHeight > dialogArea.mHeight) ? 1 : 0);

	HWND dialogHandle = CreateWindow(kDefaultDialogClass, kDefaultDialogTitle, WS_DLGFRAME | isScrolling /* | WS_POPUP */,
		100, 100, dialogArea.mWidth, dialogArea.mHeight, parentWindow, NULL, instanceHandle, NULL);

	//Resize the client area to be the expected dialogWidth/dialogHeight!
	RECT clientArea;
	GetClientRect(dialogHandle, &clientArea);
	const int differenceX = dialogArea.mWidth - (clientArea.right - clientArea.left);
	const int differenceY = dialogArea.mHeight - (clientArea.bottom - clientArea.top);
	SetWindowPos(dialogHandle, NULL, 0, 0, dialogArea.mWidth + differenceX, dialogArea.mHeight + differenceY, SWP_NOMOVE | SWP_NOACTIVATE | SWP_NOOWNERZORDER | SWP_NOZORDER);

	//Vertical scrolling is nice if there are more slots filled than visible.
	if (1 == isScrolling)
	{	//Windows does better setting this itself??
		int scrollRange = filledHeight - dialogArea.mHeight;
		SetScrollRange(dialogHandle, SB_VERT, 0, scrollRange, false);
		SetScrollPos(dialogHandle, SB_VERT, 0, true);
	}

	return dialogHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

HWND tbImplementation::CreateDialogControl(HWND dialogHandle, HINSTANCE instanceHandle, const DialogControl& dialogControl)
{
	//const int kControlWidth(kWindowsControlBaseWidth);
	//const int kControlHeight(kWindowsControlBaseHeight);

	//const int controlTop = kWindowsControlPaddingY + ((kControlHeight + kWindowsControlPaddingY) * dialogControl.mSlotInBucket);
	//const int leftEdgeOffset = static_cast<int>(static_cast<float>(kControlWidth) * static_cast<float>(dialogControl.mLeftEdgeOffset) / 100.0f);
	//const int controlLeft = kWindowsControlPaddingX + ((kControlWidth + kWindowsControlPaddingX) * dialogControl.mGuide) + leftEdgeOffset;
	//const int controlWidth = static_cast<int>(static_cast<float>(kControlWidth) * static_cast<float>(dialogControl.mWidthPercentage) / 100.0f);

	AreaRectangle controlArea;
	GetDialogControlArea(controlArea, dialogControl);

	if (kButtonControl == dialogControl.mType)
	{
		tbiWindows::WindowsString windowStr = tbiWindows::ToWindowsString(dialogControl.mValue);
		return CreateWindow(WC_BUTTON, tbiWindows::ToWindowsString(dialogControl.mValue).c_str(), 
			WS_CHILD | WS_VISIBLE | BS_PUSHBUTTON | WS_TABSTOP, controlArea.mLeft, controlArea.mTop, controlArea.mWidth, controlArea.mHeight,
			dialogHandle, (HMENU)dialogControl.mIdentifier, instanceHandle, NULL);
	}
	else if (kCheckboxControl == dialogControl.mType)
	{
		bool isChecked = false;
		tbCore::tbString controlName;
		tbImplementation::StringToCheckbox(dialogControl.mValue, isChecked, controlName);

		HWND controlWindow = CreateWindow(WC_BUTTON, tbiWindows::ToWindowsString(controlName).c_str(), 
			WS_CHILD | WS_VISIBLE | BS_CHECKBOX | WS_TABSTOP, controlArea.mLeft, controlArea.mTop, controlArea.mWidth, controlArea.mHeight / 2,
			dialogHandle, (HMENU)dialogControl.mIdentifier, instanceHandle, NULL);

		CheckDlgButton(dialogHandle, dialogControl.mIdentifier, isChecked);
		return controlWindow;
	}
	else if (kLabelControl == dialogControl.mType)
	{
		return CreateWindow(WC_STATIC, tbiWindows::ToWindowsString(dialogControl.mValue).c_str(), 
			WS_CHILD | WS_VISIBLE | SS_LEFT, controlArea.mLeft, controlArea.mTop, controlArea.mWidth, controlArea.mHeight, 
			dialogHandle, (HMENU)dialogControl.mIdentifier, instanceHandle, NULL);
	}
	else if (kTextControl == dialogControl.mType)
	{
		return CreateWindowEx(WS_EX_CLIENTEDGE, WC_EDIT, tbiWindows::ToWindowsString(dialogControl.mValue).c_str(),
			WS_BORDER | ES_AUTOHSCROLL | ES_LEFT | WS_CHILD | WS_VISIBLE | WS_TABSTOP,
			controlArea.mLeft, controlArea.mTop, controlArea.mWidth, controlArea.mHeight,
			dialogHandle, (HMENU)dialogControl.mIdentifier, instanceHandle, NULL);
	}
	else if (kDropdownControl == dialogControl.mType)
	{
		std::vector<tbCore::tbString> values;
		int selectedIndex = 0;
		tbImplementation::StringToDropdown(dialogControl.mValue, selectedIndex, values);

		HWND controlWindow = CreateWindow(WC_COMBOBOX, TEXT(""), WS_CHILD | WS_VISIBLE | CBS_HASSTRINGS | CBS_DROPDOWNLIST | WS_TABSTOP | WS_VSCROLL,
			controlArea.mLeft, controlArea.mTop, controlArea.mWidth, controlArea.mHeight * kMaximumDisplayedDropdownItems,
			dialogHandle, (HMENU)dialogControl.mIdentifier, instanceHandle, NULL);

		for (size_t valueIndex = 0; valueIndex < values.size(); ++valueIndex)
		{
			SendMessage(controlWindow, CB_ADDSTRING, 0, (LPARAM)tbiWindows::ToWindowsString(values[valueIndex]).c_str());
		}

		SendMessage(controlWindow, CB_SETCURSEL, selectedIndex, 0);

		return controlWindow;
	}

	tb_error_if(true, "tbInternalError: Could not create dialog control.");
	return NULL;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::HandleAsDialogMessage(MSG& message)
{
	for (DialogTable::iterator dialogItr = tbiDialogTable.begin(), dialogEnd = tbiDialogTable.end(); dialogItr != dialogEnd; ++dialogItr)
	{
		if (TRUE == IsDialogMessage(dialogItr->second.mDialogHandle, &message))
		{	//This message was handled by the dialog box, stop trying to process.
			return true;
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

LRESULT CALLBACK tbImplementation::DialogProcedure(HWND dialogHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	DialogWindowInformation* dialogInformationPointer = FindDialogInformationByWindow(dialogHandle);

	switch (message)
	{
	case WM_COMMAND: {
		tb_error_if(nullptr == dialogInformationPointer, "tbInternalError: Handling message for unknown dialog.");

		DialogWindowInformation& dialogInformation(*dialogInformationPointer);
		tbApplication::DialogControlIdentifier controlIdentifier = LOWORD(wParam);

		if (IDCANCEL == wParam)
		{	//This should be happening on escape press, and hopefully only escape press.
			//This will not only happen on KeyPresses, this will also happen on a Button click with an Identifier of 2.
			//TODO: TIM: Find a way to handle this case, even if Windows must add 1000 to each controlIdentifier, then subtract...
			break;
		}

		//TODO: TIM: Be sure that the controlIdentifier exists in this dialog!!
		DialogControl* dialogControl = GetDialogControl(dialogInformation.mDialogIdentifier, controlIdentifier);
		tb_error_if(nullptr == dialogControl, "tbInternalError: Handling message for unknown dialog control.");

		if (BN_CLICKED == HIWORD(wParam))
		{	//ButtonControl and CheckboxControl - Notify about the DialogAction
			if (kCheckboxControl == dialogControl->mType)
			{
				bool isNowChecked = (BST_CHECKED == IsDlgButtonChecked(dialogHandle, dialogControl->mIdentifier) ? false : true);
				dialogControl->mValue = CheckboxToString(isNowChecked, StringToCheckboxName(dialogControl->mValue));
				CheckDlgButton(dialogHandle, dialogControl->mIdentifier, isNowChecked);
			}

			tbiApplicationHandler->OnDialogAction(dialogInformation.mDialogIdentifier, dialogControl->mIdentifier);
		}

		else if (CBN_CLOSEUP == HIWORD(wParam))
		{	////DropdownControl - Update the index selection from the Combo Box when it closes.
			if (kDropdownControl == dialogControl->mType)
			{
				int selectedIndex = SendMessage((HWND)lParam, CB_GETCURSEL, 0, 0);
				int throwAwayIndex = 0;
				std::vector<tbCore::tbString> values;
				tbImplementation::StringToDropdown(dialogControl->mValue, throwAwayIndex, values);
				dialogControl->mValue = tbImplementation::DropdownToString(selectedIndex, values);
			}

			tbiApplicationHandler->OnDialogAction(dialogInformation.mDialogIdentifier, dialogControl->mIdentifier);
		}

		else if (EN_SETFOCUS == HIWORD(wParam))
		{	//TextControl - Select the text when the focus is grabbed by a TextControl
			SendMessage(GetDlgItem(dialogInformation.mDialogHandle, dialogControl->mIdentifier), EM_SETSEL, 0, -1);
		}
		else if (EN_KILLFOCUS == HIWORD(wParam))// || EN_SETFOCUS == HIWORD(wParam))
		{	//TextControl - Update the string with what is in the control and notify of Dialog Action
			TCHAR buffer[2048];
			GetDlgItemText(dialogHandle, dialogControl->mIdentifier, buffer, 2048);
			dialogControl->mValue = tbCore::ToString(buffer);
			tbiApplicationHandler->OnDialogAction(dialogInformation.mDialogIdentifier, dialogControl->mIdentifier);
		}
		else if (EN_CHANGE == HIWORD(wParam))	//true to handle this?
		{	//Should we or should  we not update the item value and call OnDialogAction?  This happens very often as typing.
		}

		break; }

		//The following commands are handled in order to get the vertical scroll bar to work properly.
		case WM_VSCROLL: {
			int scrollMinimum = 0;
			int scrollMaximum = 0;
			long scrollPosition = GetScrollPos(dialogHandle, SB_VERT);
			GetScrollRange(dialogHandle, SB_VERT, &scrollMinimum, &scrollMaximum);
			
			long previousScrollPosition = scrollPosition;
			switch (LOWORD(wParam))
			{
			case SB_TOP:			scrollPosition = 0;	break;
			case SB_BOTTOM:			scrollPosition = scrollMaximum;	break;
			case SB_LINEUP:			scrollPosition--;		break;
			case SB_LINEDOWN:		scrollPosition++;		break;
			case SB_THUMBPOSITION:
			case SB_THUMBTRACK:
				scrollPosition = HIWORD(wParam);
				break;
			};

			if (previousScrollPosition != scrollPosition)
			{
				long displayMoveAmount = (scrollPosition - previousScrollPosition);
				SetScrollPos(dialogHandle, SB_VERT, scrollPosition, TRUE);
				ScrollWindow(dialogHandle, 0, -displayMoveAmount, NULL, NULL);
				UpdateWindow(dialogHandle);
			}
		break; }
	};

	return DefWindowProc(dialogHandle, message, wParam, lParam);
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_windows */
