///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved --> 
///-----------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_windows

#include "../tbi_system_application_message_box.h"

namespace TurtleBrains
{
	namespace Application
	{
		extern const bool kResultOkay;
		extern const bool kResultCancel;
	};	/* namespace Application */
};	/* namespace TurtleBrains */

//#define WIN32_LEAN_AND_MEAN
#include <windows.h>

//mingw seemed not to have this defined, hopefully doing this will keep things working happily.
#ifdef tb_mingw
  #ifndef OFN_DONTADDTORECENT
    #define OFN_DONTADDTORECENT 0x02000000
  #endif /* OFN_DONTADDTORECENT */
#endif /* tb_mingw */

namespace tbiWindows
{	//Definition for this is actually in tbi_windows_application_dialog.cpp currently.
	typedef std::basic_string<TCHAR> WindowsString;
	WindowsString ToWindowsString(const tbCore::tbString& input);
};

namespace tbImplementation
{
	extern HWND tbiWindowHandle;

};	/* namespace tbImplementation */


//-------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::ModalMessageBox(const tbCore::tbString& title, const tbCore::tbString& message,
	const tbCore::tbString& primaryButton, const tbCore::tbString& secondaryButton)
{
	tbiWindows::WindowsString titleForWindows = tbiWindows::ToWindowsString(title);
	tbiWindows::WindowsString messageForWindows = tbiWindows::ToWindowsString(message);

	//Note: It may be worth looking into this as a way to change the primary/secondary button text.
	//http://stackoverflow.com/questions/12570201/message-box-with-cancel-button-only
	UINT messageBoxType((true == secondaryButton.empty()) ? MB_OK : MB_OKCANCEL);
	return (IDOK == MessageBox(tbiWindowHandle, messageForWindows.c_str(), titleForWindows.c_str(), messageBoxType)) ? 
		TurtleBrains::Application::kResultOkay : TurtleBrains::Application::kResultCancel;
}

//-------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::ModalOpenFileDialog(tbCore::tbString& fileToOpen, const tbCore::tbString& startDirectory,
	const tbCore::tbString& fileExtension)
{
	const size_t kBufferSize(2048);
	TCHAR fileBuffer[kBufferSize];
//	TCHAR filterBuffer[128];
	fileBuffer[0] = NULL;

	OPENFILENAME openFileName;
	memset(&openFileName, 0, sizeof(openFileName));
	openFileName.lStructSize = sizeof(OPENFILENAME);
	openFileName.hwndOwner = tbiWindowHandle;
	openFileName.hInstance = GetModuleHandle(nullptr);

	//TODO: TIM: Don't hard code this, use the fileExtension if it makes sense to with Mac open...
	//tbCore::tbString = tbCore::ToString("Files\0*") + fileExtension + tbCore::ToString("\0\0");	//This won't work because of null terms.
	openFileName.lpstrFilter = "All Files\0*.*\0\0";
	openFileName.lpstrCustomFilter = nullptr;
	openFileName.nMaxCustFilter = 0;
	openFileName.nFilterIndex = 1;
	openFileName.lpstrFile = fileBuffer;
	openFileName.nMaxFile = kBufferSize;
	openFileName.Flags = OFN_DONTADDTORECENT | OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_LONGNAMES | OFN_NOCHANGEDIR;

	bool didOpenFile = false;
	if (TRUE == GetOpenFileName(&openFileName))
	{
		for (size_t i = 0; i < kBufferSize; ++i)
		{
			if ('\\' == fileBuffer[i]) { fileBuffer[i] = '/'; }
			if ('\0' == fileBuffer[i]) { break; }
		}

		fileToOpen = tbCore::ToString(fileBuffer);
		didOpenFile = true;
	}

	return didOpenFile;
}

//-------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::ModalSaveFileDialog(tbCore::tbString& fileToSave, const tbCore::tbString& startDirectory,
	const tbCore::tbString& fileExtension)
{
	const size_t kBufferSize(2048);
	TCHAR fileBuffer[kBufferSize];
	fileBuffer[0] = NULL;

	OPENFILENAME saveFileName;
	memset(&saveFileName, 0, sizeof(saveFileName));
	saveFileName.lStructSize = sizeof(OPENFILENAME);
	saveFileName.hwndOwner = tbiWindowHandle;
	saveFileName.hInstance = GetModuleHandle(nullptr);

	//TODO: TIM: Don't hard code this, use the fileExtension if it makes sense to with Mac open...
	//tbCore::tbString = tbCore::ToString("Files\0*") + fileExtension + tbCore::ToString("\0\0");	//This won't work because of null terms.
	saveFileName.lpstrFilter = ("All Files\0*.*\0\0");
	saveFileName.lpstrCustomFilter = nullptr;
	saveFileName.nMaxCustFilter = 0;
	saveFileName.nFilterIndex = 1;
	saveFileName.lpstrFile = fileBuffer;
	saveFileName.nMaxFile = kBufferSize;
	saveFileName.Flags = OFN_DONTADDTORECENT | OFN_OVERWRITEPROMPT | OFN_FILEMUSTEXIST | OFN_PATHMUSTEXIST | OFN_LONGNAMES |  OFN_NOCHANGEDIR;

	bool didSaveFile = false;
	if (TRUE == GetSaveFileName(&saveFileName))
	{
		for (size_t i = 0; i < kBufferSize; ++i)
		{
			if ('\\' == fileBuffer[i]) { fileBuffer[i] = '/'; }
			if ('\0' == fileBuffer[i]) { break; }
		}

		fileToSave = tbCore::ToString(fileBuffer);
		didSaveFile = true;
	}

	return didSaveFile;
}

//-------------------------------------------------------------------------------------------------------------------//

#endif /* tb_windows */
