///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_ApplicationDialog_h_
#define _TurtleBrains_ApplicationDialog_h_

#include "../core/tb_string.h"
#include "../core/tb_noncopyable.h"

#include <map>
#include <vector>
#include <set>

namespace tbImplementation { class DialogControl; }

namespace TurtleBrains
{
	namespace Application
	{
		///
		/// @details The DialogBucketType chooses the width for the ApplicationDialog in number of buckets for dialog
		///   controls to be added to and placed within.
		///
		/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
		///
		enum DialogBucketType
		{
			kSingleBucket,		//1 guide,  (left bucket)
			kDoubleBucket,		//3 guides, (left bucket, right bucket, center guide)
			//kTripleBucket,	//5 guides, (left bucket, center bucket, right bucket) ... (mid-left guide, mid-center guide) ??? uns
		};

		///
		/// @details The key to an application dialog lies within this identifier, the value must be unique for different
		///   dialogs in your project.  The ApplicationDialog object takes the identifier and uses it to hide the messy
		///   implementation details behind the scenes.  Multiple ApplicationDialog objects with the same identifier will
		///   access the settings for the same dialog even though they are different ApplicationDialog objects.
		///
		typedef unsigned short DialogIdentifier;

		///
		/// @details Each control added to a dialog must have a unique identifier that is used to access and modify the
		///   dialog control at any point after it is added.  The DialogControlIdentifier is also used to handle the action
		///   of the dialog control being interacted with by the user.  It is possible to use the same identifier for
		///   multiple controls by using ApplicationDialog::SetControlSafeForDuplication().  However identifiers that are
		///   safe for duplication cannot be accessed or modified and the action may not be reported.
		///
		typedef unsigned short DialogControlIdentifier;

		///
		/// @details Expresses the horizonal position, column, of a dialog control in the dialog.
		///
		/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
		///
		typedef int DialogControlGuide;

		///
		/// @details Expresses the vertical position, row, of a dialog control in the dialog where 0 represents the
		///   top-most control.
		///
		/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
		///
		typedef int DialogControlSlot;

		///
		/// @details This slot value represents "the bottom most row" of the ApplicationDialog given a specified guide.
		///
		/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
		///
		extern const DialogControlSlot kDefaultSlot;

		///
		/// @brief Create a customized dialog prompt with simple controls.
		///
		/// @details The ApplicationDialog represents a reusuable dialog object that common controls can quickly be added
		///   to and remain easy to customize without worrying about the look on each targeted system.  When the user
		///   interacts with a control, the ApplicationHandlerInterface::OnDialogAction() will be invoked.
		///
		/// TODO: TIM: Documentation: Create an example of creating / using an ApplicaitonDialog.
		/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
		///
		class ApplicationDialog : public Core::Noncopyable
		{
		public:
			///
			/// @details When you create an ApplicationDialog object with a DialogIdentifier, you can later create another
			///   ApplicationDialog object with that same DialogIdentifier value and BOTH of those dialog objects will refer
			///   to and edit the same dialog settings.
			///
			explicit ApplicationDialog(const DialogIdentifier& dialogIdentifier);

			///
			/// @details This will destruct the ApplicationDialog object, but will not destroy resources or remove the dialog
			///   settings from memory or remove the dialog if it were visible.  Creating another ApplicationDialog object
			///   with the same DialogIdentifier value will give access to these settings once again.  To close the dialog
			///   you should use RealtimeApplication::CloseDialog()
			///
			/// @note this does not cause the dialog to be destroyed, or even cause any settings to be removed from memory,
			///   as creating another ApplicationDialog object with the same DialogIdentifier will provide access again.
			///
			~ApplicationDialog(void);

			///
			/// @details Retrieves the DialogIdentifier as specified when the object was created.  Once created, the
			///   identifier cannot change for an ApplicationDialog object.
			///
			const DialogIdentifier& GetIdentifier(void) const;

			///
			/// @details By default a custom ApplicationDialog will contain a single column of five (5) slots for controls to
			///   be added.  Resizing the dialog by changing the bucket type and number of visible slots will create a dialog
			///   that fits your project and needs better.
			///
			/// @param bucketType               The bucketType of the dialog prompt effects the width of the prompt and how
			///   many controls can fit side by side.  This also effect the number, <em>and values,</em> of the guides used
			///   to position the dialog controls horizontally.
			/// @param visibleSlots             Choose how many controls can be visible vertically before (if any) scrolling
			///   is required for lower controls to be seen.  This effects the height of the prompt.
			/// @param allowVerticalScrolling   Allow controls to be added in slots greater than the number of visible slots.
			///   If this value is false, an error will be triggered when a control is added, or repositioned, to a slot that
			///   is not visible without scroll bars.
			///
			/// @see DialogBucketType, DialogControlSlot
			///
			/// @note Vertical scrolling is currently only implemented on Windows and not working on other platforms, for
			///   maximum portability do not allow scrolling, and do not reposition controls to slots greater than the
			///   number of visible slots requested here.
			///
			/// TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			void ResizeDialog(const DialogBucketType& bucketType, const int visibleSlots, const bool allowVerticalScrolling = false);

			///
			/// @details Add a label control to the dialog prompt for the user to read.  The user cannot interact with a
			///   label control, however it can give them valuable information about the other controls on the dialog prompt.
			///
			/// @param controlIdentifier  The identifier of the control which can be used to access the control, modify
			///   the control or handle interaction with the dialog control.
			/// @param controlName        The text that should be displayed by the label to help or inform the user.
			/// @param guide              The guide which the control will be placed.aligned with.  See TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @note Any controls added after calling OpenDialog from the RealtimeApplication, will not change the dialog.
			///   The control will not be added until the dialog is closed and opened again.
			///
			void AddLabelControl(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlName, const DialogControlGuide& guide = 0);

			///
			/// @details Add a button control to the dialog prompt for the user to interact with.  When the button is pressed
			///   by the user, the ApplicationHandlerInterface::OnDialogAction() will be invoked with the controlIdentifier
			///   specified here, and the dialogIdentifier of the prompt it was clicked from.
			///
			/// @param controlIdentifier  The identifier of the control which can be used to access the control, modify
			///   the control or handle interaction with the dialog control.  An error will be triggered if this value is
			///   an identifier that has been set safe to duplicate.
			/// @param controlName        The text that should be displayed on the button to get the user to click.
			/// @param guide              The guide which the control will be placed.aligned with.  See TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @note Any controls added after calling OpenDialog from the RealtimeApplication, will not change the dialog.
			///   The control will not be added until the dialog is closed and opened again.
			///
			void AddButtonControl(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlName, const DialogControlGuide& guide = 0);

			///
			/// @details Add a checkbox control to the dialog prompt for the user to interact with.  When the checkbox is
			///   toggled by the user, the ApplicationHandlerInterface::OnDialogAction() will be invoked with the
			///   controlIdentifier specified here, and the dialogIdentifier of the prompt it was clicked from.
			///
			/// @param controlIdentifier     The identifier of the control which can be used to access the control, modify
			///   the control or handle interaction with the dialog control.  An error will be triggered if this value is
			///   an identifier that has been set safe to duplicate.
			/// @param controlName           The text that should be displayed next to the checkbox control.
			/// @param isChecked             The starting state of the checkbox, true if it should be checked and false for
			///   the checkbox to start unchecked, user interaction will toggle this state.
			/// @param guide                 The guide which the control will be placed.aligned with.  See TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @note Any controls added after calling OpenDialog from the RealtimeApplication, will not change the dialog.
			///   The control will not be added until the dialog is closed and opened again.
			///
			void AddCheckboxControl(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlName, const bool isChecked, const DialogControlGuide& guide = 0);

			///
			/// @details Add a text edit control to the dialog prompt for the user to interact with.  When the text control
			///   loses focus, the user clicks away from the control, the ApplicationHandlerInterface::OnDialogAction() will
			///   be invoked with the controlIdentifier specified here, and the dialogIdentifier of the prompt it was clicked
			///   from.
			///
			/// @param controlIdentifier     The identifier of the control which can be used to access the control, modify
			///   the control or handle interaction with the dialog control.  An error will be triggered if this value is
			///   an identifier that has been set safe to duplicate.
			/// @param controlValue          The text that should be used as the default input value for the text control.
			/// @param guide                 The guide which the control will be placed.aligned with.  See TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @note Any controls added after calling OpenDialog from the RealtimeApplication, will not change the dialog.
			///   The control will not be added until the dialog is closed and opened again.
			///
			void AddTextControl(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlValue, const DialogControlGuide& guide = 0);

			///
			/// @details Add a dropdown control to the dialog prompt for the user to interact with.  When the selection is
			///   changed or the dropdown closes, the ApplicationHandlerInterface::OnDialogAction() method will be invoked
			///   with the controlIdentifier specified here, and the dialogIdentifier of the prompt it was clicked from.
			///   GetValueAsInteger() is used to retieve the zero-based index of the selection, and likewise SetValueAsInteger
			///   is used to modifiy the selection, again zero-based index.  GetValueAsString will return the selected option
			///   as a string.
			///
			/// @param controlIdentifier     The identifier of the control which can be used to access the control, modify
			///   the control or handle interaction with the dialog control.  An error will be triggered if this value is
			///   an identifier that has been set safe to duplicate.
			/// @param controlValues         A container of strings of each option the dropdown control should contain.
			/// @param guide                 The guide which the control will be placed.aligned with.  See TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @note Any controls added after calling OpenDialog from the RealtimeApplication, will not change the dialog.
			///   The control will not be added until the dialog is closed and opened again.
			///
			void AddDropdownControl(const DialogControlIdentifier& controlIdentifier, const std::vector<tbCore::tbString>& controlValues, const DialogControlGuide& guide = 0);

			///
			/// @details For most dialog controls this will retrieve the displayed name of the control but for Text Controls
			///   the value the user can modify is returned, and for Dropdown Controls the currently selected string value is
			///   retrieved.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.
			///
			tbCore::tbString GetValueAsString(const DialogControlIdentifier& controlIdentifier) const;

			///
			/// @details For most dialog controls this will simply return a zero, 0.0f, without attempting to convert from a
			///   string.  For Text Controls the return will be the string value of the control, converted into a float using
			///   TurtleBrains::Core::FromString.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.
			///
			float GetValueAsFloat(const DialogControlIdentifier& controlIdentifier) const;

			///
			/// @details The value retrieved depends greatly on the type of control the identitifier represents.  Label and
			///   Button controls will return zero, 0, without attempting to convert from a string.  For Text Controls the
			///   return will be the string value of the control, converted into an int using TurtleBrains::Core::FromString.
			///   Checkbox Controls will return the status of the checkbox, 0 for unchecked, 1 for checked.  The Dropdown
			///   Controls will return the zero-based index of the currently selected item.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			///
			int GetValueAsInteger(const DialogControlIdentifier& controlIdentifier) const;

			/** @cond */

			///
			/// @details Not yet implemented but likely to be added to framework.
			/// @details For most dialog controls this will modify the displayed name of the control but for Text Controls
			///   the value the user can interact with is modified instead.  For Dropdown Controls this function does not
			///   modify any names or change the selection, instead an error will be triggered, to modify the selection of
			///   a Dropdown Control SetValueAsInteger may be used to set the selected index.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param controlValue          The text that should modified to, representing typically the name / display of
			///   the control.  For Text Controls it represents the value the user can interact with.
			///
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			//void SetValueAsString(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlValue);

			///
			/// @details Not yet implemented but likely to be added to framework.
			/// @details For almost all dialog controls this will trigger an error condition.  Text Controls are the only
			///   controls that should use this, and the float will be converted into a string using the ToString function in
			///   TurtleBrains::Core then set.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param controlValue          The value that should be converted into a string and set into the Text Control.
			///
			/// @note This will trigger an error condition if the controlIdentifier is not a Text Control.
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			//void SetValueAsFloat(const DialogControlIdentifier& controlIdentifier, const float controlValue);

			///
			/// @details Not yet implemented but likely to be added to framework.
			/// @details The value modified depends greatly on the type of control the identifier represents.  For Label and
			///   Button Controls an error will be triggered.  For Text Control the value will be converted into a string
			///   using TurtleBrains::Core::ToString and then set as the interactable text.  For Checkbox Controls the value
			///   can check or uncheck the checkbox, a value of 0 represents unchecked, and a value of 1 is checked, other
			///   values will trigger an error.  For the Dropdown Controls, the value is the index to be selected, which is
			///   a zero-based index of the options added when the dropdown was created.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param controlValue
			///
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			//void SetValueAsInteger(const DialogControlIdentifier& controlIdentifier, const int controlValue);

			/** @endcond */

			///
			/// @details Modifies the control specified by controlIdentifier so that the user can interact with the enabled
			///   control, disable the control so the user can see the control but not interact with it.  Typically the
			///   control will be displayed greyed out to indicated it is disabled.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param isEnabled             True to enable the control and allow the user to interact, or false to disable
			///   interaction and display the control greyed out to indicated a disabled control.
			///
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			void SetEnabled(const DialogControlIdentifier& controlIdentifier, const bool isEnabled);

			/** @cond */

			///
			/// @details Not yet implemented but likely to be added to framework.
			///   Modifies the control specified by controlIdentifier so that the control can be seen on the dialog
			///   prompt or left hidden from view.  When hidden, the control cannot be interacted with.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param isVisible             True for the control to be visible and allow interaction, or false to hide the
			///   control completely and not allow user interaction.
			///
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			/// @note This has not been implemented, and may not be implemented.  If you don't want a control to be visible,
			///   it would be best not to add it to the dialog prompt at all, however, this could prove useful functionality
			///   and implementation will be considered.
			///
			//void SetVisible(const DialogControlIdentifier& controlIdentifier, const bool isVisible);

			/** @endcond */

			///
			/// @details For dialog prompts that need any more than the very basic level of placement, controls can be moved
			///   and resized for customized prompts.  This is a multipurpose function that both repositions and resizes the
			///   control.  For more sizing and positioning information see TODO: TIM: DocLink: Link to an example of control placement and sizing.
			///
			/// @param controlIdentifier     The identifier of the control that to retrieve the value as a string from, this
			///   is the same value as the control had when added to the custom dialog prompt.  An error will be triggered if
			///   this controlIdentifier is set safe for duplication.
			/// @param guide                 The guide which the control will be placed.aligned with once repositioned.
			/// @param leftEdge              An offset value for the positioning of the left edge of the control relative to
			///   the choosen guide.  The value is based on a percentage of the width of the @e standard control, so a value
			///   of 30 would move the left edge of the control, 30% of the standard control width to the right of the guide.
			/// @param width                 The new desired with of control represented as a percentage based on the width
			///   of the standard control.  The default value of 100 keeps the control width unchanged from when first added
			///   to the dialog prompt.  A value of 50 would be half as wide as the standard control width.
			/// @param slot                  The slot to place the control into.  A value of kDefaultSlot, which is the 
			///   default value, will keep the slot unchanged keeping the control at the same vertical position.
			///
			/// @note Any controls modified after calling OpenDialog from the RealtimeApplication, will not change the dialog
			///   prompt.  The control will not be modified until the dialog is closed and opened again.
			///
			void RepositionControl(const DialogControlIdentifier& controlIdentifier, const DialogControlGuide& guide,
				 const int leftEdge = 0, const int width = 100, const DialogControlSlot& slot = kDefaultSlot);

			///
			/// @details Sometimes you don't care to modify or retrieve information from a dialog control, mainly labels.
			///   Since each control must have an identifier value attached to it, this allows you to create an identifier
			///   that is safe for duplicating.  Once an identifier is made safe for duplication it can not be made unsafe
			///   and access to any controls with the identifier will be lost.  Calling GetValue / SetEnabled / Reposition
			///   or any accessor/modifier functions for an identifier safe for duplication will trigger errors.
			///
			/// @param controlIdentifier   The identifier value that is safe for duplication, no controls created with
			///   this identifier can be modified or accessed beyond creation.
			///
			/// @note Once an identifier is set safe for duplication, the value can not be used to access or modifiy any
			///   controls with that identifier, and the value cannot be later removed from safe for duplication.
			///
			static void SetControlSafeForDuplication(const DialogControlIdentifier& controlIdentifier);

		private:
			const DialogIdentifier mIdentifier;
		};

	}; /* namespace Application */
}; /* namespace TurtleBrains */

namespace tbApplication = TurtleBrains::Application;

#endif /* _TurtleBrains_ApplicationDialog_h_ */
