///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_ApplicationStatus_h_
#define _TurtleBrains_ApplicationStatus_h_

#include "../core/tb_types.h"
#include <map>
#include <vector>

namespace TurtleBrains
{
	namespace Application
	{

		///
		/// @details The key to an application status lies within this identifier, the value must be unique for different
		///   status in your project.  The ApplicationStatus object takes the identifier and uses it to hide the messy
		///   implementation details behind the scenes.  Multiple ApplicationStatus objects with the same identifier will
		///   access the settings for the same status bar even though they are different ApplicationStatus objects.
		///
		typedef unsigned short StatusIdentifier;

		///
		/// @details Each item added to a status bar must have a unique identifier that is used to access and modify the
		///   item at any point after it is added.  This identifier cannot be made safe for duplication.
		///
		typedef unsigned short StatusItemIdentifier;

		///
		/// @brief Create a customized status bar area to feed the user status information.
		///
		/// @details The ApplicationStatus object represents a reusuable status bar object that can be partitioned to hold
		///   up to 16 partitioned items that can be updated/modified individually.
		///
		/// TODO: TIM: Documentation: Create an example of creating / using an ApplicaitonStatus.
		///
		class ApplicationStatus
		{
		public:
			///
			/// @details When you create an ApplicationStatus object with a StatusIdentifier, you can later create another
			///   ApplicationStatus object with that same StatusIdentifier value and BOTH of those status objects will refer
			///   to and edit the same status bar settings.
			///
			explicit ApplicationStatus(const StatusIdentifier& statusIdentifier);

			///
			/// @details This will destruct the ApplicationStatus object, but will not destroy resources or remove the status
			///   settings from memory or remove the status if it were visible.  Creating another ApplicationStatus object
			///   with the same StatusIdentifier value will give access to these settings once again.
			///
			/// @note this does not cause the status to be destroyed, or even cause any settings to be removed from memory,
			///   as creating another ApplicationStatus object with the same StatusIdentifier will provide access again.
			///
			~ApplicationStatus(void);

			///
			/// @details Retrieves the StatusIdentifier as specified when the object was created, this identifier cannot change
			///   for an ApplicationStatus object.
			///
			const StatusIdentifier& GetIdentifier(void) const;

			///
			/// @details Removes all items and partitions from the status bar area.  This will not be displayed to the user
			///   until the next SetWindowStatus call.
			///
			void ClearStatus(void);

			///
			/// @details Partition the status bar and add an item with a message that can later be modified using the
			///   SetStatusItem methdod with the same itemIdentifier.  The new item will not be displayed to the user
			///   until the next SetWindowStatus call.
			///
			/// @param itemIdentifier   A unique identifier to represent the status item / partition so it can be modified
			///   or accessed later.  No two items can have the same identifier, so an error will be triggered if attempted.
			/// @param statusMessage    The message to display for this item/partition.  Messages wider than the partition
			///   width may, or may not, flow into the next partition or remain partially hidden.
			/// @param partitionPlacement    A percentage based placement of the left edge the partition.  A value of 0 is
			///   the far left edge of the window, and 100 being the right most edge.  Values out of that range will trigger
			///   an error.
			///
			/// @note Maximum of 16 different status items can be added to a single ApplicationStatus object.
			///
			void AddStatusItem(const StatusItemIdentifier& itemIdentifier, const tbCore::tbString& statusMessage, const char partitionPlacement);

			///
			/// @details Update or modify the message of a specified status item.  The item will not be changed until the
			///   next SetWindowStatus call.
			///
			/// @param itemIdentifier   The identifier of the status item to be modified as added via AddStatusItem.
			/// @param statusMessage    The message to display for this item/partition.  Messages wider than the partition
			///   width may, or may not, flow into the next partition or remain partially hidden.
			///
			void SetStatusItem(const StatusItemIdentifier& itemIdentifier, const tbCore::tbString& statusMessage);

		private:
			const StatusIdentifier mIdentifier;
		};

	}; /* namespace Application */
}; /* namespace TurtleBrains */

namespace tbApplication = TurtleBrains::Application;

#endif /* _TurtleBrains_ApplicationStatus_h_ */
