///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_RealtimeApplication_h_
#define _TurtleBrains_RealtimeApplication_h_

#include "../core/tb_noncopyable.h"
#include "../core/tb_types.h"
#include "tb_application_handler_interface.h"

namespace TurtleBrains
{
	namespace Application
	{

		class ApplicationDialog;
		class ApplicationMenu;
		class ApplicationStatus;

//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details This is the value that is returned when the user presses Okay from a MessageBox or presses an action
		///   button on one of the supported system prompts/dialogs (like pressing Open on the OpenFileDialog).
		///
		extern const bool kResultOkay;

		///
		/// @details This is the value that is returned when the user presses Cancel from a MessageBox or presses escape
		///   or a cancelling action on the supported system prompts/dialogs (like pressing close on the OpenFileDialog).
		///
		extern const bool kResultCancel;

		///
		/// @details Create an application using OpenGL to render a realtime viewport.  A realtime message pump will be
		///   used instead of blocking until a system message/event is fired.
		///
		class RealtimeApplication : tbCore::Noncopyable
		{
		public:

			///
			/// @details Prepares an application to create a window with an OpenGL context.  The window will not be created
			///   or opened in the constructor of the RealtimeApplication, instead it will be delayed until Open() is called.
			///
			/// @param applicationHandler    The object to be invoked when the system fires an event or the user acts upon
			///   items within the application.  The scope of this object must outlive the scope of the RealtimeApplication
			///   object.
			///
			/// @note only a single RealtimeApplication object can exist at a single-time despite not being a singleton, this
			///   may change to either a singleton or allow multiple Application objects in future versions of TurtleBrains.
			///
			explicit RealtimeApplication(ApplicationHandlerInterface& applicationHandler);

			///
			/// @details Cleanup any resources created when preparing or running a RealtimeApplication.
			///
			~RealtimeApplication(void);

			///
			/// @details Creates and opens a realtime window with OpenGL clearing and displaying a black screen, but
			///   waits until Run() is called to enter the real-time message pump of the OS.
			///
			void Open(void);

			///
			/// @details Begins processing messages from the operating system, this should be called from the main thread
			///   as it handles application / interface objects, and will not return the application has been closed.  
			///   Once running, the ApplicationHandlerInterface object will be invoked during system events and user actions.
			///
			/// @note This function will block until Close() is called.
			///
			void Run(void);

			//
			// @details Stops processing messages from the operating system and causes Run to exit.  This allows
			//   cleanup to occur before Close() destroys the the renderer context and window.
			//
			// TODO: TIM: Implementation: This needs to be slightly refactored / reimplemented. 
			//
			//void Stop(void);

			///
			/// @details Mark the application to close, the actual close event will be delayed and may be performed after the
			///   current frame when the operating system messages are processed.
			///
			void Close(void);

			///
			/// @details Apply an ApplicationMenu to the top of the Window in a manner the user will expect depending on the
			///   operating system used.
			///
			/// @note On Mac the very first menu item will be named with the application name, it may be desired to add an
			///   additional menu item for this.
			///
			void SetWindowMenu(const ApplicationMenu& menu);

			///
			/// @details Apply an ApplicationMenu to open as the context menu when the user right-clicks / secondary clicks
			///   within the window.  This will not open or show the menu until the user takes action.
			///
			void SetContextMenu(const ApplicationMenu& menu);

			///
			/// @details Apply a status bar to the application window to inform the user about selection or status of the
			///   application.
			///
			void SetWindowStatus(const ApplicationStatus& status);

			///
			/// @details Set the window title to a specified title so the user knows where/what your application is.
			///
			void SetWindowTitle(const tbCore::tbString& windowTitle);

			///
			/// @details Opens a custom built, modeless, dialog for your application which will continue to update while
			///   your application runs.  This is not a modal dialog prompt and the function will return before the user
			///   has finished using your dialog.  The dialog will remain visible until closed with a call to CloseDialog().
			///
			/// @param dialog   The dialog box to be opened.
			///
			void OpenDialog(const ApplicationDialog& dialog);

			///
			/// @details Closes a custom built dialog prompt, must be called for any dialog opened with OpenDialog and will
			///   clean up any system resources allocated for the dialog box.
			///
			/// @param dialog   The dialog box to be closed.
			///
			void CloseDialog(const ApplicationDialog& dialog);

			///
			/// @details Opens a small, modal, message box displaying a title, a message and an Okay button.  Due to the
			///   nature of a modal prompt this function will not return until the user has pressed the Okay button.
			///
			/// @param title    The title of the message box, it is wise to contain your application name for user reference.
			/// @param message  The message that the user needs to see and accept with the Okay button.
			///
			/// @return kResultOkay will be returned once the user presses the Okay button.
			///
			bool OkayMessageBox(const tbCore::tbString& title, const tbCore::tbString& message);

			///
			/// @details Opens a small, modal, message box displaying a title, a message and two buttons to the user to
			///   press Okay or Cancel.  The function will not return until the user has pressed either to Okay or Cancel
			///   buttons, and the result will be returned.
			///
			/// @param title    The title of the message box, it is wise to contain your application name for user reference.
			/// @param message  The message that the user needs to see and accept or decline with the Okay/Cancel buttons.
			///
			/// @return kResultOkay if the user pressed the Okay button or kResultCancel if the user pressed Cancel.
			///
			bool OkayCancelMessageBox(const tbCore::tbString& title, const tbCore::tbString& message);

			///
			/// @details Opens a dialog prompt to navigate/open a file using the native prompt on the operating system.
			///   This is a modal dialog prompt and the function will not return until the user has finished.
			///
			/// @return kResultOkay if the user pressed Open on the file dialog prompt, or kResultCancel if the user closed
			///   the prompt, hit escape or some other action to close without accepting the file.
			///
			/// @note Currently the fileExtension parameter is ignored so any file with any extension is used.
			///
			bool OpenFileDialog(tbCore::tbString& fileToOpen, const tbCore::tbString& fileExtension);

			///
			/// @details Opens a dialog prompt to navigate to a file for saving using the native prompt on the operating 
			///   system.  This is a modal dialog prompt and the function will not return until the user has finished.
			///
			/// @return kResultOkay if the user pressed Save on the file dialog prompt, or kResultCancel if the user closed
			///   the prompt, hit escape or some other action to close without accepting the file.
			///
			/// @note Currently the fileExtension parameter is ignored so any file with any extension is used.
			///
			bool SaveFileDialog(tbCore::tbString& fileToSave, const tbCore::tbString& fileExtension);

		private:
			ApplicationHandlerInterface& mApplicationHandler;
		};

	}; /* namespace Application */
}; /* namespace TurtleBrains */

namespace tbApplication = TurtleBrains::Application;

#endif /* _TurtleBrains_RealtimeApplication_h_ */
