///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tbi_audio_data.h"
#include "../../core/tb_defines.h"

#define STB_VORBIS_HEADER_ONLY
#include "stb_vorbis.c"

#include <cstring>
#include <openal/al.h>
#include <openal/alc.h>

namespace //tbAudioImplementation
{
	struct MemoryBuffer: std::streambuf
	{
		MemoryBuffer(char* base, std::ptrdiff_t n)
		{
			setg(base, base, base + n);
		}
	};
}; /* namespace tbAudioImplementation */

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioData::AudioData(const tbCore::tbString& fileName) :
	mFileName(fileName),
	mAudioData(nullptr),
	mAudioFile(new std::ifstream(fileName.c_str(), std::ios_base::binary | std::ios_base::in)),
	mSampleRate(0),
	mBitsPerSample(0),
	mChannels(0),
	mAudioDataSize(0)
{
	tb_error_if(nullptr == mAudioFile, "tbInternalError: Failed to allocate audio file stream.\n");
	tb_error_if(false == mAudioFile->good(), "tbExternalError: Failed to load audio file, \"%s\".", fileName.c_str());
	tb_error_if(false == InitializeData(), "tbError: Was unable to initialize audio data for file, \"$s\".", fileName.c_str());
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioData::AudioData(const unsigned char* fileDataInMemory, const size_t& fileSizeInBytes) :
	mFileName(tbCore::ToString("")),
	mAudioData(nullptr),
	mAudioFile(nullptr),
	mSampleRate(0),
	mBitsPerSample(0),
	mChannels(0),
	mAudioDataSize(0)
{
	tb_error_if(nullptr == fileDataInMemory, "tbExternalError: Invalid file data, expected valid memory buffer containing file.");
	tb_error_if(0 == fileSizeInBytes, "tbExternalError: Invalid file size, 0 bytes.");

	//Is expect the streamBuffer read only, and thus the case should be okay to improve peformance (instead of allocating
	//and copying the memory into the stream buffer just use it in place. -TIM 20150517
	MemoryBuffer streamBuffer((char*)fileDataInMemory, fileSizeInBytes);
	mAudioFile = new std::istream(&streamBuffer);
	tb_error_if(false == InitializeData(), "tbError: Was unable to initialize audio data for file, \"$s\".", mFileName.c_str());

	mAudioData = new tbCore::uint8[mAudioDataSize];
	mAudioFile->read(reinterpret_cast<std::fstream::char_type*>(mAudioData), mAudioDataSize);
	//Don't need to reset file location since we are closing it.	

	tb_safe_delete(mAudioFile); //No longer needed.

}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioData::~AudioData(void)
{
	tb_safe_delete(mAudioFile);	
	tb_safe_array_delete(mAudioData);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioData::GetAudioData(tbCore::uint8* audioData, size_t& dataSize) const
{
	if (nullptr != mAudioData)
	{
		memcpy(audioData, mAudioData, dataSize);
	}
	else
	{
		tb_error_if(nullptr == mAudioFile, "tbInternalError: Expected audio file to be valid.");
		tb_error_if(dataSize != mAudioDataSize, "tbInternalError: Collecting partial audio data for streaming players is not yet supported.");
		const std::fstream::streampos dataStart(mAudioFile->tellg());
		mAudioFile->read(reinterpret_cast<std::fstream::char_type*>(audioData), dataSize);
		mAudioFile->seekg(dataStart);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbImplementation::AudioData::GetAudioDataSize(void) const
{
	return mAudioDataSize;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::uint32 tbImplementation::AudioData::GetFormat(void) const
{
//	if (1 == mChannels && 8 == mBitsPerSample) { return AL_FORMAT_MONO8; }
//	if (2 == mChannels && 8 == mBitsPerSample) { return AL_FORMAT_STEREO8; }
//	if (1 == mChannels && 16 == mBitsPerSample) { return AL_FORMAT_MONO16; }
//	if (2 == mChannels && 16 == mBitsPerSample) { return AL_FORMAT_STEREO16; }

	if (1 == mChannels)
	{
		if (8 == mBitsPerSample) { return AL_FORMAT_MONO8; }
		else { return AL_FORMAT_MONO16; }
	}
	else if (2 == mChannels)
	{
		if (8 == mBitsPerSample) { return AL_FORMAT_STEREO8; }
		else { return AL_FORMAT_STEREO16; }
	}

	return 0;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::uint32 tbImplementation::AudioData::GetFrequency(void) const
{
	return mSampleRate;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::tbString& tbImplementation::AudioData::GetFileName(void) const
{
	return mFileName;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::AudioData::InitializeData(void)
{
	if (tbCore::tbString::npos != mFileName.find(".ogg"))
	{
		return InitializeOggData();
	}

	const size_t kRiffHeaderSize(44);
	std::fstream::char_type riffHeader[kRiffHeaderSize];
	mAudioFile->read(riffHeader, kRiffHeaderSize);

	if ('R' != riffHeader[0] || 'I' != riffHeader[1] || 'F' != riffHeader[2] || 'F' != riffHeader[3] ||
		'W' != riffHeader[8] || 'A' != riffHeader[9] || 'V' != riffHeader[10] || 'E' != riffHeader[11] ||
		1 != *(reinterpret_cast<short*>(&riffHeader[20])))
	{
		tb_error_if(true, "tbExternalError: Failed to process audio file, \"%s\", audio file is not a supported WAV file.", mFileName.c_str());
		return false;
	}

	mChannels = *(reinterpret_cast<short*>(&riffHeader[22]));
	mSampleRate = *(reinterpret_cast<int*>(&riffHeader[24]));
	mBitsPerSample = *(reinterpret_cast<short*>(&riffHeader[34]));
	mAudioDataSize = *(reinterpret_cast<unsigned int*>(&riffHeader[40]));

	while ('d' != riffHeader[36] || 'a' != riffHeader[37] || 't' != riffHeader[38] || 'a' != riffHeader[39])
	{
		const size_t currentPosition(static_cast<size_t>(mAudioFile->tellg()));
		mAudioFile->seekg(currentPosition + mAudioDataSize);
		mAudioFile->read(&riffHeader[36], 8);
		mAudioDataSize = *(reinterpret_cast<unsigned int*>(&riffHeader[40]));

		if (mAudioFile->eof() || mAudioFile->fail())
		{
			tb_error("tbExternalError: Was unable to find data chunk in wave file.");
			return false;
		}
	}

	tb_error_if(1 != mChannels && 2 != mChannels, "tbExternalError: Audio file \"%s\" is not a mono or stereo WAV file.", mFileName.c_str());
	tb_error_if(8 != mBitsPerSample && 16 != mBitsPerSample, "tbExternalError: Audio file \"%s\" is not a 8bit or 16bit WAV file.", mFileName.c_str());

	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::AudioData::InitializeOggData(void)
{
//	int errorCode(0);
//	stb_vorbis* oggData = stb_vorbis_open_filename(mFileName.c_str(), &errorCode, nullptr);
//	if (nullptr == oggData)
//	{
//		return false;
//	}
//
//	stb_vorbis_info oggInfo = stb_vorbis_get_info(oggData);
//	mChannels = oggInfo.channels;
//	mSampleRate = oggInfo.sample_rate;
//	//mBitsPerSample
//	//mAudioDataSize
//
//	stb_vorbis_close(oggData);
//	oggData = nullptr;

	int channels(0);
	int sampleRate(0);
	short* outputData(nullptr);
	int decodedSamples = stb_vorbis_decode_filename(mFileName.c_str(), &channels, &sampleRate, &outputData);
	if (-1 == decodedSamples)
	{
		return false;
	}

	mChannels = static_cast<tbCore::uint32>(channels);
	mSampleRate = static_cast<tbCore::uint32>(sampleRate);
	mBitsPerSample = 16;

	mAudioDataSize = (2 * mChannels * decodedSamples);
	mAudioData = new tbCore::uint8[mAudioDataSize];
	memcpy(mAudioData, outputData, mAudioDataSize);

	free(outputData);
	outputData = nullptr;

	return true;
}

//--------------------------------------------------------------------------------------------------------------------//
