///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tbi_audio_player.h"
#include "tbi_audio_data.h"
#include "../../core/tb_error.h"
#include "../../debug_tool_set/tb_debug_logger.h"

#include <openal/al.h>
#include <openal/alc.h>

#include <map>

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioPlayer::AudioPlayer(const tbAudio::AudioChannel& audioChannel) :
	mIdentifier(audioChannel),
	mBuffer(0),
	mSource(0),
	mIsLooping(false)
{
	alGenBuffers(1, &mBuffer);
	tb_error_if(AL_NO_ERROR != alGetError(), "tbInternalError: Could not create buffer for audio player.");

	alGenSources(1, &mSource);
	tb_error_if(AL_NO_ERROR != alGetError(), "tbInternalError: Could not create source for audio player.");

	alSourcei(mSource, AL_SOURCE_RELATIVE, AL_TRUE);
	alSourcei(mSource, AL_ROLLOFF_FACTOR, 1);

	alSourcef(mSource, AL_REFERENCE_DISTANCE, 100.0f);
	alSourcef(mSource, AL_MAX_DISTANCE, 110.0f);

	alSourcef(mSource, AL_PITCH, 1.0f);
	alSourcef(mSource, AL_GAIN, 1.0f);
	alSource3f(mSource, AL_POSITION, 0.0f, 0.1f, 0.0f);
	alSource3f(mSource, AL_VELOCITY, 0.0f, 0.0f, 0.0f);

	tb_error_if(alGetError() != AL_NO_ERROR, "tbInternalError: Could not set source parameters for audio player.");
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioPlayer::~AudioPlayer(void)
{
	alDeleteSources(1, &mSource);
	alDeleteBuffers(1, &mBuffer);
	tb_log_if(alGetError() != AL_NO_ERROR, "tbInternalWarning: Failed to delete source/buffer IDs for audio player.");
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioPlayer::PlayAudio(const AudioData& audioData, const bool isLooping)
{
	mIsLooping = isLooping;

	//Rewind the source position and clear the buffer queue.
	alSourceRewind(mSource);
	alSourcei(mSource, AL_BUFFER, 0);

	size_t rawSize(audioData.GetAudioDataSize());
	tbCore::uint8* rawData = new tbCore::uint8[rawSize];
	audioData.GetAudioData(rawData, rawSize);

	//Should use something like: alSourcei(mSource, AL_BUFFER, buffer) with the buffered sounds...

	alBufferData(mBuffer, audioData.GetFormat(), rawData, static_cast<int>(rawSize), audioData.GetFrequency());
	tb_error_if(alGetError() != AL_NO_ERROR, "tbInternalError: Could not set audio data in buffer.");

	alSourceQueueBuffers(mSource, 1, &mBuffer);
	alSourcePlay(mSource);
	tb_log_if(alGetError() != AL_NO_ERROR, "tbInternalWarning: Could not start audio player.");
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioPlayer::Stop(void)
{
	alSourceStop(mSource);
	alSourceUnqueueBuffers(mSource, 1, &mBuffer);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioPlayer::StopLooping(void)
{
	mIsLooping = false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioPlayer::SetPitch(float pitch)
{
	tb_error_if(pitch < 0.0f, "tbExternalError: Expected pitch to remain a positive value.");
	alSourcef(mSource, AL_PITCH, pitch);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::AudioPlayer::SetVolume(float volume)
{
	tb_error_if(volume < 0.0f || volume > 1.0f, "tbExternalError: Expected volume to remain within range: (0.0f <= volume <= 1.0f).");
	alSourcef(mSource, AL_GAIN, volume);
}

//--------------------------------------------------------------------------------------------------------------------//

float tbImplementation::AudioPlayer::GetVolume(void) const
{
	float volume(1.0f);
	alGetSourcef(mSource, AL_GAIN, &volume);
	return volume;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::AudioPlayer::Update(void)
{
	int processedCount(0);
	alGetSourcei(mSource, AL_BUFFERS_PROCESSED, &processedCount);
	while (processedCount > 0)
	{
		if (true == mIsLooping)
		{
			alSourceRewind(mSource);
			alSourcePlay(mSource);
		}
		else
		{
			unsigned int unqueuedBuffer(0);
			alSourceUnqueueBuffers(mSource, 1, &unqueuedBuffer);
		}
		
		--processedCount;
	}

	return IsComplete();
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::AudioPlayer::IsComplete(void) const
{
	int state(0);
	alGetSourcei(mSource, AL_SOURCE_STATE, &state);

	int queuedBuffers(0);
	alGetSourcei(mSource, AL_BUFFERS_QUEUED, &queuedBuffers);
	tb_log_if(alGetError() != AL_NO_ERROR, "tbInternalWarning: Could not get player information.");

	return (AL_STOPPED == state || (0 == queuedBuffers && false == mIsLooping)) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbAudio::AudioChannel& tbImplementation::AudioPlayer::GetIdentifier(void) const
{
	return mIdentifier;
}

//--------------------------------------------------------------------------------------------------------------------//

//--------------------------------------------------------------------------------------------------------------------//

//--------------------------------------------------------------------------------------------------------------------//
