///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_audio_manager.h"

#include "implementation/tbi_audio_data.h"
#include "implementation/tbi_audio_player.h"
#include "implementation/tbi_audio_event_table.h"
#include "implementation/tbi_turtle_brains_jingle.h"

#include "../debug_tool_set/tb_debug_logger.h"
#include "../core/tb_json_parser.h"
#include "../core/tb_dynamic_structure.h"
#include "../math/tb_random_numbers.h"

#include <openal/al.h>
#include <openal/alc.h>

#include <map>
#include <list>

tbAudio::AudioManager tbAudio::theAudioManager;
const tbAudio::AudioHandle tbAudio::kInvalidAudio(0);
const tbAudio::AudioChannel tbAudio::kInvalidChannel(0);

namespace tbImplementation
{
	const tbAudio::AudioHandle& GetAudioFromCache(const tbCore::tbString& filename);
	void IncrementReferenceOf(const tbAudio::AudioHandle& audioHandle);
	void OnDestroyAudioData(const tbAudio::AudioHandle& audioHandle);

	tbAudio::AudioChannel GetFreeAudioChannelIdentifier(void);
	tbImplementation::AudioPlayer* GetAudioChannel(const tbAudio::AudioChannel& audioChannel, const tbAudio::AudioHandle& audioData = tbAudio::kInvalidAudio);

	typedef std::map<tbCore::tbString, tbAudio::AudioHandle> AudioCacheContainer;
	typedef std::map<tbAudio::AudioHandle, AudioData*> AudioDataContainer;
	typedef std::list<std::pair<tbImplementation::AudioPlayer*, tbAudio::AudioHandle> > AudioPlayerContainer;
	typedef std::map<tbCore::tbString, AudioEventTable*> EventTableContainer;

	AudioCacheContainer theAudioCache;  //Stores audio handles to all textures loaded by file.
	AudioDataContainer theAudioData;    //Stores ALL audio data, this is where the sound data is.
	AudioPlayerContainer theAudioPlayers;
	EventTableContainer theEventTables;

	ALCdevice* tbiAudioDevice(nullptr);
	ALCcontext* tbiAudioContext(nullptr);

	tbAudio::AudioChannel tbiAudioChannel(tbAudio::kInvalidChannel);
	const size_t kMaxSoundPlayers(32);
	const bool kAudioManagerLogging(true);

	//Shared / accessed in tbGame::DefaultScene
	tbAudio::AudioHandle tbiTurtleBrainsJingleAudio(tbAudio::kInvalidAudio);

}; /* namespace tbImplementation */

using namespace tbImplementation;

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

const tbAudio::AudioHandle& tbImplementation::GetAudioFromCache(const tbCore::tbString& filename)
{
	AudioCacheContainer::const_iterator audioIterator(theAudioCache.find(filename));
	if (audioIterator != theAudioCache.end())
	{
		return audioIterator->second;
	}
	return tbAudio::kInvalidAudio;
}

//--------------------------------------------------------------------------------------------------------------------//

//void tbImplementation::IncrementReferenceOf(const tbAudio::AudioHandle& audioHandle)
//{
//	AudioDataContainer::iterator audioIterator(theAudioData.find(audioHandle));
//	tb_error_if(audioIterator == theAudioData.end(), "tbError: Attempting to IncrementReferenceOf an AudioHandle that was not found.");
//	++audioIterator->second->mReferences;
//}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OnDestroyAudioData(const tbAudio::AudioHandle& audioHandle)
{
	AudioDataContainer::iterator audioDataIterator(theAudioData.find(audioHandle));
	if (audioDataIterator != theAudioData.end())
	{
		AudioData* audioData(audioDataIterator->second);
//		tb_error_if(0 == audioData.mReferences, "tbInternalError: Never ")
//		--audioData.mReferences;

//		if (0 == audioData.mReferences)
		{
			theAudioCache.erase(audioData->GetFileName());
			theAudioData.erase(audioDataIterator);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioChannel tbImplementation::GetFreeAudioChannelIdentifier(void)
{
	++tbiAudioChannel;
	if (tbAudio::kInvalidChannel == tbiAudioChannel)
	{
		tb_error(L"tbInternalError: Audio Channel Identifier Rolled Over.");
		++tbiAudioChannel; //This may already be used.
	}

	return tbiAudioChannel;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::AudioPlayer* tbImplementation::GetAudioChannel(const tbAudio::AudioChannel& audioChannel, const tbAudio::AudioHandle& audioData)
{
	for (AudioPlayerContainer::const_iterator itr = theAudioPlayers.begin(), itrEnd = theAudioPlayers.end(); itr != itrEnd; ++itr)
	{
		tbImplementation::AudioPlayer& audioPlayer(*(itr->first));
		if (audioChannel == audioPlayer.GetIdentifier())
		{ //If the audio data matches, or match not important, return true, otherwise false the caller audioData was important and did not match.
			return (tbAudio::kInvalidAudio == audioData || itr->second == audioData) ? itr->first : nullptr;
		}
	}
	return nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioManager::AudioManager(void)
{
	tbiAudioDevice = alcOpenDevice(NULL);
	tb_error_if(nullptr == tbiAudioDevice, "AudioManager could not create the default audio device.");
	tbiAudioContext = alcCreateContext(tbiAudioDevice, NULL);
	tb_error_if(nullptr == tbiAudioContext, "AudioManager could not create an audio context.");
	alcMakeContextCurrent(tbiAudioContext);
	tb_error_if(nullptr == tbiAudioContext, "AudioManager could not set audio context.");

	tb_log_if(true == kAudioManagerLogging, "AudioManager opened: \"%s\"\n", alcGetString(tbiAudioDevice, ALC_DEVICE_SPECIFIER));

	float zero[] = { 0.0f, 0.0f, 0.0f };
	float orientation[] = { 0.0f, 0.0f, -1.0f,    0.0f, 1.0f, 0.0f }; //at then up
	alListenerfv(AL_POSITION, zero);
	tb_error_if(AL_NO_ERROR != alGetError(), "AudioManager had issues setting listener position.");
	alListenerfv(AL_VELOCITY, zero);
	tb_error_if(AL_NO_ERROR != alGetError(), "AudioManager had issues setting listener velocity.");
	alListenerfv(AL_ORIENTATION, orientation);
	tb_error_if(AL_NO_ERROR != alGetError(), "AudioManager had issues setting listener orientation.");
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioManager::~AudioManager(void)
{
	//TODO: TIM: We really should be cleaning up here, but because of the static AudioManager and containers, the
	//containers have already been destroyed, possibly...
//	for (AudioDataContainer::iterator itr = theAudioData.begin(); itr != theAudioData.end(); ++itr)
//	{ //Clean up after all the loaded sounds that have not been cleaned up, perhaps issue a warning?
//		delete itr->second;
////		itr = theAudioData.erase(itr);
//	}
//
//	theAudioData.clear();
//	theAudioCache.clear();
//
//	while (false == theAudioPlayers.empty())
//	{ //Cleanup after all the players created by playing multiple sounds.
//		delete theAudioPlayers.back();
//		theAudioPlayers.pop_back();
//	}

	alcMakeContextCurrent(nullptr);

	if (nullptr != tbiAudioContext)
	{
		alcDestroyContext(tbiAudioContext);
		tbiAudioContext = nullptr;
	}

	if (nullptr != tbiAudioDevice)
	{
		alcCloseDevice(tbiAudioDevice);
		tbiAudioDevice = nullptr;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbAudio::AudioManager::LoadEventTable(const tbCore::tbString& eventTableName, const tbCore::tbString& eventTableFile)
{
	if (theEventTables.end() != theEventTables.find(eventTableName))
	{
		tb_error("EventTable \"%s\" has already been loaded.");
		return false;
	}

	const tbCore::DynamicStructure& eventTableData(tbCore::LoadJsonFile(eventTableFile));

	if (true == eventTableData.IsNil() || false == eventTableData.IsStructure())
	{
		tb_error("tbDataError: Expected event table data to be a structure of information. Table: \"%s\"", eventTableFile.c_str());
		return false;
	}

	const tbCore::DynamicStructure& audioEvents(eventTableData["audio_events"]);
	if (true == audioEvents.IsNil() || false == audioEvents.IsArray())
	{
		tb_error("tbDataError: Expected event table to have an array named: audio_events. Table: \"%s\"", eventTableFile.c_str());
		return false;
	}

	AudioEventTable* audioEventTable = new AudioEventTable();

	for (size_t eventIndex(0); eventIndex < audioEvents.Size(); ++eventIndex)
	{
		const tbCore::DynamicStructure& eventData(audioEvents[eventIndex]);
		if (true == eventData["name"].IsNil())
		{ //Maybe just make this a warning and discard the event?
			tb_error("tbDataError: Expected each audio_event object to have a \"name\". Table: \"%s\"", eventTableFile.c_str());
			delete audioEventTable;
			return false;
		}

		if (true == eventData["file"].IsNil())
		{ //Maybe just make this a warning and discard the event?
			delete audioEventTable;
			tb_error("tbDataError: Expected each audio_event object to have a \"file\". Table: \"%s\"", eventTableFile.c_str());
			return false;
		}

		bool isLooping = eventData["looping"].AsBooleanWithDefault(false);

		const float eventVolume[3] = {
			eventData["volume"].AsFloatWithDefault(1.0f),
			eventData["min_volume"].AsFloatWithDefault(1.0f),
			eventData["max_volume"].AsFloatWithDefault(1.0f)
		};

		const float eventPitch[3] = {
			eventData["pitch"].AsFloatWithDefault(1.0f),
			eventData["min_pitch"].AsFloatWithDefault(1.0f),
			eventData["max_pitch"].AsFloatWithDefault(1.0f)
		};

		const tbCore::tbString eventName(eventData["name"].AsString());
		const tbCore::DynamicStructure& eventFileData(eventData["file"]);
		if (true == eventFileData.IsArray())
		{
			for (size_t fileIndex(0); fileIndex < eventFileData.Size(); ++fileIndex)
			{
				if (true == eventFileData[fileIndex].IsStructure())
				{
					tbCore::tbString audioFilename(eventFileData[fileIndex]["name"]);
					AudioHandle audioHandle(CreateSoundFromFile(audioFilename));
					if (kInvalidAudio == audioHandle)
					{
						delete audioEventTable;
						tb_error("tbDataError: Expected to load audio file: \"%s\" on Table: \"%s\"", audioFilename.c_str(), eventTableName.c_str());
						return false;
					}

					const tbCore::DynamicStructure& fileData(eventFileData[fileIndex]);
					const float fileVolume[3] = {
						fileData["volume"].AsFloatWithDefault(eventVolume[0]),
						fileData["min_volume"].AsFloatWithDefault(eventVolume[1]),
						fileData["max_volume"].AsFloatWithDefault(eventVolume[2])
					};

					const float filePitch[3] = {
						fileData["pitch"].AsFloatWithDefault(eventPitch[0]),
						fileData["min_pitch"].AsFloatWithDefault(eventPitch[1]),
						fileData["max_pitch"].AsFloatWithDefault(eventPitch[2])
					};

					audioEventTable->AddAudioEvent(eventName, audioHandle, isLooping);
					audioEventTable->SetAudioEventVolume(eventName, fileVolume[0], fileVolume[1], fileVolume[2]);
					audioEventTable->SetAudioEventPitch(eventName, filePitch[0], filePitch[1], filePitch[2]);
				}
				else
				{
					tbCore::tbString audioFilename(eventFileData[fileIndex].AsString());
					AudioHandle audioHandle(CreateSoundFromFile(audioFilename));
					if (kInvalidAudio == audioHandle)
					{
						delete audioEventTable;
						tb_error("tbDataError: Expected to load audio file: \"%s\" on Table: \"%s\"", audioFilename.c_str(), eventTableName.c_str());
						return false;
					}
					audioEventTable->AddAudioEvent(eventName, audioHandle, isLooping);
					audioEventTable->SetAudioEventVolume(eventName, eventVolume[0], eventVolume[1], eventVolume[2]);
					audioEventTable->SetAudioEventPitch(eventName, eventPitch[0], eventPitch[1], eventPitch[2]);
				}
			}
		}
		else
		{
			tbCore::tbString audioFilename(eventFileData.AsString());
			AudioHandle audioHandle(CreateSoundFromFile(audioFilename));
			if (kInvalidAudio == audioHandle)
			{
				delete audioEventTable;
				tb_error("tbDataError: Expected to load audio file: \"%s\" on Table: \"%s\"", audioFilename.c_str(), eventTableName.c_str());
				return false;
			}

			audioEventTable->AddAudioEvent(eventName, audioHandle, isLooping);
			audioEventTable->SetAudioEventVolume(eventName, eventVolume[0], eventVolume[1], eventVolume[2]);
			audioEventTable->SetAudioEventPitch(eventName, eventPitch[0], eventPitch[1], eventPitch[2]);
		}
	}

	theEventTables[eventTableName] = audioEventTable;
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioHandle tbAudio::AudioManager::CreateSoundFromFile(const tbCore::tbString& filename)
{
	AudioHandle audioHandle = GetAudioFromCache(filename);

	if (kInvalidAudio == audioHandle)
	{
		audioHandle = static_cast<AudioHandle>(theAudioData.size() + 1);
		AudioData* audioData = new AudioData(filename);

		theAudioCache.insert(AudioCacheContainer::value_type(filename, audioHandle));
		theAudioData.insert(AudioDataContainer::value_type(audioHandle, audioData));
	}
	else
	{
		//TODO: TIM: Reference Counting for the AudioData?
	}

	return audioHandle;
}

//--------------------------------------------------------------------------------------------------------------------//

//tbAudio::AudioHandle tbAudio::AudioManager::CreateSoundFromData(const unsigned char* fileDataInMemory, const size_t& fileSizeInBytes)
//{
//	AudioHandle audioHandle(static_cast<AudioHandle>(theAudioData.size() + 1));
//	AudioData* audioData = new AudioData(fileDataInMemory, fileSizeInBytes);
//
//	//theAudioCache.insert(AudioCacheContainer::value_type(filename, audioHandle));
//	theAudioData.insert(AudioDataContainer::value_type(audioHandle, audioData));
//
//	return audioHandle;
//}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioManager::DestroySound(const AudioHandle& audioHandle)
{
	tbImplementation::OnDestroyAudioData(audioHandle);
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioController tbAudio::AudioManager::PlayEvent(const tbCore::tbString& eventTableName, const tbCore::tbString& eventName)
{
	EventTableContainer::iterator eventTableIterator(theEventTables.find(eventTableName));
	tb_error_if(eventTableIterator == theEventTables.end(), "tbExternalError: No audio table with name \"%s\" exists, was it loaded or mispelled?", eventTableName.c_str());

	const tbImplementation::AudioEvent& audioEvent(eventTableIterator->second->GetAudioEvent(eventName));
	AudioController audioController(PlaySound(audioEvent.mAudioHandle, audioEvent.mIsLooping));

	float volume(audioEvent.mVolumeControls[0]);
	const float volumeRange(audioEvent.mVolumeControls[2] - audioEvent.mVolumeControls[1]);
	if (volumeRange > 0.01f) { volume = audioEvent.mVolumeControls[1] + (tbMath::RandomFloat() * volumeRange); }
	audioController.SetVolume(volume);

	float pitch(audioEvent.mPitchControls[0]);
	const float pitchRange(audioEvent.mPitchControls[2] - audioEvent.mPitchControls[1]);
	if (pitchRange > 0.01f) { pitch = audioEvent.mPitchControls[1] + (tbMath::RandomFloat() * pitchRange); }
	audioController.SetPitch(pitch);

	return audioController;
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioController tbAudio::AudioManager::PlaySound(const AudioHandle& audioHandle, const bool isLooping)
{	
	if (kInvalidAudio == tbiTurtleBrainsJingleAudio)
	{	//Load up the Jingle file from memory if it hasn't already been loaded.
		AudioData* audioData = new AudioData(theTurtleBrainsJingleSourceData, theTurtleBrainsJingleSourceSize);
		if (nullptr != audioData)
		{	//TODO: TIM: Fix: This is not a valid way to create the audio handle, consider the case if they get unloaded.
			tbiTurtleBrainsJingleAudio = static_cast<AudioHandle>(theAudioData.size() + 1);
			theAudioData.insert(AudioDataContainer::value_type(tbiTurtleBrainsJingleAudio, audioData));
		}
	}

	tb_error_if(kInvalidAudio == audioHandle, "tbExternalError: Attempting to play an invalid audio handle.");
	AudioDataContainer::iterator audioIterator(theAudioData.find(audioHandle));
	tb_error_if(audioIterator == theAudioData.end(), "tbError: Unable to find audio handle to play, was it unloaded?");

	const AudioData& audioData(*audioIterator->second);

	//Find a player to start playing this sound, or create a new player.
//	AudioPlayerContainer::iterator itr = theAudioPlayers.begin();
//	AudioPlayerContainer::iterator itrEnd = theAudioPlayers.end();
//	for ( ; itr != itrEnd; ++itr)
//	{
//		tbImplementation::AudioPlayer& audioPlayer(*(itr->first));
//		if (true == audioPlayer.IsComplete())
//		{
//			audioPlayer.PlayAudio(audioData);
//			break;
//		}
//	}

//	if (itr == itrEnd)
	{
		if (theAudioPlayers.size() < kMaxSoundPlayers)
		{
			AudioChannel audioChannel(tbImplementation::GetFreeAudioChannelIdentifier());

			AudioController audioController(audioHandle);
			audioController.mAudioChannel = audioChannel;

			theAudioPlayers.push_back(AudioPlayerContainer::value_type(new tbImplementation::AudioPlayer(audioChannel), audioHandle));
			theAudioPlayers.back().first->PlayAudio(audioData, isLooping);

			return audioController;
		}
		else
		{
			tb_log_if(kAudioManagerLogging, "tbExternalError: Attempting to play more than max (%d) sounds simultaneously.\n", kMaxSoundPlayers);
		}
	}

	return AudioController(kInvalidAudio);
}

//--------------------------------------------------------------------------------------------------------------------//

//tbAudio::AudioController tbAudio::AudioManager::PlaySound(const tbCore::tbString& filename, const bool isLooping)
//{
//	AudioHandle audioHandle = CreateSoundFromFile(filename);
//	tb_error_if(kInvalidAudio == audioHandle, "tbExternalError: Failed to load audio file: \"%s\"", filename.c_str());
//	return PlaySound(audioHandle, isLooping);
//}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioManager::Update(void)
{
	for (AudioPlayerContainer::iterator itr = theAudioPlayers.begin(), itrEnd = theAudioPlayers.end() ; itr != itrEnd; /* in loop */)
	{
		tbImplementation::AudioPlayer& audioPlayer(*(itr->first));
		audioPlayer.Update();

		if (true == audioPlayer.IsComplete())
		{
			delete itr->first;
			itr = theAudioPlayers.erase(itr);
		}
		else
		{
			++itr;
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioController::AudioController(const AudioHandle& audioHandle) :
	mAudioData(audioHandle),
	mAudioChannel(kInvalidChannel)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioController::AudioController(const AudioController& other) :
	mAudioData(other.mAudioData),
	mAudioChannel(other.mAudioChannel)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbAudio::AudioController::~AudioController(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioController::Stop(void)
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		audioPlayer->Stop();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioController::StopLooping(void)
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		audioPlayer->StopLooping();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioController::SetPitch(const float pitch)
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		audioPlayer->SetPitch(pitch);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbAudio::AudioController::SetVolume(const float volume)
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		audioPlayer->SetVolume(volume);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

float tbAudio::AudioController::GetVolume(void) const
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		return audioPlayer->GetVolume();
	}

	return 0.0f;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbAudio::AudioController::IsComplete(void) const
{
	tbImplementation::AudioPlayer* audioPlayer(tbImplementation::GetAudioChannel(mAudioChannel, mAudioData));
	if (nullptr != audioPlayer)
	{
		return audioPlayer->IsComplete();
	}

	return true;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
