///
/// @file
///	@details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_DynamicStructure_h_
#define _TurtleBrains_DynamicStructure_h_

#include "tb_string.h"

#include <cmath>
#include <vector>
#include <map>
#include <string>

namespace TurtleBrains
{
	namespace Core
	{

		///
		/// @details Provide a simple interface for creating structures that can be changed dynamically, for file 
		///   loading, scripting and other exchanges of complex objects or structures in a semi defined manner.
		///
		class DynamicStructure
		{
		public:
			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			typedef std::vector<DynamicStructure> ArrayContainer;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			typedef std::map<tbCore::tbString, DynamicStructure> StructureContainer;

			///
			/// @details Creates an empty "Nil" structure with no values or data contained.
			///
			DynamicStructure(void);

			///
			/// @details Creates a DynamicStructure object from an integer value that will represent the integer.
			///	@note implicit conversion is desired in this case so we can do: object.AddMember("gearCount", 5);
			///
			DynamicStructure(const int& integerValue);

			///
			/// @details Creates a DynamicStructure object from a float value that will represent the float.
			///	@note implicit conversion is desired in this case so we can do: object.AddMember("gearRatio", 2.13f);
			///
			DynamicStructure(const float& floatValue);

			///
			/// @details Creates a DynamicStructure object from a boolean value that will represent the boolean.
			///	@note implicit conversion is desired in this case so we can do: object.AddMember("hasReverse", false);
			///
			DynamicStructure(const bool& booleanValue);

			///
			/// @details Creates a DynamicStructure object from a string value that will contain and represent the string.
			///	@note implicit conversion is desired in this case so we can do: object.AddMember("modelName", "MX5");
			///
			DynamicStructure(const tbCore::tbString& stringValue);

			///
			/// @details Creates a DynamicStruction object from an existing object.  Because the objects can be quite
			///   large and complex, this could potentially use a fair bit of memory or copy overhead.
			///
			DynamicStructure(const DynamicStructure& other);

			///
			/// @details Cleans up the current object then assigns itself to an existing object.  Because the objects 
			///   can be quite large and complex, this could potentially use a fair bit of memory or copy overhead.
			///
			DynamicStructure& operator=(const DynamicStructure& other);

			///
			/// @details Cleans up after the DynamicStructure which for simple objects does not require any cleanup
			///   but for strings, arrays or structure objects can involve deleting any memory allocated during the
			///   lifespan of the structure, and any child structure as well.
			///
			~DynamicStructure(void);


			///
			/// @details Checks to see if the dynamic structure is a Nil object, representing no data, empty/clean,
			///   and returns the result, true if the object is Nil otherwise false.
			///
			/// @note This may be renamed to IsNull().
			///
			bool IsNil(void) const;

			///
			/// @details Checks to see if the object is holding an array returning true if it is, otherwise false.
			///
			bool IsArray(void) const;

			///
			/// @details Checks to see if the object is a structure type which could hold multiple other objects within
			///   returning true if it is a structure type, otherwise false.
			///
			bool IsStructure(void) const;

			///
			/// @details Checks to see if the object is holding an integer type returning true if it is.
			///
			bool IsInteger(void) const;

			///
			/// @details Checks to see if the object is holding an float type returning true if it is.
			///
			bool IsFloat(void) const;

			///
			/// @details Checks to see if the object is holding an boolean type returning true if it is.
			///
			bool IsBoolean(void) const;

			///
			/// @details Checks to see if the object is holding an string type returning true if it is.
			///
			bool IsString(void) const;


			///
			/// @details Gets the value of the object as an integer using implicit conversions if allowed, otherwise
			///   an error condition will be triggered if the Type is not an integer and implicitConversion is false.
			///
			/// @param implicitConversion Set to true to allow other basic types (Bool, Float, String, Nil) to convert
			///   to an integer value automatically.  If false, no conversion will be attempted and an error condition
			///   will be triggered.
			///
			int AsInteger(bool implicitConversion = kImplicitConversions) const;

			///
			/// @details Gets the value of the object as a float using implicit conversions if allowed, otherwise
			///   an error condition will be triggered if the Type is not a float and implicitConversion is false.
			///
			/// @param implicitConversion Set to true to allow other basic types (Bool, Integer, String, Nil) to convert
			///   to an integer value automatically.  If false, no conversion will be attempted and an error condition
			///   will be triggered.
			///
			float AsFloat(bool implicitConversion = kImplicitConversions) const;

			///
			/// @details Gets the value of the object as a bool using implicit conversions if allowed, otherwise
			///   an error condition will be triggered if the Type is not a bool and implicitConversion is false.
			///
			/// @param implicitConversion Set to true to allow other basic types (Integer, Float, String, Nil) to convert
			///   to an integer value automatically.  If false, no conversion will be attempted and an error condition
			///   will be triggered.
			///
			bool AsBoolean(bool implicitConversion = kImplicitConversions) const;

			///
			/// @details Gets the value of the object as a string using implicit conversions if allowed, otherwise
			///   an error condition will be triggered if the Type is not a string and implicitConversion is false.
			///
			/// @param implicitConversion Set to true to allow other basic types (Bool, Integer, Float, Nil) to convert
			///   to an integer value automatically.  If false, no conversion will be attempted and an error condition
			///   will be triggered.
			///
			tbCore::tbString AsString(bool implicitConversion = kImplicitConversions) const;

			//
			//  TODO: TIM: Planning: Do we want to support this?  If so implement and document.
			//	@note Cannot implicitly convert to a string, will assert if type is not a string.
			//
			//const tbCore::tbString& AsStringRef(void) const;


			///
			/// @details First checks to make sure the type of the object is an integer, then returns the value as an
			///   integer without any conversions*.  If the type is not an integer the defaultValue will be returned.
			///
			/// @param defaultValue The integer value to return if the type is not specifically an integer, even if the
			///   type could be implicitly converted.
			///
			///	@note Current implementation does not allow any implicit conversions, but in the future this may change
			///   and it may become the default behavior of the WithDefault functions.
			///
			int AsIntegerWithDefault(const int defaultValue) const;

			///
			/// @details First checks to make sure the type of the object is a float, then returns the value as a
			///   float without any conversions*.  If the type is not a float the defaultValue will be returned.
			///
			/// @param defaultValue The float value to return if the type is not specifically a float, even if the
			///   type could be implicitly converted.
			///
			///	@note Current implementation does not allow any implicit conversions, but in the future this may change
			///   and it may become the default behavior of the WithDefault functions.
			///
			float AsFloatWithDefault(const float defaultValue) const;

			///
			/// @details First checks to make sure the type of the object is a boolean, then returns the value as a
			///   boolean without any conversions*.  If the type is not a boolean the defaultValue will be returned.
			///
			/// @param defaultValue The boolean value to return if the type is not specifically a boolean, even if the
			///   type could be implicitly converted.
			///
			///	@note Current implementation does not allow any implicit conversions, but in the future this may change
			///   and it may become the default behavior of the WithDefault functions.
			///
			bool AsBooleanWithDefault(const bool defaultValue) const;

			///
			/// @details First checks to make sure the type of the object is a string, then returns the value as a
			///   string without any conversions*.  If the type is not a string the defaultValue will be returned.
			///
			/// @param defaultValue The string value to return if the type is not specifically a string, even if the
			///   type could be implicitly converted.
			///
			///	@note Current implementation does not allow any implicit conversions, but in the future this may change
			///   and it may become the default behavior of the WithDefault functions.
			///
			tbCore::tbString AsStringWithDefault(const tbCore::tbString& defaultValue) const;


			///
			/// @details Attempts to change the value of the object to the value specified, if the object Type is an
			///   integer, Nil or implicityTypeChange is allowed this will succeed, otherwise an error condition will
			///   be triggered.  The object type will become an integer type during this process.
			///
			/// @param integerValue The value for the object to hold as an integer type.
			/// @param implicitTypeChange Set to true the value and type will always be set properly, however if false,
			///   and the object is not already an integer or Nil type this will cause an error condition to trigger.
			///
			void SetValue(const int& integerValue, bool implicitTypeChange = kImplicitTypeChange);

			///
			/// @details Attempts to change the value of the object to the value specified, if the object Type is a
			///   float, Nil or implicityTypeChange is allowed this will succeed, otherwise an error condition will
			///   be triggered.  The object type will become a float type during this process.
			///
			/// @param floatValue The value for the object to hold as a float type.
			/// @param implicitTypeChange Set to true the value and type will always be set properly, however if false,
			///   and the object is not already a float or Nil type this will cause an error condition to trigger.
			///
			void SetValue(const float& floatValue, bool implicitTypeChange = kImplicitTypeChange);

			///
			/// @details Attempts to change the value of the object to the value specified, if the object Type is a
			///   boolean, Nil or implicityTypeChange is allowed this will succeed, otherwise an error condition will
			///   be triggered.  The object type will become a float type during this process.
			///
			/// @param booleanValue The value for the object to hold as a boolean type.
			/// @param implicitTypeChange Set to true the value and type will always be set properly, however if false,
			///   and the object is not already a boolean or Nil type this will cause an error condition to trigger.
			///
			void SetValue(const bool& booleanValue, bool implicitTypeChange = kImplicitTypeChange);

			///
			/// @details Attempts to change the value of the object to the value specified, if the object Type is a
			///   string, Nil or implicityTypeChange is allowed this will succeed, otherwise an error condition will
			///   be triggered.  The object type will become a string type during this process.
			///
			/// @param stringValue The value for the object to hold as a string type.
			/// @param implicitTypeChange Set to true the value and type will always be set properly, however if false,
			///   and the object is not already a string or Nil type this will cause an error condition to trigger.
			///
			void SetValue(const tbCore::tbString& stringValue, bool implicitTypeChange = kImplicitTypeChange);


			//Used ONLY for Array Values.  Type must be kArrayType or kNilType to PushValue

			///
			/// @details Attempts to push the value onto the back of a standard vector if this object is either a Nil
			///   type or an Array type.  If the type is not Nil or an Array an error condition will be triggered.  Also
			///   if kTypeSafeArrays is set to true, the type of the value object must match the first element in the
			///   array or an error condition will be triggered, if this would be the first element no error will trigger.
			///
			/// @param value The value to push into the array of values, it can be of any type but if kTypeSafeArrays is
			///   true and the type doesn't match the first element an error condition will be triggered.
			///
			/// @note This can only be called on DynamicStructures with the Nil type or Array type, any other type will
			///   cause an error condition to be triggered.
			///
			DynamicStructure& PushValue(const DynamicStructure& value);

			///
			/// @details Returns a const reference to the value in the array at the specified index.  If this object is
			///   not an Array type an error condition will be triggered.
			///
			/// @param arrayIndex is the index of the desired element.  Must be within the range: 0 <= arrayIndex <= Size()
			///   or an error condition will be triggered for going out of range.
			///
			/// @note This can only be called on DynamicStructures with the Array type, any other type, including Nil
			///   will cause an error condition to be triggered.
			///
			const DynamicStructure& GetValue(const size_t& arrayIndex) const;

			///
			/// @details Returns a reference to the value in the array at the specified index.  If this object is
			///   not an Array type an error condition will be triggered.
			///
			/// @param arrayIndex is the index of the desired element.  Must be within the range: 0 <= arrayIndex <= Size()
			///   or an error condition will be triggered for going out of range.
			///
			/// @note This can only be called on DynamicStructures with the Array type, any other type, including Nil
			///   will cause an error condition to be triggered.
			///
			DynamicStructure& GetValue(const size_t& arrayIndex);

			///
			/// @details Returns a const reference to the value in the array at the specified index.  If this object is
			///   not an Array type an error condition will be triggered.
			///
			/// @param arrayIndex is the index of the desired element.  Must be within the range: 0 <= arrayIndex <= Size()
			///   or an error condition will be triggered for going out of range.
			///
			/// @note This can only be called on DynamicStructures with the Array type, any other type, including Nil
			///   will cause an error condition to be triggered.
			///
			const DynamicStructure& operator[](const size_t& arrayIndex) const;

			///
			/// @details Returns a reference to the value in the array at the specified index.  If this object is
			///   not an Array type an error condition will be triggered.
			///
			/// @param arrayIndex is the index of the desired element.  Must be within the range: 0 <= arrayIndex <= Size()
			///   or an error condition will be triggered for going out of range.
			///
			/// @note This can only be called on DynamicStructures with the Array type, any other type, including Nil
			///   will cause an error condition to be triggered.
			///
			DynamicStructure& operator[](const size_t& arrayIndex);

			//
			// TODO: TIM: Planning: Do we need to support int too? Teach the user how to use this.
			//
			//const DynamicStructure& operator[](const int& arrayIndex) const;

			//
			// TODO: TIM: Planning: Do wee need to support int too? Teach the user how to use this.
			//
			//DynamicStructure& operator[](const int& arrayIndex);



			
			//Used ONLY for Structure Values.  Type must be kStructureType or kNilType to AddMember

			///
			/// @details Attempts to add a member with name and value to an object with type Structure or Nil.  Any
			///   other types will cause an error condition to be triggered.  If the type is Nil, the type will become
			///   Structure and the first member will be added.  If a value already exists for the member an error
			///   condition will be triggered.  A reference to the added value object will then be returned.
			///
			/// @param memberName The name of the member to be able to retrieve the value later.  If a value with the
			///   memberName has already been added to the Structure an error condition will be triggered.
			/// @param memberValue The value of the member which will be retrieved by it's name.
			///
			DynamicStructure& AddMember(const tbCore::tbString& memberName, const DynamicStructure& memberValue);

			///
			/// @details Sets the value of an existing structure member, or adds the member if it did not exists, to an object
			///   with type Structure or Nil.  Any other types will cause an error condition to be triggered.  If the type is
			///   Nil, the object/type will become a Structure and the member will be added.  If a value already exists for
			///   the member, it will be set to the new value.  A reference to the object will then be returned.
			///
			/// @param memberName The name of the member to be able to retrieve the value later.  If a value with the
			///   memberName has already been added to the Structure an error condition will be triggered.
			/// @param memberValue The value of the member which will be retrieved by it's name.
			///
			DynamicStructure& SetMember(const tbCore::tbString& memberName, const DynamicStructure& memberValue);

			///
			/// @details Attempts to retrieve a DynamicStructure object for the memberName given, if this object is of
			///   type Nil then a Nil object will be returned,* but if this object is a Structure the member will be
			///   searched for and returned if found otherwise if not found a Nil object will be returned.*
			///
			/// @param memberName The name of the member as it was added during AddMember.
			///
			/// @note The Nil object and unfound member issues may cause an error condition to trigger in the future.
			///
			const DynamicStructure& GetMember(const tbCore::tbString& memberName) const;

			///
			/// @copydoc GetMember()
			///
			DynamicStructure& GetMember(const tbCore::tbString& memberName);

			///
			/// @copydoc GetMember()
			///
			const DynamicStructure& operator[](const tbCore::tbString& memberName) const;

			///
			/// @copydoc GetMember()
			///
			DynamicStructure& operator[](const tbCore::tbString& memberName);

			///
			/// @copydoc GetMember()
			///
			const DynamicStructure& operator[](const char* const memberName) const;

			///
			/// @copydoc GetMember()
			///
			DynamicStructure& operator[](const char* const memberName);

			///
			/// @details Returns a structure iterator at the start of the container for a Structure value. To be used to iterate
			///   over each member in the structure, typically for saving/exporting the dynamic structure.
			///
			StructureContainer::const_iterator BeginStructure(void) const;

			///
			/// @details Returns a structure iterator representing the end of the container for a Structure value. To be used
			///   to iterate over each member in the structure.
			///
			StructureContainer::const_iterator EndStructure(void) const;

			///
			/// @details Returns a structure iterator at the start of the container for a Structure value. To be used to iterate
			///   over each member in the structure, typically for saving/exporting the dynamic structure.
			///
			StructureContainer::iterator BeginStructure(void);

			///
			/// @details Returns a structure iterator representing the end of the container for a Structure value. To be used
			///   to iterate over each member in the structure.
			///
			StructureContainer::iterator EndStructure(void);

			//TODO: TIM: Implementation: (Test) Add a way to iterate over all Members in a structure.

			//Used ONLY for Structure or Array Values.

			///
			/// @details Attempts to return details about the Size of the object.  If the Type is Nil a size of 0 will
			///   be returned so a loop could be used on an Array type without checking that the array object was valid.
			///   If the type is an Array the size returned will be the number of items in the array, if the type is a
			///   Structure the the size returned will be the number of objects in the structure.  Any other type will
			///   return kInvalidSize, which may be 0 or may change.
			///
			size_t Size(void) const;

			//Acceptable Inputs:
			//	"engine.pistons[0].isDamaged"
			//const MagicValue& FromPath(const tbCore::tbString& path) const;

			///
			/// @details This operator will cast a DynamicStructure object into an integer with implicit conversion allowed.
			///
			operator int() const { return AsInteger(true); }

			///
			/// @details This operator will cast a DynamicStructure object into a float with implicit conversion allowed.
			///
			operator float() const { return AsFloat(true); }

			///
			/// @details This operator will cast a DynamicStructure object into a bool with implicit conversion allowed.
			///
			operator bool() const { return AsBoolean(true); }

			///
			/// @details This operator will cast a DynamicStructure object into a bool with implicit conversion allowed.
			///
			operator const bool() const { return AsBoolean(true); }

			///
			/// @details This operator will cast a DynamicStructure object into a string with implicit conversion allowed.
			///
			operator tbCore::tbString() const { return AsString(true); }

			///
			/// @details Checks two DynamicStructures for equality.  This can only check the basic types (Nil, Integer,
			///   Float, Boolean and String) for equality with another of the same type.  Checking for equality with an
			///   Array or Structure type will result in an error condition being triggered.
			///
			/// @note Checking for equality of an Array ot Structure type will trigger an error condition.
			///
			bool operator==(const DynamicStructure& rightSide) const;

		private:
			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			void SetToNil(void);

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			int ConvertToInteger(void) const;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			float ConvertToFloat(void) const;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			bool ConvertToBoolean(void) const;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			tbCore::tbString ConvertToString(void) const;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			enum DynamicStructureValueType
			{
				kNilType,       ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.

				kIntegerType,   ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
				kFloatType,     ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
				kBooleanType,   ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
				kStringType,    ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.

				kArrayType,     ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
				kStructureType, ///< TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			};

//			///
//			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
//			///
//			typedef std::vector<DynamicStructure> ArrayContainer;
//
//			///
//			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
//			///
//			typedef std::map<tbCore::tbString, DynamicStructure> StructureContainer;

			///
			/// TODO: TIM: InternalDoc: This is a TurtleBrains implementation detail, and should be documented as such.
			///
			DynamicStructureValueType mValueType;

			//Unnamed anonymous union so that DynamicStrucure always starts
			union
			{
				char mRawBytes[8];	//Reserve 64-bits for whatever types follow.
				int mInteger;
				bool mBoolean;
				float mFloat;

				tbCore::tbString* mString;
				ArrayContainer* mArray;
				StructureContainer* mStructure;
			};

		public:
			static const DynamicStructure kNullValue;	///< @details The value returned to represent a const-reference to null.
			static const tbCore::tbString kNullAsString;		///< @details The value returned when a null structure is converted to a string.
			static const tbCore::tbString kTrueAsString;		///< @details The value returned when a true boolean is converted to a string.
			static const tbCore::tbString kFalseAsString;	///< @details The value returned when a false boolean is converted to a string.
			static const unsigned int kInvalidSize;		///< @details The value returned when Size() is called with an invalid type, likely to be 0.
			static const bool kTypeSafeArrays;			///< @details Forces an ArrayType to hold every value of the same type.
			static const bool kImplicitConversions;		///< @details Changes the behavior of the DynamicStructure's AsType functions.
			static const bool kImplicitTypeChange;		///< @details Allows the DynamicStructure to change type on the fly.
			static const float kFloatElipson;			///< @details Consider two floating point values equal if the difference is less than elipson.
		};

		///@cond simple

		/* Overloads to make the DynamicStructure behave like a built in Integer */
		inline bool operator==(const DynamicStructure& leftSide, const int& rightSide) { return (leftSide.AsInteger() == rightSide) ? true : false; }
		inline bool operator==(const int& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsInteger() == leftSide) ? true : false; }
		inline bool operator!=(const DynamicStructure& leftSide, const int& rightSide) { return (leftSide.AsInteger() != rightSide) ? true : false; }
		inline bool operator!=(const int& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsInteger() != leftSide) ? true : false; }

		inline bool operator==(const DynamicStructure& leftSide, const float& rightSide) { return (fabs(leftSide.AsFloat() - rightSide) <= DynamicStructure::kFloatElipson)  ? true : false; }
		inline bool operator==(const float& leftSide, const DynamicStructure& rightSide) { return (fabs(rightSide.AsFloat() - leftSide) <= DynamicStructure::kFloatElipson) ? true : false; }
		inline bool operator!=(const DynamicStructure& leftSide, const float& rightSide) { return (fabs(leftSide.AsFloat() - rightSide) > DynamicStructure::kFloatElipson) ? true : false; }
		inline bool operator!=(const float& leftSide, const DynamicStructure& rightSide) { return (fabs(rightSide.AsFloat() - leftSide) > DynamicStructure::kFloatElipson) ? true : false; }

		inline bool operator==(const DynamicStructure& leftSide, const bool& rightSide) { return (leftSide.AsBoolean() == rightSide) ? true : false; }
		inline bool operator==(const bool& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsBoolean() == leftSide) ? true : false; }
		inline bool operator!=(const DynamicStructure& leftSide, const bool& rightSide) { return (leftSide.AsBoolean() != rightSide) ? true : false; }
		inline bool operator!=(const bool& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsBoolean() != leftSide) ? true : false; }

		inline bool operator==(const DynamicStructure& leftSide, const tbCore::tbString& rightSide) { return (leftSide.AsString() == rightSide) ? true : false; }
		inline bool operator==(const tbCore::tbString& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsString() == leftSide) ? true : false; }
		inline bool operator!=(const DynamicStructure& leftSide, const tbCore::tbString& rightSide) { return (leftSide.AsString() != rightSide) ? true : false; }
		inline bool operator!=(const tbCore::tbString& leftSide, const DynamicStructure& rightSide) { return (rightSide.AsString() != leftSide) ? true : false; }

		///@endcond

	};	/* namespace Core */
};	/* namespace TurtleBrains */

namespace tbCore = TurtleBrains::Core;

#endif /* _TurtleBrains_DynamicStructure_h_ */
