///
/// @file
/// @details Provides some error definitions, functions and handlers.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Error_h_
#define _TurtleBrains_Error_h_

#include <string>

///
/// @defgroup Error Error Handling Macros
/// @{
///

///
///	@details Creates formatted message like using printf and outputs to stdout before proceeding to trigger an error
///   condition then invokes OnErrorFired() on each ErrorHandlerInterface that has been added to the handler list 
///   by calling: TurtleBrains::Core::Error::AddErrorHandler().
///
#define tb_error(message, ...) { tbCore::Error::TriggerError(__FILE__, __LINE__, message, ##__VA_ARGS__); }

///
/// @details If the errorTest condition passes, print out message using printf, then invoke OnErrorFired() on each
///   ErrorHandlerInterface that has been added via TurtleBrains::Core::Error::AddErrorHandler.
///
/// @param errorTest    An expression that will be evaluated to true/false.
/// @param message      A printf formatted const char* const null-terminated string with information about the failure.
/// @param ...          The parameters that should be used with the printf formatted string.
///
/// @see TurtleBrains::Core::Error::ErrorIf(),
/// @see printf documentation
/// @snippet tb_error_example.cpp exampleErrorIf
///
#define tb_error_if(errorTest, message, ...) { tbCore::Error::TriggerErrorIf((errorTest), __FILE__, __LINE__, message, ##__VA_ARGS__); }

///
///	@}
///

namespace TurtleBrains
{
	namespace Core
	{
		namespace Error
		{
			///
			/// @brief Customize what happens when an error is fired.
			///
			/// @details Whether an error condition should throw an exception, log the error, display a message box, or
			///   any other customized action, it can happen through a custom handler.  To use, make a class derived from
			///   the ErrorHandlerInterface and override OnErrorFired to handle the error as desired.
			///
			/// @snippet tb_error_example.cpp exampleErrorHandler
			///
			class ErrorHandlerInterface
			{
			public:
				///
				/// @details Constructs an ErrorHandler object.
				///
				ErrorHandlerInterface(void);

				///
				/// @details Currently does nothing, destructs the ErrorHandler object.
				///
				virtual ~ErrorHandlerInterface(void);

				///
				/// @details This will be called when an error happens using the tb_error_if or tbErrorIf macros, 
				///   which use the TurtleBrains::Core::Error::ErrorIf functions for each ErrorHandler added with 
				///   TurtleBrains::Core::Error::AddErrorHandler
				///
				/// @param errorMessage A message about what happened to fire off the error being handled.
				///
				virtual void OnErrorFired(const std::string& errorMessage) = 0;

			protected:
				///
				/// @details Returns the line number of the file where the error was fired from.
				///
				/// @note May be incorrect or unknown when using the ErrorIf() function directly, prefer use of the tb_error_if macro.
				///
				int GetLineNumber(void) const;

				///
				/// @details Returns the filename of the file where the error was fired from.
				///
				/// @note May be incorrect or unknown when using the ErrorIf() function directly, prefer use of the tb_error_if macro.
				///
				const std::string& GetFileName(void) const;
			};

			///
			///	@details Triggers an error condition so OnErrorFired() will be called and all error handlers will be notified.
			///
			void TriggerError(const char* const message, ...);

			///
			///	@details Triggers an error condition so OnErrorFired() will be called and all error handlers will be notified.
			///
			void TriggerError(const char* const fromFile, int onLineNumber, const char* const message, ...);

			///
			/// @details If testResult is true, OnErrorFired() will be called and all error handlers will be notified.
			///
			/// @note Prefer use of the macro tb_error_if instead of directly calling the function so the file and
			/// line number information is known.
			///
			void TriggerErrorIf(bool testResult, const char* const message, ...);

			///
			/// @details If testResult is true, OnErrorFired() will be called and all error handlers will be notified.
			///
			/// @note Prefer use of the macro tb_error_if instead of directly calling the function so the file and
			/// line number information is correct.
			///
			void TriggerErrorIf(bool testResult, const char* const fromFile, int onLineNumber, const char* const message, ...);

			///
			/// @details Each ErrorHandlerInterface will have a chance to handle a fired error, in the order they were
			///   added unless the path of the code is modified by termination, exceptions or other means within a handler.
			///
			/// @param errorHandler     The errorHandler to be added, must be non-null.
			///
			/// @note This may become an implementation detail that is not available to the programmer.  TODO: TIM: Investigate
			/// TODO: TIM: Planning: Check if the ctor should register and unregister themselves in dtor.
			///
			/// @snippet tb_error_example.cpp exampleErrorHandler
			///
			void AddErrorHandler(ErrorHandlerInterface *errorHandler);

			///
			/// @details Removes a handler from the list of handlers.
			///
			/// @param errorHandler     The errorHandler to be added, must be non-null.
			///
			/// @note This may become an implementation detail that is not available to the programmer.  TODO: TIM: Investigate
			/// TODO: TIM: Planning: Check if the ctor should register and unregister themselves in dtor.
			///
			void RemoveErrorHandler(ErrorHandlerInterface *errorHandler);

			///
			/// @details Instead of firing each ErrorHandlerInterface when an error is fired, this will throw an exception
			///   which can be caught and handled by the application developer.  By default, exception throwing is disabled.
			///
			/// @see EnableThrowOnError, ErrorHandlerInterface, AddErrorHandler
			///
			/// @snippet tb_error_example.cpp exampleErrorExceptions
			///
			void EnableThrowOnError(void);

			///
			/// @details Disables the exception throwing upon a triggered error condition, which can be enabled or disabled
			///   at runtime by calling this or EnableThrowOnError().  When disabled, the triggered error condition will be
			///   handled through the stack of errorHandlers added via AddErrorHandler().
			///
			/// @see EnableThrowOnError, ErrorHandlerInterface, AddErrorHandler
			///
			void DisableThrowOnError(void);

		}; /* namespace Error */
	}; /* namespace Core */
}; /* namespace TurtleBrains */

namespace tbCore = TurtleBrains::Core;

#endif /* _TurtleBrains_Error_h_ */
