///
/// @file
/// @details Provides an interface to take a string of json text and convert it into a DynamicStructure.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_json_parser.h"
#include "implementation/json.h"

#include <fstream>
#include <cstring>

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure JsonValueToDynamicStructure(const json::Value& value)
{
	tbCore::DynamicStructure resultingStructure;

	switch (value.GetType())
	{
	case json::NULLVal: /* resultingStructure is already null */ break;
	case json::BoolVal: resultingStructure.SetValue(value.ToBool()); break;
	case json::IntVal: resultingStructure.SetValue(value.ToInt()); break;
	case json::FloatVal: resultingStructure.SetValue(value.ToFloat()); break;
	case json::DoubleVal: resultingStructure.SetValue(static_cast<float>(value.ToDouble())); break;
	case json::StringVal: resultingStructure.SetValue(value.ToString()); break;

	case json::ObjectVal: {
			const json::Object object(value.ToObject());
			for (json::Object::ValueMap::const_iterator itr = object.begin(), itrEnd = object.end(); itr != itrEnd; ++itr)
			{
				const tbCore::DynamicStructure& obj = JsonValueToDynamicStructure(itr->second);
				resultingStructure.AddMember(itr->first, obj);
			}
			break; }

	case json::ArrayVal: {
			const json::Array array(value.ToArray());
			for (json::Array::ValueVector::const_iterator itr = array.begin(), itrEnd = array.end(); itr != itrEnd; ++itr)
			{
				resultingStructure.PushValue(JsonValueToDynamicStructure(*itr));
			}

			break; }
	};

	return resultingStructure;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure tbCore::ParseJson(const tbCore::tbString& jsonString)
{
	const json::Value& value(json::Deserialize(tbCore::ToStdString(jsonString)));
	return JsonValueToDynamicStructure(value);
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure tbCore::LoadJsonFile(const tbCore::tbString& filepath)
{
	std::ifstream inputFile(tbCore::ToStdString(filepath).c_str());
	if (false == inputFile.is_open())
	{
		tb_error("Was unable to load json file: %s\n", filepath.c_str());
		return tbCore::DynamicStructure::kNullValue;
	}

	inputFile.seekg(0, std::ios_base::end);
	size_t fileLength = static_cast<size_t>(inputFile.tellg());
	inputFile.seekg(0, std::ios_base::beg);

	char *fileData = new char[fileLength + 1];
	if (nullptr == fileData)
	{
		tb_error("Was unable to allocate memory to load json file: %s", filepath.c_str());
		return tbCore::DynamicStructure::kNullValue;
	}

	memset(fileData, 0, fileLength);
	inputFile.read(fileData, fileLength);
	inputFile.close();
	fileData[fileLength] = '\0';

	tbString jsonData(tb_string(fileData));
	delete [] fileData;

	return tbCore::ParseJson(jsonData);
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbCore::GetPathFromFilepath(const tbCore::tbString& filepath)
{
//	tb_error_if(tbCore::tbString::npos != tileSetImage.find("/") || tbCore::tbString::npos != tileSetImage.find("\\"), "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" to contain image without \\ or / in name.", tiledFilepath.c_str(), tileSetName.c_str());

	tbCore::tbString::size_type lastForwardSlash(filepath.find_last_of("/"));
	tbCore::tbString::size_type lastBackSlash(filepath.find_last_of("\\"));
	tbCore::tbString::size_type finalSlash = 0;
	if (tbCore::tbString::npos == lastForwardSlash)
	{
		finalSlash = lastBackSlash + 1;
	}
	else if (tbCore::tbString::npos == lastBackSlash)
	{
		finalSlash = lastForwardSlash + 1;
	}
	else //Both are valid positions, so take the maximum.
	{
		finalSlash = (lastForwardSlash > lastBackSlash) ? lastForwardSlash + 1 : lastBackSlash + 1;
	}

//	tbCore::tbString tileSetTexture = tiledFilepath.substr(0, finalSlash) + tileSetImage;
	return filepath.substr(0, finalSlash);
}

//--------------------------------------------------------------------------------------------------------------------//

bool DoesFileExist(const tbCore::tbString& filepath)
{
	std::ifstream inputFile(filepath.c_str(), std::ios_base::in);
	if (inputFile.is_open())
	{
		inputFile.close();
		return true;
	}
	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbCore::GetChildFilepath(const tbCore::tbString& parentFilepath, const tbCore::tbString& childFilename)
{
	if (true == DoesFileExist(childFilename))
	{
		return childFilename;
	}

	tbString parentDirectory = GetPathFromFilepath(parentFilepath);
	tbString parentChildPath = parentDirectory + childFilename;

	if (true == DoesFileExist(parentChildPath))
	{
		return parentChildPath;
	}

	return tbCore::tbString();
}

//--------------------------------------------------------------------------------------------------------------------//
