///
/// @file 
/// @details Check the version number of the TurtleBrains framework at runtime, which can be useful with dynamic
///   libraries and checking that the version is as expected.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Version_h_
#define _TurtleBrains_Version_h_

#include "tb_types.h"

namespace TurtleBrains
{
	namespace Core
	{
		namespace Version
		{

			///
			/// @details Retrieves the major part of the TurtleBrains framework version number.
			///
			/// @return If the version of the framework is: 1.3.2, the value returned will be 1
			///
			int GetMajor(void);

			///
			/// @details Retrieves the minor part of the TurtleBrains framework version number.
			///
			/// @return If the version of the framework is: 1.3.2, the value returned will be 3
			///
			int GetMinor(void);

			///
			/// @details Retrieves the revision part of the TurtleBrains framework version number.
			///
			/// @return If the version of the framework is: 1.3.2, the value returned will be 2
			///
			int GetRevision(void);

			///
			/// @details Retrieves a string with the major, minor and revision of the framework comined with the framework
			///   name in a string like: "TurtleBrains v1.0.0"
			///
			/// @return If the version of the framework is: 1.3.2, the value returned will be "TurtleBrains v1.3.2"
			///
			const tbString& GetVersionString(void);

			///
			/// @details Compares the version with the inputs to be sure that the framework version matches.
			///
			/// @param major       The value of the major portion to be compared with the current major of the framework.
			/// @param minor       The value of the minor portion to be compared with the current minor of the framework.
			/// @param revision    The value of the revision to be compared with the current revision of the framework.
			///
			/// @return True will be returned if the version of the framework exactly matches each component supplied
			///   otherwise false will be returned if the framework version is newer or older than desired.
			///
			bool IsExactly(const int major, const int minor, const int revision);

			///
			/// @details Compares the version with the inputs to be sure that the framework is equal or newer, in the event
			///   new features are added to the framework, you can use this to make sure those features are available.
			///
			/// @param major       The value of the major portion to be compared with the current major of the framework.
			/// @param minor       The value of the minor portion to be compared with the current minor of the framework.
			/// @param revision    The value of the revision to be compared with the current revision of the framework.
			///
			/// @return True will be returned if the version of the framework is equal or newer than the version created by
			///   each component supplied.  1.3.2 is newer than 1.2.4  2.1.0 is newer than 1.9.9.  False will be returned if
			///   the framework is older than the version created by each supplied compenent.
			///
			bool IsAtLeast(const int major, const int minor, const int revision);

		}; /* namespace Version */
	}; /* namespace Core */
}; /* namespace TurtleBrains */

namespace tbCore = TurtleBrains::Core;

#endif /* _TurtleBrains_Version_h_ */
