///
/// @file
/// @details Provides an exception handler that will create a small error report with a stack trace if available.  The
///   header is private implementation details of TurtleBrains.
///
/// Credits: Implementation makes extreme use of code written by 8monkeyLabs, edited to fit the needs of TurtleBrains.
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef	_TurtleBrains_DebugExceptionHandler_h_
#define _TurtleBrains_DebugExceptionHandler_h_

#if defined(tb_visual_cpp)

#define tb_debug_handle_exceptions

//#ifdef tb_debug_handle_exceptions
  #ifndef WIN32
	#error "WIN32 not defined"
  #else

#include "tbi_stack_tracer.h"
#include <time.h>
#include <string>
#include <fstream>
#include <iostream>

namespace tbImplementation
{

	BOOL HandleException(LPEXCEPTION_POINTERS params)
	{
		PEXCEPTION_RECORD nfo = params->ExceptionRecord;
		std::string ename;

		switch( nfo->ExceptionCode )
		{
			#define catchE(enm)	case (enm): { ename = #enm; break; }

			//common exceptions
			catchE( EXCEPTION_ACCESS_VIOLATION )
			catchE( EXCEPTION_STACK_OVERFLOW )

			//debugging exceptions
			catchE( EXCEPTION_BREAKPOINT )
			catchE( EXCEPTION_SINGLE_STEP )

			//floating point exceptions (not usually generated)
			catchE( EXCEPTION_FLT_DENORMAL_OPERAND )
			catchE( EXCEPTION_FLT_DIVIDE_BY_ZERO )
			catchE( EXCEPTION_FLT_INEXACT_RESULT )
			catchE( EXCEPTION_FLT_INVALID_OPERATION )
			catchE( EXCEPTION_FLT_OVERFLOW )
			catchE( EXCEPTION_FLT_STACK_CHECK )
			catchE( EXCEPTION_FLT_UNDERFLOW )

			//integer exceptions (not usually generated)
			catchE( EXCEPTION_INT_DIVIDE_BY_ZERO )
			catchE( EXCEPTION_INT_OVERFLOW )

			//misc. memory errors
			catchE( EXCEPTION_ARRAY_BOUNDS_EXCEEDED )
			catchE( EXCEPTION_DATATYPE_MISALIGNMENT )
			catchE( EXCEPTION_IN_PAGE_ERROR )
			catchE( EXCEPTION_GUARD_PAGE )

			//bad instructions?
			catchE( EXCEPTION_ILLEGAL_INSTRUCTION )
			catchE( EXCEPTION_PRIV_INSTRUCTION )

			//who friggin knows
			catchE( EXCEPTION_INVALID_DISPOSITION )			// >:|
			catchE( EXCEPTION_NONCONTINUABLE_EXCEPTION )
			catchE( EXCEPTION_INVALID_HANDLE )

			default:
			{
				ename = "UNKNOWN";
				break;
			}
		}

		#ifndef _XBOX
			std::cout << "Exception of type " << ename << " occured! All is lost." << std::endl;
		#endif

		//find the process name
		char procname[1024]; GetModuleFileNameA( NULL, procname, 1024 );
		std::string shortprocname( procname );
		size_t o = shortprocname.find_last_of('.');
		if( o != std::string::npos ) { shortprocname.erase( o, std::string::npos ); }
		o = shortprocname.find_last_of( '\\' );
		if( o != std::string::npos ) { shortprocname.erase( 0, o+1 ); }

		//open the crash log for writing
		std::ofstream	crashlog;
		std::string		crashfilepath( shortprocname + "_crash.log" );
		crashlog.open( crashfilepath.c_str() );

		//write some initial information into the log
		crashlog << "---------\n";
		crashlog << "Crash Log\n";
		crashlog << "---------\n\n";
		char date[10]; _strdate_s(date,10); char time[10]; _strtime_s(time,10);
		crashlog << "On: " << date << ' ' << time << '\n';
		crashlog << "By: " << procname << "\n";
		crashlog << "Type: " << ename << "\n\n";

		//get a stack trace
		DebugToolSetPrivate::StackTracer	tracer;
		CONTEXT mycontext = *(params->ContextRecord);
		tracer.trace( mycontext );
		tracer.translate();

		//print the stack trace
		crashlog << "Stack Trace:\n";
		crashlog << "-----------------------\n";
		for( unsigned i=0; i<tracer.result.size(); ++i )
		{
			DebugToolSetPrivate::StackTracer::stack_entry& entry = tracer.result[i];

			//name
			crashlog << entry.function_name << "()";
			int desiredtabs = 7; const int tabwidth = 8;
			desiredtabs -= int(entry.function_name.size()+2)/tabwidth;
			while( desiredtabs > 0 ) { crashlog << '\t'; --desiredtabs; }

			//file and line number
			std::string shortfilename( entry.file_name );
			size_t lastslash = shortfilename.find_last_of('\\');
			if( lastslash != std::string::npos )
			{ shortfilename.erase( 0, lastslash+1 ); }
			crashlog << " At line " << entry.line_number << " in " << shortfilename;

			//module name
			if( entry.module_name.size() )
			{ crashlog << " [" << entry.module_name << "]"; }
			crashlog << "\n";
		}
		crashlog << "-----------------------\n";

		crashlog.close();

		#ifndef _XBOX
			std::cout << "Crash log written to " << crashfilepath << " - please read it!" << std::endl;
		#endif

		return EXCEPTION_EXECUTE_HANDLER;
	}

}; /* namespace tbImplementation */

  #endif /* Defined Win32 */
//#endif /* tb_debug_handle_exceptions */

#endif /* tb_visual_cpp */
#endif	/* _TurtleBrains_DebugExceptionHandler_h_ */
