///
/// @file
/// @details Provides an exception handler that will create a small error report with a stack trace if available.  The
///   header is private implementation details of TurtleBrains.
///
/// Credits: Implementation makes extreme use of code written by 8monkeyLabs, editted to fit the needs of TurtleBrains.
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

/*
	NOTES:

		- Will only build under VC 7 or greater
		- Not 64 bit compatible
		- Will require linking to/loading of dbghelp.lib/dbghelp.dll (comes with windows)
		- Cannot perform a stack trace without the .pdb file to accompany the executable
*/

#ifndef _DebugToolSetPrivate_StackTracer_h_
#define _DebugToolSetPrivate_StackTracer_h_

#ifdef tb_visual_cpp

#ifndef _WIN32_WINNT
  #define _WIN32_WINNT	0x0400
#endif
#include <windows.h>
#include <dbghelp.h>
#pragma comment(lib, "dbghelp.lib")
#include <map>
#include <vector>
#include <fstream>

//TODO: Perhaps define these elsewhere?
//#define _USE_DEBUGTOOLSET_STACKTRACER_
#define DTS_NOINLINE __declspec(noinline)

namespace DebugToolSetPrivate
{

	class StackTracer
	{
	public:
		struct stack_entry_raw
		{
			DWORD64		offset;

			inline bool	operator<( const stack_entry_raw& other ) const
			{ return offset < other.offset; }
		};

		struct stack_entry
		{
			std::string		function_name;
			std::string		file_name;
			std::string		module_name;
			int				line_number;

			stack_entry()
			{ line_number = -1; }
		};

		std::vector<stack_entry>		result;
		std::vector<stack_entry_raw>	raw_result;

		typedef	std::map<stack_entry_raw,stack_entry>	stack_map;

		void clear( void )
		{
			result.clear();
			raw_result.clear();
		}

		void clearTranslations( void )
		{
			result.clear();
		}


		DTS_NOINLINE void trace( int ignoreframes = 0 )
		{
		#ifdef _USE_DEBUGTOOLSET_STACKTRACER_
			CONTEXT mycontext;
			memset( &mycontext, 0, sizeof(CONTEXT) );
			RtlCaptureContext( &mycontext );
			mycontext.ContextFlags = CONTEXT_FULL;

			trace( mycontext, ignoreframes );
		#endif	//_USE_DEBUGTOOLSET_STACKTRACER_
		}

		DTS_NOINLINE void trace( CONTEXT& mycontext, int ignoreframes = 0 )
		{
		#ifdef _USE_DEBUGTOOLSET_STACKTRACER_
			clear();

			//get handles to the process and the thread
			HANDLE myproc = GetCurrentProcess();
			HANDLE mythread = GetCurrentThread();

			loadSymbolsForTranslation( myproc );

			//initialize the stack frame structure with our context info
			STACKFRAME64 mystackframe;
			memset(&mystackframe, 0, sizeof(mystackframe));
			mystackframe.AddrPC.Offset = mycontext.Eip;
			mystackframe.AddrPC.Mode = AddrModeFlat;
			mystackframe.AddrFrame.Offset = mycontext.Ebp;
			mystackframe.AddrFrame.Mode = AddrModeFlat;
			mystackframe.AddrStack.Offset = mycontext.Esp;
			mystackframe.AddrStack.Mode = AddrModeFlat;

			//loop to walk the stack
			unsigned stack_depth = 0;
			unsigned max_stack_depth = 512;
			while ( StackWalk64(IMAGE_FILE_MACHINE_I386, myproc, mythread, &mystackframe, &mycontext,
								NULL, SymFunctionTableAccess64, SymGetModuleBase64, NULL) )
			{
				++stack_depth;
				if( stack_depth >= max_stack_depth ) { break; }

				//skip the specified number of stack frames
				if( ignoreframes > 0 ) { --ignoreframes; continue; }

				//store the address
				raw_result.push_back( stack_entry_raw() );
				raw_result.back().offset = mystackframe.AddrPC.Offset;
			}
		#endif	//_USE_DEBUGTOOLSET_STACKTRACER_
		}

		static void	loadSymbolsForTranslation( HANDLE current_proc )
		{
		#ifdef _USE_DEBUGTOOLSET_STACKTRACER_
			static bool loaded = false;
			if( !loaded )
			{
				SetErrorMode( SEM_NOOPENFILEERRORBOX | SEM_FAILCRITICALERRORS ); //shut up, dont tell me to insert disks
				SymInitialize( current_proc, NULL, TRUE );
				loaded = true;
			}
		#endif	//_USE_DEBUGTOOLSET_STACKTRACER_
		}

		unsigned translate( stack_map* map_cheat = 0, bool fileinfo=true, bool modulename=true )
		{
		#ifdef _USE_DEBUGTOOLSET_STACKTRACER_
			result.clear();
			if (map_cheat == 0)
			{
				result.reserve(raw_result.size());
			}
			HANDLE myproc = GetCurrentProcess();
			loadSymbolsForTranslation(myproc);

			unsigned translations_performed = 0;
			const unsigned max_namelen = 2048;
			SYMBOL_INFO* sym = (SYMBOL_INFO*)malloc(sizeof(SYMBOL_INFO) + max_namelen);
			for (unsigned i = 0; i < raw_result.size(); ++i)
			{
				stack_entry* entry = 0;

				//see if we can use a map cheat to see if this translation exists already
				if (map_cheat)
				{
					stack_entry& e = (*map_cheat)[ raw_result[i] ];
					if( e.function_name.size() > 0 )
					{	//translation is already in the map
						continue;
					}
					else
					{	//make a new translation, and keep it in the map
						entry = &e;
					}
				}
				else
				{	//make a new translation
					result.push_back( stack_entry() );
					entry = &(result.back());
				}

				DWORD64	disp64 = 0;
				DWORD	disp = 0;
				memset( sym, 0, sizeof(SYMBOL_INFO) + max_namelen );
				sym->SizeOfStruct = sizeof(SYMBOL_INFO);
				sym->MaxNameLen = max_namelen - 1;
				translations_performed++;
				if( SymFromAddr( myproc, raw_result[i].offset, &disp64, sym ) != FALSE )
				{
					//function name
					entry->function_name = sym->Name;

					//file name & line number
					if( fileinfo )
					{
						IMAGEHLP_LINE64	theline;
						memset( &theline, 0, sizeof(theline) );
						theline.SizeOfStruct = sizeof(theline);
						if( SymGetLineFromAddr64( myproc, raw_result[i].offset, &disp, &theline ) )
						{
							entry->file_name = theline.FileName;
							entry->line_number = theline.LineNumber;
						}
					}

					//module name
					if( modulename )
					{
						IMAGEHLP_MODULE64	themodule;
						memset( &themodule, 0, sizeof(themodule) );
						themodule.SizeOfStruct = sizeof(themodule);
						if( SymGetModuleInfo64( myproc, raw_result[i].offset, &themodule ) )
						{
							entry->module_name = themodule.ModuleName;
						}
					}
				}
				else
				{
					//can't get more stack info, error!
					DWORD lasterror = GetLastError();
					break;
				}
			}
			free(sym);
			return	translations_performed;
		#else	//if !DEFINED(_USE_DEBUGTOOLSET_STACKTRACER_)
			return 0;
		#endif	//_USE_DEBUGTOOLSET_STACKTRACER_
		}

		#ifdef _M8_LOGS_H_
		void quickPrint( void ) const
		{
			ERROR_LOG( "<< Stack Trace >>" << ENDL );
			for( unsigned i=0; i<result.size(); ++i )
			{
				ERROR_LOG( result[i].function_name << " at line " << result[i].line_number << ENDL );
			}
			ERROR_LOG( "<< ----------- >>" << ENDL );
		}
		#endif
	};

};  /* DebugToolSetPrivate */

#endif /* tb_visual_cpp */
#endif /* _DebugToolSetPrivate_StackTracer_h_ */
