///
/// @file
/// @details Setup preprocessor macros based on user settings, Release/Debug build etc so that when the macro version
///   of the functions are used (tb_dts_project_entry_point) vs (TurtleBrains::DebugToolSet::ProjectEntryPointF) they
///   may get dismissed at compile time if tb_debug_tool_set_disable is defined or a release build without the define
///   tb_debug_tool_set_enable.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_DebugToolSet_h_
#define _TurtleBrains_DebugToolSet_h_

//	These defines control how the DebugToolSet works.  Define these BEFORE including this header file.
//
//	#define _DEBUG							- By default this is what enables/disables the DebugToolSet
//	#define tb_debug_tool_set_enable		- Can be used to force the DebugToolSet during Release or other builds.
//	#define tb_debug_tool_set_disable		- Is used to disabled the DebugToolSet, regardless of _any_ other defines, including enable.

#include "../core/tb_configuration.h"

#ifdef tb_enable_debug_set
  #define tb_use_debug_set
#endif /* tb_enable_debug_set */

#ifndef tb_use_debug_set
  #ifdef tb_debug_build
    #define tb_use_debug_set
  #endif /* tb_debug_build */
#endif /* tb_use_debug_set */

#ifdef tb_disable_debug_set
  #ifdef tb_use_debug_set
    #undef tb_use_debug_set
  #endif /* tb_use_debug_set */
#endif /* tb_disable_debug_set */

///
/// @defgroup Debug Debug Log Macros
/// @{
///

#ifdef tb_use_debug_set

	///
	///	@details Setups a special exception handler on Windows built with Visual CPP that will create a crash report 
	///   with a callstack when an exception is thrown and caught within.  For other platforms and compilers the
	///   exception will be caught but no crash report will be created as far as TurlteBrains v0.2.0.  Hopeful future support.
	///
	///   Underneath this simply calls the Debug::ProjectEntryPoint() function, however the define should be prefferred
	///   when you want to have the option of turning off all logging with the tb_disable_debug_set define in the
	///   tb_configuration.h file.
	///
	/// @param entryPoint A function pointer to a function that takes no parameters and returns no parameters, 
	///   void foo(void); that will be called after setting up an exception handler to create a crash report.
	///
	#define tb_debug_project_entry_point(entryPoint)		TurtleBrains::Debug::ProjectEntryPoint(entryPoint);

	///
	///	@details Setups a special exception handler on Windows built with Visual CPP that will create a crash report 
	///   with a callstack when an exception is thrown and caught within.  For other platforms and compilers the
	///   exception will be caught but no crash report will be created as far as TurlteBrains v0.2.0.  Hopeful future support.
	///
	///   Underneath this simply calls the Debug::ProjectEntryPoint() function, however the define should be prefferred
	///   when you want to have the option of turning off all logging with the tb_disable_debug_set define in the
	///   tb_configuration.h file.
	///
	/// @param entryPoint A function pointer to a function that takes an argumentCount and values parameters just like
	///   the would be passed to a main and returns an int: int foo(int c, char* v[]); that will be called after setting 
	///   up an exception handler to create a crash report.
	/// @param argumentCount The number of values in the argumentValues array, typically just pass the same argumentCount
	///   you would get from main(int c, char* v[]);
	/// @param argumentValues The actual values of each argument as null-terminated strings, typically just pass the
	///   same argumentValues you would get from main(int c, char* v[]);
	///
	#define tb_debug_project_entry_point_with(entryPoint, argumentCount, argumentValues) \
		TurtleBrains::Debug::ProjectEntryPoint(entryPoint, argumentCount, argumentValues);
#else	//NO tb_use_debug_set
  #define tb_debug_project_entry_point(entryPoint)		entryPoint##()
  #define tb_debug_project_entry_point_with(entryPoint, argumentCount, argumentValues) entryPoint(argumentCount, argumentValues)
#endif //tb_use_debug_set

///
/// @}
///

namespace TurtleBrains
{
	namespace Debug
	{

		///
		///	@details Setups a special exception handler on Windows built with Visual CPP that will create a crash report 
		///   with a callstack when an exception is thrown and caught within.  For other platforms and compilers the
		///   exception will be caught but no crash report will be created as far as TurlteBrains v0.2.0.  Hopeful future support.
		///
		///   Underneath this simply calls the Debug::ProjectEntryPoint() function, however the define should be prefferred
		///   when you want to have the option of turning off all logging with the tb_disable_debug_set define in the
		///   tb_configuration.h file.
		///
		/// @param entryPoint A function pointer to a function that takes no parameters and returns no parameters, 
		///   void foo(void); that will be called after setting up an exception handler to create a crash report.
		///
		/// @note Prefer use of the macro tb_debug_project_entry_point so that nothing will happen, compiled out, if 
		///   debugging is disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		void ProjectEntryPoint(void (*entryPoint)(void));

		///
		///	@details Setups a special exception handler on Windows built with Visual CPP that will create a crash report 
		///   with a callstack when an exception is thrown and caught within.  For other platforms and compilers the
		///   exception will be caught but no crash report will be created as far as TurlteBrains v0.2.0.  Hopeful future support.
		///
		/// @param entryPoint A function pointer to a function that takes an argumentCount and values parameters just like
		///   the would be passed to a main and returns an int: int foo(int c, char* v[]); that will be called after setting 
		///   up an exception handler to create a crash report.
		/// @param argumentCount The number of values in the argumentValues array, typically just pass the same argumentCount
		///   you would get from main(int c, char* v[]);
		/// @param argumentValues The actual values of each argument as null-terminated strings, typically just pass the
		///   same argumentValues you would get from main(int c, char* v[]);
		///
		/// @note Prefer use of the macro tb_debug_project_entry_point_with so that nothing will happen, compiled out, if 
		///   debugging is disabled by the tb_disable_debug_set define fount in the tb_configuration.h file.
		///
		int ProjectEntryPoint(int (*entryPoint)(const int argumentCount, const char* argumentValues[]), 
			const int argumentCount, const char* argumentValues[]);

	}; /* namespace Debug */
}; /* namespace TurtleBrains */

#endif /* _TurtleBrains_DebugToolSet_h_ */
