///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tb_entity.h"
#include "tb_entity_manager.h"
#include "tb_entity_behavior_interface.h"
#include "../core/tb_string.h"

const tbGame::EntityType tbGame::Entity::kInvalidType(tb_string(""));

//--------------------------------------------------------------------------------------------------------------------//

tbGame::Entity::Entity(const tbGame::EntityType& entityType) :
	mEntityManager(nullptr),
	mIsCollidable(false)
{
	AddEntityType(entityType);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::Entity::~Entity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityManager* tbGame::Entity::GetEntityManager(void) const
{
	return mEntityManager;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::Simulate(void)
{
	if (true == IsActive())
	{
		OnSimulate();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

//
// Entity Behavior System
//

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::PushBehavior(tbGame::EntityBehaviorInterface* entityBehavior)
{
	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnPaused();
	}

	mBehaviorStack.push(entityBehavior);
	entityBehavior->OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::PopBehavior(void)
{
	tb_error_if(true == mBehaviorStack.empty(), "tbExternalError: There is no behavior to pop off.");
	ReallyPopBehavior();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::PopBehaviorIf(tbGame::EntityBehaviorInterface* entityBehavior)
{
	tb_error_if(true == mBehaviorStack.empty(), "tbExternalError: There is no behavior to pop off.");
	if (mBehaviorStack.top() == entityBehavior)
	{
		ReallyPopBehavior();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::PopAllBehaviors(void)
{
	while (false == mBehaviorStack.empty())
	{
		ReallyPopBehavior();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGame::EntityBehaviorInterface* tbGame::Entity::GetActiveBehavior(void) const
{
	return (true == mBehaviorStack.empty()) ? nullptr : mBehaviorStack.top();
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::EntityBehaviorInterface* tbGame::Entity::GetActiveBehavior(void)
{
	return (true == mBehaviorStack.empty()) ? nullptr : mBehaviorStack.top();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::ReallyPopBehavior(void)
{
	mBehaviorStack.top()->OnRemoved();
	delete mBehaviorStack.top();
	mBehaviorStack.pop();

	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnResume();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

//
// Collision and Bounding Volume
//

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::IsCollidable(void) const
{
	return mIsCollidable;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::SetCollidable(const bool isCollidable)
{
	mIsCollidable = isCollidable;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::HasBoundingVolume(void) const
{
	return (true == mBoundingVolumes.empty()) ? false : true;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::RemoveBoundingVolumes(void)
{
	SetCollidable(false);
	mBoundingVolumes.clear();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::AddBoundingCircle(const float circleRadius, const tbMath::Vector2& centerOffset)
{
	mBoundingVolumes.push_back(BoundingVolumeInformation(tbMath::Unstable::BoundingVolume(circleRadius), centerOffset));
	SetCollidable(true);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::AddBoundingBox(const float boxWidth, const float boxHeight, const tbMath::Vector2& centerOffset)
{
	mBoundingVolumes.push_back(BoundingVolumeInformation(tbMath::Unstable::BoundingVolume(boxWidth, boxHeight), centerOffset));
	SetCollidable(true);	
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::CheckCollisionWith(const tbMath::Vector2& point) const
{
	for (BoundingVolumeContainer::const_iterator itr = mBoundingVolumes.begin(), itrEnd = mBoundingVolumes.end(); itr != itrEnd; ++itr)
	{
		if (true == itr->first.CheckCollisionWith(GetPosition() + itr->second, point))
		{
			return true;
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::CheckCollisionWith(const tbMath::Vector2& center, const float radius) const
{
	if (true == HasBoundingVolume())
	{
		for (BoundingVolumeContainer::const_iterator itr = mBoundingVolumes.begin(), itrEnd = mBoundingVolumes.end(); itr != itrEnd; ++itr)
		{
			if (true == itr->first.CheckCollisionWith(GetPosition() + itr->second, tbMath::Unstable::BoundingVolume(radius), center))
			{
				return true;
			}
		}
	}
	else
	{
		tbMath::Unstable::BoundingVolume boundingVolume(radius);
		if (true == boundingVolume.CheckCollisionWith(center, GetPosition()))
		{
			return true;
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::CheckCollisionWith(const tbMath::Vector2& center, const float width, const float height) const
{
	if (true == HasBoundingVolume())
	{
		for (BoundingVolumeContainer::const_iterator itr = mBoundingVolumes.begin(), itrEnd = mBoundingVolumes.end(); itr != itrEnd; ++itr)
		{
			if (true == itr->first.CheckCollisionWith(GetPosition() + itr->second, tbMath::Unstable::BoundingVolume(width, height), center))
			{
				return true;
			}
		}
	}
	else
	{
		tbMath::Unstable::BoundingVolume boundingVolume(width, height);
		if (true == boundingVolume.CheckCollisionWith(center, GetPosition()))
		{
			return true;
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::CheckCollisionWith(const Entity& otherEntity) const
{
	if (false == IsCollidable() || false == otherEntity.IsCollidable())
	{
		return false;
	}

	for (auto myItr = mBoundingVolumes.begin(), myItrEnd = mBoundingVolumes.end(); myItr != myItrEnd; ++myItr)
	{
		for (auto otherItr = otherEntity.mBoundingVolumes.begin(), otherItrEnd = otherEntity.mBoundingVolumes.end(); otherItr != otherItrEnd; ++otherItr)
		{
			if (true == myItr->first.CheckCollisionWith(GetPosition() + myItr->second, otherItr->first, otherEntity.GetPosition() + otherItr->second))
			{
				return true;
			}
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

//
// Entity Type and other Properties
//

//--------------------------------------------------------------------------------------------------------------------//

//This will be deprecated shortly.
const tbGame::EntityType& tbGame::Entity::GetEntityType(void) const
{
	return (mEntityTypes.empty()) ? kInvalidType : mEntityTypes.front();
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGame::EntityTypeContainer& tbGame::Entity::GetEntityTypes(void) const
{
	return mEntityTypes;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::AddEntityType(const tbGame::EntityType& entityType)
{
	if (nullptr != mEntityManager)
	{
		EntityTypeContainer oldTypes = mEntityTypes;
		mEntityTypes.push_back(entityType);
		mEntityManager->EntityTypeChanged(*this, oldTypes);
	}
	else
	{
		mEntityTypes.push_back(entityType);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::RemoveEntityType(const tbGame::EntityType& entityType)
{
	if (nullptr != mEntityManager)
	{
		EntityTypeContainer oldTypes = mEntityTypes;
		mEntityTypes.remove(entityType);
		mEntityManager->EntityTypeChanged(*this, oldTypes);
	}
	else
	{
		mEntityTypes.remove(entityType);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Entity::IsEntityOfType(const tbGame::EntityType& entityType) const
{
#ifdef tb_linux
    for (const tbGame::EntityType& type : mEntityTypes)
    {
        if (entityType == type)
        {
            return true;
        }
    }
    return false;
#else
	EntityTypeContainer::const_iterator typeIterator = std::find(mEntityTypes.begin(), mEntityTypes.end(), entityType);
	return (typeIterator == mEntityTypes.end()) ? false : true;
#endif /* tb_linux */
}

//--------------------------------------------------------------------------------------------------------------------//

//
// Common Game Loop
//

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnSimulate(void)
{
	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnSimulate();
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnUpdate(const float deltaTime)
{
	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnUpdate(deltaTime);
	}

	GraphicList::OnUpdate(deltaTime);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnRender(void) const
{
	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnRender();
	}

	GraphicList::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnCollideWith(Entity& other)
{
	if (false == mBehaviorStack.empty())
	{
		mBehaviorStack.top()->OnCollideWith(other);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnAdded(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::Entity::OnRemoved(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//
