///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Entity_h_
#define _TurtleBrains_Entity_h_

#include "../core/tb_string.h"
#include "../graphics/tb_graphic_list.h"
#include "../math/unstable/tbu_bounding_volume.h"

#include <vector>
#include <stack>
#include <list>
#include <utility> //std::pair

namespace TurtleBrains
{
	namespace Game
	{

		///
		/// @details Each entity contains one or more entity types that are used to identify it during collisions
		///   or other interactions with the other entities.
		///
		typedef tbCore::tbString EntityType;

		///
		/// @details A container that holds multiple entity types that can be iterated over or checked on.
		///
		typedef std::list<EntityType> EntityTypeContainer;

		class EntityManager;
		class EntityBehaviorInterface;

		///
		///	@details The Entity object is a GraphicList to give it the ability to display itself with Graphic objects
		///   and the ability for subclasses to overload OnUpdate.  Since entities will likely be performing game
		///   logic, Simulate and OnSimulate have been added to the interface to provide a fixed time step update that
		///   may be called 0 to N times per frame.
		///
		class Entity : public tbGraphics::GraphicList
		{
		public:
			///
			///	@details TODO: TIM: Documentation: Teach the user how to use this.
			/// TODO: TIM: This may get deprecated and replaced with "properties" / "flags", or, multiple types.
			///
			static const EntityType kInvalidType;

			///
			///	@details Constructs an empty entity object which contains no Graphics and position is defaults to 0,0 but has
			///   the specified type.
			///
			explicit Entity(const EntityType& entityType);

			///
			///	@details Destructs an entity which currently (TurtleBrains v0.2.0) does not have any resources to clean
			///   up after.
			///
			virtual ~Entity(void) = 0;

			///
			/// @details Returns the EntityManager the entity is managed by, which can be nullptr.
			///
			EntityManager* GetEntityManager(void) const;

			///
			///	@details First checks to see if the entity IsActive() and calls OnSimulate() if it is.  If the entity is
			///   not active nothing happens.
			///
			void Simulate(void);

			//
			// Entity Behavior System
			//

			///
			/// @details Push the behavior onto the Entity so it becomes the active behavior controlling the Entity and
			///   the current behavior gets paused until this is popped. The behavior should be allocated with new, and
			///   the Entity will call delete on it when finished.
			///
			/// @param entityBehavior The behavior to push and activate on the top of the entities behavior stack. Must
			///   be allocated with new, as the Entity will delete the behavior later.
			///
			void PushBehavior(EntityBehaviorInterface* entityBehavior);

			///
			/// @details Pops the behavior at the top of the stack, and deletes the resources for it. Resumes the next
			///   Behavior which is now the active one and on top of the stack.
			///
			void PopBehavior(void);

			///
			/// @details Checks if the address of the current behavior, at the top of the stack, matches the address
			///   supplied, and only if these match does the behavior get popped by calling PopBehavior().
			///
			/// @param entityBehavior The behavior to check against the top of the stack and pop if they match.
			///
			void PopBehaviorIf(EntityBehaviorInterface* entityBehavior);

			///
			/// @details Removes all the behaviors from the entities behavior stack.
			///
			void PopAllBehaviors(void);

			///
			/// @details TODO: TIM: Documentation: Teach the user how to use this.
			/// @note This was implemented for a quick check and may be deprecated from official release.
			///
			const EntityBehaviorInterface* GetActiveBehavior(void) const;

			///
			/// @details TODO: TIM: Documentation: Teach the user how to use this.
			/// @note This was implemented for a quick check and may be deprecated from official release.
			///
			EntityBehaviorInterface* GetActiveBehavior(void);

			//
			// Collision and Bounding Volume
			//

			///
			///	@details Returns true if the entity will perform collision automatically each frame, it must also contain a
			///   bounding volume for collision to occur, call AddBoundingCircle() or AddBoundingBox() to create the bounding
			///   volume for the collision.
			///
			/// @note Adding a bounding volume with AddBoundingCircle or AddBoundingBox will cause the entity to become
			///   collidable.
			///
			bool IsCollidable(void) const;

			///
			///	@details Sets a flag that when false causes the entity to skip automated collision checks within the entity
			///   manager and when true will check for collision and call OnCollideWith() when a collision event occurs.
			///   Defaults to false, and will lead to better performance if false when the object does not need to check
			///   collision with other objects.
			///
			/// @note Adding a bounding volume with AddBoundingCircle or AddBoundingBox will cause the entity to become
			///   collidable, so make sure the bounding volumes are set before attempting to turn of collision.
			///
			void SetCollidable(const bool isCollidable);

			///
			///	@details Returns true if the entity has a bounding volume.
			///
			bool HasBoundingVolume(void) const;

			///
			/// @details Removes all bounding volumes the entity may have and will make the entity no longer collidable.
			///
			/// @note Removing all bounding volumes will cause the entity to no longer be collidable.
			///
			void RemoveBoundingVolumes(void);

			///
			///	@details Adds a bounding Circle to the entity to perform collisions and enables the entity to be collidable.
			///   Multiple bounding volumes can be added to the entity, and if any bodies collide the collision check will
			///   return true.
			///
			/// @note The entity collision detector is very simple.  Scaling the entity will NOT scale the bounding volumes
			///   and rotating the entity will NOT rotate the centerOffset of the bounding volume.
			///
			/// @note Adding a bounding volume with AddBoundingCircle or AddBoundingBox will cause the entity to become
			///   collidable.
			///
			void AddBoundingCircle(const float circleRadius, const tbMath::Vector2& centerOffset = tbMath::Vector2::kZero);

			///
			///	@details Adds a bounding Box to the entity to perform collisions and enables the entity to be collidable.
			///   Multiple bounding volumes can be added to the entity, and if any bodies collide the collision check will
			///   return true.
			///
			/// @note The entity collision detector is very simple.  Scaling the entity will NOT scale the bounding volumes
			///   and rotating the entity will NOT rotate the centerOffset of the bounding volume.
			///
			/// @note Adding a bounding volume with AddBoundingCircle or AddBoundingBox will cause the entity to become
			///   collidable.
			///
			void AddBoundingBox(const float boxWidth, const float boxHeight, const tbMath::Vector2& centerOffset = tbMath::Vector2::kZero);

			///
			/// @details Checks to see if the Entity collides with the provided point in the same 2D space.
			///
			/// @param point The point to check if it is contained within one of the bounding volumes for the entity.
			///
			/// @note This will check collision with the point regardless of the IsCollidable state.
			///
			bool CheckCollisionWith(const tbMath::Vector2& point) const;

			///
			/// @details Checks to see if the entity's bounding volumes collide with the circle provided, if the entity does
			///   not have any bounding volumes, this will instead check if the position of the entity is inside the circle,
			///   returning true if it is, false otherwise.
			///
			/// @param center The center of the bounding circle to check for a collision with the entities bounding volumes.
			/// @param radius The radius of the bounding circle for the collision check, for most projects this will be in
			///   pixels but it is possible that units are different.
			///
			/// @note This will check collision with the circle regardless of the IsCollidable state.
			///
			bool CheckCollisionWith(const tbMath::Vector2& center, const float radius) const;

			///
			/// @details Checks to see if the entity's bounding volumes collide with the axis-aligned box provided, if the
			///   entity does not have any bounding volumes, this will instead check if the position of the entity is inside
			///   the box, returning true if it is, false otherwise.
			///
			/// @param center The center of the bounding box to check for a collision with the entities bounding volumes.
			/// @param width The horizontal size of the bounding box for the collision check, for most projects this will be in
			///   pixels but it is possible that units are different.
			/// @param height The vertical size of the bounding box for the collision check, for most projects this will be in
			///   pixels but it is possible that units are different.
			///
			/// @note This will check collision with the axis-aligned box regardless of the IsCollidable state.
			///
			bool CheckCollisionWith(const tbMath::Vector2& center, const float width, const float height) const;

			///
			/// @details Checks to see if the Entity collides with the other entity provided.
			///
			/// @param otherEntity The Entity to check for collisions with.
			///
			/// @note THis will check collision with the other entity regardless of this or the other entities IsCollidable
			///   state.  If either this or the otherEntity does not have a bounding volume no collision will occur.
			///
			bool CheckCollisionWith(const Entity& otherEntity) const;

			//
			// EntityType and other Properties
			//

			///
			/// @details Returns the EntityType of the Entity.
			///
			/// @note This was implemented when entities only had a single type. For backwards compatibility it returns the
			///   first entity type that gets added, but this will eventually be removed, so start using IsEntityOfType().
			///
			const EntityType& GetEntityType(void) const;

			///
			/// @details Returns a container of all the EntityTypes for the Entity.
			///
			const EntityTypeContainer& GetEntityTypes(void) const;

			///
			/// @details Adds the entity type to the entity, in this way you can have multiple types for an entity. If the
			///   entityType is kInvalidType an error condition will be triggered.
			///
			void AddEntityType(const EntityType& entityType);

			///
			/// @details Finds and removes an EntityType from the entity so it no longer counts as that type.
			///
			void RemoveEntityType(const EntityType& entityType);

			///
			/// @details Checks through the entity types that have been added to the Entity for a match, returning true if a
			///   match is found or false if no match was found.
			///
			bool IsEntityOfType(const EntityType& entityType) const;

		protected:
			///
			///	@details This should be overridden by a subclass if it needs to perform any game logic updates at a
			///   fixed time step for determinism. This function will be invoked when Simulate() is called and the
			///   entity IsActive(). This may be called from 0 to N times per frame.
			///
			/// @note OnSimulate() should not be called directly, even from a subclass. Use Simulate() which will first
			///   check if the Entity IsActive() before simulating with OnSimulate().
			///
			virtual void OnSimulate(void);

			virtual void OnUpdate(const float deltaTime);

			virtual void OnRender(void) const;

			///
			/// @details This should be overridded by a subclass if it needs to perform any game logic when a collision occurs
			///   with another entity type. This will be invoked on both entities, so if you had a bullet and a target the
			///   following calls: target.OnCollideWith(bullet) and bullet.OnCollideWith(target) would be made in no particular
			///   order.
			///
			virtual void OnCollideWith(Entity& other);

			///
			/// @details This is called just after the Entity is added to an EntityManager, <em>aka: the scene,</em> so the
			///   entity can perform any logic before starting to perform in the game loop.
			///
			virtual void OnAdded(void);

			///
			/// @details This is called just before the Entity is removed from an EntityManager, <em>aka the scene,</em> so
			///   that any logic or resource cleanup can be performed.
			///
			virtual void OnRemoved(void);

		private:
			void ReallyPopBehavior(void);

			typedef std::stack<EntityBehaviorInterface*> BehaviorContainer;
			typedef std::pair<tbMath::Unstable::BoundingVolume, tbMath::Vector2> BoundingVolumeInformation;
			typedef std::vector<BoundingVolumeInformation> BoundingVolumeContainer;

			BehaviorContainer mBehaviorStack;
			BoundingVolumeContainer mBoundingVolumes;
			EntityTypeContainer mEntityTypes;

			EntityManager* mEntityManager;
			bool mIsCollidable;

			friend class EntityManager;
		};

	}; /* namespace Game */
}; /* namespace TurtleBrains */

namespace tbGame = TurtleBrains::Game;

#endif /* _TurtleBrains_Entity_h_ */
