///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_GameTimer_h_
#define _TurtleBrains_GameTimer_h_

#include "../core/tb_types.h"

namespace TurtleBrains
{
	namespace Game
	{

		///
		///	@details The GameTimer provides a simple interface for creating a timer to manually increment and decrement
		///   each Simulate() step for game logic.  Internally milliseconds are stored and can be greater than or equal to
		///   zero allowing use cases to count the elapsed time since an event (by setting the timer to zero and incrementing
		///   each step), or to count down to some event.
		///
		class GameTimer
		{
		public:
			///
			/// @details The storage type of the milliseconds used internally in the GameTimer.
			///
			typedef tbCore::uint32 Milliseconds;

			///
			/// @details A constant GameTimer object representing a time value of zero.
			///
			static GameTimer kZero;

			///
			/// @details A constant GameTimer object representing the maximum value for the timer, infinity.
			///
			static GameTimer kInfinity;

			///
			///	@details Constructs a GameTimer object with an internal value of 0 milliseconds elapsed/remaining.
			///
			GameTimer(void);

			///
			///	@details Constructs a GameTimer object with the desired value in milliseconds.
			///
			/// @param valueMilliseconds The initial value of the timer object, in milliseconds, to track time from.
			///
			GameTimer(const Milliseconds& valueMilliseconds);

			///
			///	@details Copy constructor to create a GameTimer object from another object, copying all internal values.
			///
			GameTimer(const GameTimer& other);

			///
			///	@details Destructs a GameTimer object, which is minimal since the timer does not need to release any resources.
			///
			~GameTimer(void);

			///
			///	@details Set the value of the GameTimer to the time value of another timer object.
			///
			GameTimer& operator=(const GameTimer& other);

			///
			///	@details Decrement the value of the timer by the milliseconds in a single Simulate step and returns true if
			///   the internal value is now at 0, including if it was already at 0.
			///
			/// @note Will not allow the timer object to underflow, and will remain at 0 even if step size is greater than value.
			///
			bool DecrementStep(void);

			///
			///	@details Increments the value of the timer by the milliseconds in a single Simulate step so that GetElapsedTime()
			///   can be used to track how long the timer has been running for. This will also prevent the timer from overflowing
			///   should it ever get large enough, or approach a value greater than maximumValue supplied. Returns true if the
			///   increment reached or stopped at the maximumValue during the call, false otherwise.
			///
			/// @param maximumValue The highest value the timer can increment to, defaults to kInfinity, the maximum value possible.
			///
			/// @note It is possible that the timer overflows, in this version of the GameTimer the millisecond value is stored
			///   as a 32 bit unsigned integer value which will hold approximately 49 days.  This may be changed to 64bit in
			///   the future.
			///
			bool IncrementStep(const GameTimer& maximumValue = kInfinity);

			///
			///	@details Checks if the timer is at zero returning true if it is or false it if the value is greater than 0.
			///
			bool IsZero(void) const;

			///
			///	@details Returns the value of the timer in milliseconds, typically used with a timer object that is counting
			///   upwards using IncrementStep() for clarity that some amount of time has passed.
			///
			const Milliseconds& GetElapsedTime(void) const;

			///
			///	@details Returns the value of the timer in milliseconds, typically used with a timer object that is counting
			///   down towards zero using DecrementStep() for clarity that some amount of time may be remaining.
			///
			const Milliseconds& GetRemainingTime(void) const;

			///
			///	@details Computes a percentage value from 0.0f to 1.0f where the timer object is between 0 and the timeValue
			///   passed in.  If the internal milliseconds value is larger than that of the timeValue parameter, the returned
			///   percentage may be greater than 1.0f.
			///
			/// @param timeValue The range of milliseconds representing "100 percent" or a return value of 1.0f.
			///
			/// @note The returned result may be greater than 1.0f if the timers value is greater than the provided value.
			///
			float GetPercentageOf(const GameTimer& timeValue) const;

			///
			///	@details Converts a value in milliseconds to a value in partial seconds, being returned as a floating point.
			///
			/// @note The actual assigned value will be converted to seconds and in the process some precision may be lost.
			///
			static float ToSeconds(const Milliseconds& valueMilliseconds);

			///
			///	@details Converts a time value from partial seconds to milliseconds, being returned as an integer type.
			///
			/// @note The returned value will be the input parameter converted to milliseconds and in the process precision
			///   may be lost, the returned value may be slightly smaller than the input.  0.0018 seconds may become 1 millisecond.
			///
			static Milliseconds ToMilliseconds(const float& valueSeconds);

			///
			///	@details Changes the number of milliseconds per Simulate step in TurtleBrains.
			///
			/// @param millisecondsPerStep The number of milliseconds that pass for each simulate step.  10 milliseconds is
			///   the default per step and results in Simulate being called at roughly 100hz.
			///
			static void SetMillisecondsPerStep(const Milliseconds& millisecondsPerStep);

			///
			///	@details Returns the number of milliseconds per Simulate step in TurtleBrains as set by SetMillisecondsPerStep().
			///
			static Milliseconds GetMillisecondsPerStep(void);

			///
			///	@details Returns the partial seconds per Simulate step in TurtleBrains as set by SetMillisecondsPerStep().
			///
			/// @note The returned value will be converted from milliseconds and in the process precision may be lost, the
			///   returned value may be slightly smaller than the actual value.  0.0018 seconds may become 1 millisecond.
			///
			static float GetSecondsPerStep(void);

		private:
			Milliseconds mMillisecondTimer;
			static Milliseconds sMillisecondsPerStep;
		};

		inline bool operator==(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() == right.GetElapsedTime()) ? true : false;
		}

		inline bool operator!=(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() != right.GetElapsedTime()) ? true : false;
		}

		inline bool operator>=(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() >= right.GetElapsedTime()) ? true : false;
		}

		inline bool operator<=(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() <= right.GetElapsedTime()) ? true : false;
		}

		inline bool operator>(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() > right.GetElapsedTime()) ? true : false;
		}

		inline bool operator<(const TurtleBrains::Game::GameTimer& left, const TurtleBrains::Game::GameTimer& right)
		{
			return (left.GetElapsedTime() < right.GetElapsedTime()) ? true : false;
		}

	}; /* namespace Game */
}; /* namespace TurtleBrains */

namespace tbGame = TurtleBrains::Game;

#endif /* _TurtleBrains_GameTimer_h_ */
