///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved --> 
///------------------------------------------------------------------------------------------------------------------///

#include "tb_tile_system.h"
#include "implementation/tbi_tile_system_renderer.h"
#include "unstable/tbu_tile_system_collider.h"
#include "../math/unstable/tbu_bounding_volume.h"
#include "../core/tb_json_parser.h"
#include "../core/tb_defines.h"

#include "../graphics/implementation/tbi_renderer.h"

#include <vector>
#include <map>

namespace tbImplementation
{

	template<typename T> class TileSystemPropertySet : public tbCore::Noncopyable
	{
	public:
		TileSystemPropertySet(void);
		virtual ~TileSystemPropertySet(void);

		void Clear(void);
		void CollectAll(const tbGame::TileSystem& tileSystem);
		void CollectArea(const tbGame::TileSystem& tileSystem, const tbGame::TileLocation& startColumn, const tbGame::TileLocation& startRow,
			const tbGame::TileLocation& columnCount, const tbGame::TileLocation& rowCount);

	private:
		std::vector<T> mPropertyData;
	};

	class TileSystemCollider : public TileSystemPropertySet<bool>
	{
	public:
		TileSystemCollider(void);
		virtual ~TileSystemCollider(void);

		void Clear(void);
	};

}; /* namespace tbImplementation */


const tbGame::TileIndex tbGame::kInvalidTileIndex(0xFFFF);
const tbGame::TileLocation tbGame::kInvalidTileLocation(0xFFFF);
const tbGame::TileSetIndex tbGame::kInvalidTileSetIndex(0xFF);

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGame::TileSystem::TileSystem(void) :
	mCollider(new tbGame::Unstable::TileSystemCollider())
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::TileSystem::~TileSystem(void)
{
	ClearMap();
	delete mCollider;
	mCollider = nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::ClearMap(void)
{
	mTileSets.clear();
	mOrderedTileSetNames.clear();

	mTileLayers.clear();
	mOrderedLayerNames.clear();
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbGame::TileSystem::GetMapPropertyAsString(const tbCore::tbString& propertyName) const
{
	return mMapProperties.GetMember(propertyName).AsString();
}

//--------------------------------------------------------------------------------------------------------------------//

int tbGame::TileSystem::GetMapPropertyAsInteger(const tbCore::tbString& propertyName) const
{
	return mMapProperties.GetMember(propertyName).AsInteger();

}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::TileSystem::GetMapPropertyAsBoolean(const tbCore::tbString& propertyName) const
{
	return mMapProperties.GetMember(propertyName).AsBoolean();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetPropertiesForMap(const tbCore::DynamicStructure& mapProperties)
{
	mMapProperties = mapProperties;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::AddTileSet(const tbCore::tbString& tileSetName, const tbGraphics::SpriteMap& spriteMap,
	const tbCore::DynamicStructure& tileSetProperties)
{
	tb_error_if(mTileSets.find(tileSetName) != mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" has already been added to system.", tileSetName.c_str());

	TileSet& newTileSet = mTileSets.insert(TileSetTable::value_type(tileSetName, TileSet(spriteMap))).first->second;
	newTileSet.mTileSetProperties = tileSetProperties;

	mOrderedTileSetNames.push_back(tileSetName);
}

//--------------------------------------------------------------------------------------------------------------------//

//void tbGame::TileSystem::SetFlagsForTile(const tbCore::tbString& tilesetName, const TileIndex& tileIndex, const TileFlags& tileFlags)
//{
//	TileSetTable::iterator tileSetIterator(mTileSets.find(tilesetName));
//	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the system.", tilesetName.c_str());
////	tb_error_if(tileSetIterator->second.mTileProperties.find(tileIndex) != tileSetIterator->second.mTileProperties.end(),
////							"tbExternalError: The properties have already been set for Tile(%d) in TileSet \"%s\".", tileIndex, tilesetName.c_str());
//
//	tileSetIterator->second.mTileProperties[tileIndex].first = tileFlags;
//}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::AboutTileContainer tbGame::TileSystem::FindTilesWithProperty(const tbCore::tbString& tileProperty, bool onlyVisibleLayers) const
{
	AboutTileContainer returnContainer;

	for (TileSystem::TileLayerTable::const_iterator tileLayerItr = mTileLayers.begin(), tileLayerEnd = mTileLayers.end(); tileLayerItr != tileLayerEnd; ++tileLayerItr)
	{
		const TileLayer& tileLayer(tileLayerItr->second);

		if (true == onlyVisibleLayers && false == tileLayer.mIsVisible)
		{
			continue;
		}

		const bool isSingleTileSet(1 == tileLayer.mTileSetData.size());
		const TileSetIndex& singleTileSetIndex(tileLayer.mTileSetData[0]);
		//TODO: TIM: Was getting a crash on the following line and quickly added the ternary statement, is that correct?		
		//const TileSet& singleTileSet(mTileSets.find(mOrderedTileSetNames[singleTileSetIndex])->second);
		const TileSet& singleTileSet((kInvalidTileSetIndex == singleTileSetIndex) ? mTileSets.find(mOrderedTileSetNames[0])->second : mTileSets.find(mOrderedTileSetNames[singleTileSetIndex])->second);

		for (TileLocation row(0); row < tileLayer.mTileRows; ++row)
		{
			const size_t rowIndex(row * tileLayer.mTileColumns);

			for (TileLocation column(0); column < tileLayer.mTileColumns; ++column)
			{
				const TileIndex& tileIndex(tileLayer.mTileData[column + rowIndex]);
				const TileSetIndex& tilesetIndex((true == isSingleTileSet) ? singleTileSetIndex : tileLayer.mTileSetData[column + rowIndex]);
				if (tilesetIndex >= mOrderedTileSetNames.size())
				{
					continue;
				}

				const tbCore::tbString& tileSetName(mOrderedTileSetNames[(true == isSingleTileSet) ? singleTileSetIndex : tilesetIndex]);
				const TileSet& tileSet((true == isSingleTileSet) ? singleTileSet : mTileSets.find(tileSetName)->second);
				const tbCore::DynamicStructure& tileProperties(GetTileProperties(tileSetName, tileIndex));
				
				if (false == tileProperties.GetMember(tileProperty).IsNil())
				{
					AboutTile info;

					const float rowPosition(tileLayer.mPosition.y + (row * tileSet.mSpriteMap.GetFrameHeight()));
					const float columnPosition(tileLayer.mPosition.x + (column * tileSet.mSpriteMap.GetFrameWidth()));
					const float tileWidth(tileSet.mSpriteMap.GetFrameWidth());
					const float tileHeight(tileSet.mSpriteMap.GetFrameHeight());

					tbMath::Vector2 centerTilePosition(columnPosition + (tileWidth / 2.0f), rowPosition + (tileHeight / 2.0f));

					info.mColumn = column;
					info.mRow = row;
					info.mCenterX = centerTilePosition.x;
					info.mCenterY = centerTilePosition.y;
					info.mTop = rowPosition;
					info.mLeft = columnPosition;
					info.mWidth = tileWidth;
					info.mHeight = tileHeight;
					info.mTileIndex = tileIndex;
					info.mPropertyValue = tileProperties.GetMember(tileProperty);

					returnContainer.push_back(info);
				}
			}
		}
	}

	return returnContainer;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::AboutTileContainer tbGame::TileSystem::FindTilesWithPropertyEquals(const tbCore::tbString& tileProperty,
	bool propertyValue, bool onlyVisibleLayers) const
{
	tbGame::AboutTileContainer returnContainer;
	tbGame::AboutTileContainer results = FindTilesWithProperty(tileProperty, onlyVisibleLayers);
	returnContainer.reserve(results.size());

	for (AboutTile& info : results)
	{
		if (propertyValue == info.mPropertyValue.AsBoolean())
		{
			returnContainer.push_back(info);
		}
	}

	return returnContainer;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::AboutTileContainer tbGame::TileSystem::FindTilesWithPropertyEquals(const tbCore::tbString& tileProperty,
	int propertyValue, bool onlyVisibleLayers) const
{
	tbGame::AboutTileContainer returnContainer;
	tbGame::AboutTileContainer results = FindTilesWithProperty(tileProperty, onlyVisibleLayers);
	returnContainer.reserve(results.size());

	for (AboutTile& info : results)
	{
		if (propertyValue == info.mPropertyValue.AsInteger())
		{
			returnContainer.push_back(info);
		}
	}

	return returnContainer;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::AboutTileContainer tbGame::TileSystem::FindTilesWithPropertyEquals(const tbCore::tbString& tileProperty,
	const tbCore::tbString& propertyValue, bool onlyVisibleLayers) const
{
	tbGame::AboutTileContainer returnContainer;
	tbGame::AboutTileContainer results = FindTilesWithProperty(tileProperty, onlyVisibleLayers);
	returnContainer.reserve(results.size());

	for (AboutTile& info : results)
	{
		if (propertyValue == info.mPropertyValue.AsString())
		{
			returnContainer.push_back(info);
		}
	}

	return returnContainer;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetPropertiesForTile(const tbCore::tbString& tilesetName, const TileIndex& tileIndex, const tbCore::DynamicStructure& tileProperties)
{
	TileSetTable::iterator tileSetIterator(mTileSets.find(tilesetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the system.", tilesetName.c_str());
	tb_error_if(tileSetIterator->second.mTileProperties.find(tileIndex) != tileSetIterator->second.mTileProperties.end(),
		"tbExternalError: The properties have already been set for Tile(%d) in TileSet \"%s\".", tileIndex, tilesetName.c_str());

	tileSetIterator->second.mTileProperties[tileIndex] = tileProperties;
//	tileSetIterator->second.mTileProperties[tileIndex].second = tileProperties;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure& tbGame::TileSystem::GetTileProperties(const tbCore::tbString& tilesetName, const TileIndex& tileIndex) const
{
	TileSetTable::const_iterator tileSetIterator(mTileSets.find(tilesetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the TileSystem.", tilesetName.c_str());

	TileSet::TilePropertyTable::const_iterator tilePropertyIterator(tileSetIterator->second.mTileProperties.find(tileIndex));
//	tb_error_if(tilePropertyIterator == tileSetIterator->second.mTileProperties.end(), "tbExternalError: Tile(%d) in TileSet \"%s\" does not contain properties.", tileIndex, tilesetName.c_str());
	if (tilePropertyIterator == tileSetIterator->second.mTileProperties.end())
	{
		return tbCore::DynamicStructure::kNullValue;
	}

	return tilePropertyIterator->second;
//	return tilePropertyIterator->second.second;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbGame::TileSystem::GetTileProperties(const tbCore::tbString& tilesetName, const TileIndex& tileIndex)
{
	TileSetTable::iterator tileSetIterator(mTileSets.find(tilesetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the TileSystem.", tilesetName.c_str());

	TileSet::TilePropertyTable::iterator tilePropertyIterator(tileSetIterator->second.mTileProperties.find(tileIndex));
//	tb_error_if(tilePropertyIterator == tileSetIterator->second.mTileProperties.end(), "tbExternalError: Tile(%d) in TileSet \"%s\" does not contain properties.", tileIndex, tilesetName.c_str());
	if (tilePropertyIterator == tileSetIterator->second.mTileProperties.end())
	{
		static tbCore::DynamicStructure nullObject;
		nullObject = tbCore::DynamicStructure::kNullValue;
		return nullObject;
	}

	return tilePropertyIterator->second;	
//	return tilePropertyIterator->second.second;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbGame::TileSystem::GetTilePropertyAsString(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName) const
{
	return GetTileProperties(tileSetName, tileIndex).GetMember(propertyName).AsString();
}

//--------------------------------------------------------------------------------------------------------------------//

int tbGame::TileSystem::GetTilePropertyAsInteger(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName) const
{
	return GetTileProperties(tileSetName, tileIndex).GetMember(propertyName).AsInteger();
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::TileSystem::GetTilePropertyAsBoolean(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName) const
{
		return GetTileProperties(tileSetName, tileIndex).GetMember(propertyName).AsBoolean();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetTileProperty(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName, const tbCore::tbString& propertyValue)
{
	TileSetTable::iterator tileSetIterator(mTileSets.find(tileSetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the TileSystem.", tileSetName.c_str());
	tileSetIterator->second.mTileProperties[tileIndex].SetMember(propertyName, propertyValue);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetTileProperty(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName, const int propertyValue)
{
	TileSetTable::iterator tileSetIterator(mTileSets.find(tileSetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the TileSystem.", tileSetName.c_str());
	tileSetIterator->second.mTileProperties[tileIndex].SetMember(propertyName, propertyValue);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetTileProperty(const tbCore::tbString& tileSetName, const TileIndex& tileIndex,
	const tbCore::tbString& propertyName, const bool propertyValue)
{
	TileSetTable::iterator tileSetIterator(mTileSets.find(tileSetName));
	tb_error_if(tileSetIterator == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" was not found in the TileSystem.", tileSetName.c_str());
	tileSetIterator->second.mTileProperties[tileIndex].SetMember(propertyName, propertyValue);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::AddTileLayer(const tbCore::tbString& tileLayerName, const std::vector<tbGame::TileIndex>& tileData,
	const tbCore::tbString& tileSetName, const TileLocation& columnCount, const TileLocation& rowCount)
{
	tb_error_if(mTileLayers.find(tileLayerName) != mTileLayers.end(), "tbExternalError: A TileLayer with the name \"%s\" has already been added to system.", tileLayerName.c_str());

	TileSetTable::iterator tileSetItr(mTileSets.find(tileSetName));
	tb_error_if(tileSetItr == mTileSets.end(), "tbExternalError: A TileSet with the name \"%s\" does not exist in the TileSystem.", tileSetName.c_str());
	TileSetIndex tileSetIndex(0);
	for (NameContainer::const_iterator nameItr = mOrderedTileSetNames.begin(), nameEnd = mOrderedTileSetNames.end(); nameItr != nameEnd; ++nameItr, ++tileSetIndex)
	{
		if (*nameItr == tileSetName)
		{	//tileSetIndex is now the correct index.
			break;
		}
	}

	tb_error_if(tileSetIndex >= mOrderedTileSetNames.size(), "tbInternalError: TileSet Index is out of bounds.");

	TileLayer& newTileLayer = mTileLayers.insert(TileLayerTable::value_type(tileLayerName, TileLayer())).first->second;
	newTileLayer.mTileData = tileData;
	newTileLayer.mTileSetData = TileLayer::TileSetContainer(1, tileSetIndex);
	newTileLayer.mTileColumns = columnCount;
	newTileLayer.mTileRows = rowCount;
	newTileLayer.mIsVisible = true;
	newTileLayer.mPosition = tbMath::Vector2::kZero;

	mOrderedLayerNames.push_back(tileLayerName);

	mLayerRenderers.insert(LayerRendererTable::value_type(tileLayerName, new tbImplementation::TileSystemRenderer));
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::AddTileLayer(const tbCore::tbString& tileLayerName, const std::vector<TileIndex>& tileData,
									const std::vector<TileSetIndex>& tileSetData, const TileLocation& columnCount, const TileLocation& rowCount)
{
//	tb_error("tbInternalError: Currently this function is unimplemented.  Multiple tilesets per layer is not yet supported.");

	tb_error_if(mTileLayers.find(tileLayerName) != mTileLayers.end(), "tbExternalError: A TileLayer with the name \"%s\" has already been added to system.", tileLayerName.c_str());

	TileLayer& newTileLayer = mTileLayers.insert(TileLayerTable::value_type(tileLayerName, TileLayer())).first->second;
	newTileLayer.mTileData = tileData;
	newTileLayer.mTileSetData = tileSetData;
	newTileLayer.mTileColumns = columnCount;
	newTileLayer.mTileRows = rowCount;
	newTileLayer.mIsVisible = true;
	newTileLayer.mPosition = tbMath::Vector2::kZero;

	mOrderedLayerNames.push_back(tileLayerName);

	mLayerRenderers.insert(LayerRendererTable::value_type(tileLayerName, new tbImplementation::TileSystemRenderer));
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::SetLayerVisible(const tbCore::tbString& tileLayerName, const bool isVisible)
{
	TileLayerTable::iterator tileLayerIterator(mTileLayers.find(tileLayerName));
	tb_error_if(tileLayerIterator == mTileLayers.end(), "tbExternalError: A TileLayer with the name \"%s\" was not found in the system.", tileLayerName.c_str());

	tileLayerIterator->second.mIsVisible = isVisible;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//
//tbGame::TileLocation tbGame::TileSystem::PixelSpaceToColumn(const float locationX) const
//{
//	if (locationX < 0.0f) { return kInvalidTileLocation; }
//	int columnd(locationX / mTile
//}
//
//tbGame::TileLocation tbGame::TileSystem::PixelSpaceToRow(const float locationY) const
//{
//	return 0;
//}
//
//float tbGame::TileSystem::ColumnToPixelSpace(const tbGame::TileLocation& column) const
//{
//	return 0.0f;
//}
//
//float tbGame::TileSystem::RowToPixelSpace(const tbGame::TileLocation& row) const
//{
//	return 0.0f;
//}
//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::OnRender(void) const
{
	const tbMath::Vector2& position(GetPosition());

	//tbImplementation::Renderer::PushMatrix();
	//tbImplementation::Renderer::Translate(position.x, position.y, GetDepth());
	//ApplyTransform();

	for (NameContainer::const_iterator layerNameIterator = mOrderedLayerNames.begin(), layerNameEnd = mOrderedLayerNames.end(); layerNameIterator != layerNameEnd; ++layerNameIterator)
	{
		TileLayerTable::const_iterator tileLayerIterator(mTileLayers.find(*layerNameIterator));
		tb_error_if(tileLayerIterator == mTileLayers.end(), "tbExternalError: A TileLayer with the name \"%s\" was not found in the system.", layerNameIterator->c_str());

		const TileLayer& tileLayer(tileLayerIterator->second);

		if (false == tileLayer.mIsVisible)
		{
			continue;
		}

		tbImplementation::TileSystemRenderer& tileRenderer(*mLayerRenderers.find(*layerNameIterator)->second);

		//if (tileRenderer.NeedsUpdate())
		{
			tileRenderer.Clear(); //Only clear and add new if needed.
			const bool isSingleTileSet(1 == tileLayer.mTileSetData.size());
			const TileSetIndex& singleTileSetIndex(tileLayer.mTileSetData[0]);
			//TODO: TIM: Was getting a crash on the following line and quickly added the ternary statement, is that correct?
			const TileSet& singleTileSet((kInvalidTileSetIndex == singleTileSetIndex) ? mTileSets.find(mOrderedTileSetNames[0])->second : mTileSets.find(mOrderedTileSetNames[singleTileSetIndex])->second);

			for (TileLocation row(0); row < tileLayer.mTileRows; ++row)
			{
				const size_t rowIndex(row * tileLayer.mTileColumns);
//				const float rowPosition(tileLayer.mPosition.y + (row * tileSet.mSpriteMap.GetFrameHeight()));
	
				for (TileLocation column(0); column < tileLayer.mTileColumns; ++column)
				{
					//TODO: TIM: OPTIMIZE: This very likely has some needless cruft going on...
					const TileIndex& tileIndex(tileLayer.mTileData[column + rowIndex]);
					const TileSetIndex& tilesetIndex((true == isSingleTileSet) ? singleTileSetIndex : tileLayer.mTileSetData[column + rowIndex]);
					if (tilesetIndex >= mOrderedTileSetNames.size())
					{
						continue;
					}

					const TileSet& tileSet((true == isSingleTileSet) ? singleTileSet : mTileSets.find(mOrderedTileSetNames[tilesetIndex])->second);

					if (tileIndex >= tileSet.mSpriteMap.GetIndexCount())
					{
						continue;
					}

					const float rowPosition(tileLayer.mPosition.y + (row * tileSet.mSpriteMap.GetFrameHeight()));
					const float columnPosition(tileLayer.mPosition.x + (column * tileSet.mSpriteMap.GetFrameWidth()));
					tileRenderer.AddQuad(tileSet.mSpriteMap.GetSpriteFrameAtIndex(tileIndex), tbMath::Vector2(columnPosition, rowPosition));
				}
			}
		}

		tileRenderer.Render();
	}

//	tbGraphics::GraphicList::OnRender();
	//tbGraphics::Graphic::OnRender();

	//PopTransform();
	//tbImplementation::Renderer::PopMatrix();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGame::TileSystem::UpdateColliderInformation(void)
{
	tb_error_if(nullptr == mCollider, "tbInternalError: Expected collider to always be valid.");

	mCollider->ClearStaticObjects();

	for (TileSystem::TileLayerTable::iterator tileLayerItr = mTileLayers.begin(), tileLayerEnd = mTileLayers.end(); tileLayerItr != tileLayerEnd; ++tileLayerItr)
	{
		const tbCore::tbString tileLayerName(tileLayerItr->first);
		const TileLayer& tileLayer(tileLayerItr->second);

		if (false == tileLayer.mIsVisible)
		{
			continue;
		}

		const bool isSingleTileSet(1 == tileLayer.mTileSetData.size());
		const TileSetIndex& singleTileSetIndex(tileLayer.mTileSetData[0]);
		//TODO: TIM: Was getting a crash on the following line and quickly added the ternary statement, is that correct?		
		//const TileSet& singleTileSet(mTileSets.find(mOrderedTileSetNames[singleTileSetIndex])->second);
		const TileSet& singleTileSet((kInvalidTileSetIndex == singleTileSetIndex) ? mTileSets.find(mOrderedTileSetNames[0])->second : mTileSets.find(mOrderedTileSetNames[singleTileSetIndex])->second);

		for (TileLocation row(0); row < tileLayer.mTileRows; ++row)
		{
			const size_t rowIndex(row * tileLayer.mTileColumns);

			for (TileLocation column(0); column < tileLayer.mTileColumns; ++column)
			{
				const TileIndex& tileIndex(tileLayer.mTileData[column + rowIndex]);
				const TileSetIndex& tilesetIndex((true == isSingleTileSet) ? singleTileSetIndex : tileLayer.mTileSetData[column + rowIndex]);
				if (tilesetIndex >= mOrderedTileSetNames.size())
				{
					continue;
				}

				const tbCore::tbString& tileSetName(mOrderedTileSetNames[(true == isSingleTileSet) ? singleTileSetIndex : tilesetIndex]);
				const TileSet& tileSet((true == isSingleTileSet) ? singleTileSet : mTileSets.find(tileSetName)->second);
				const tbCore::DynamicStructure& tileProperties(GetTileProperties(tileSetName, tileIndex));

				const tbCore::DynamicStructure& isSolid = tileProperties["solid"];
				if ((false == isSolid.IsNil() && true == isSolid.AsBoolean()) || tb_string("solid") == tileLayerName)
				{
					const float rowPosition(tileLayer.mPosition.y + (row * tileSet.mSpriteMap.GetFrameHeight()));
					const float columnPosition(tileLayer.mPosition.x + (column * tileSet.mSpriteMap.GetFrameWidth()));
					const float tileWidth(tileSet.mSpriteMap.GetFrameWidth());
					const float tileHeight(tileSet.mSpriteMap.GetFrameHeight());

					tbMath::Vector2 centerTilePosition(columnPosition + (tileWidth / 2.0f), rowPosition + (tileHeight / 2.0f));
					tbMath::Unstable::BoundingVolume boundingVolume(tileWidth, tileHeight);

					mCollider->AddStaticObject(boundingVolume, centerTilePosition);

					//tileRenderer.AddQuad(tileSet.mSpriteMap.GetSpriteFrameAtIndex(tileIndex), tbMath::Vector2(columnPosition, rowPosition));
				}
			}
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::TileSystem::MoveEntity(const tbMath::Vector2& currentPosition, tbMath::Vector2& finalPosition,
	const tbMath::Unstable::BoundingVolume& boundingVolume) const
{
	return mCollider->MoveEntity(currentPosition, finalPosition, boundingVolume);
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::TileSystem::IsPointInSolid(const tbMath::Vector2& pointPosition) const
{
	return mCollider->IsPointInSolid(pointPosition);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGame::TileSystem::TileSet::TileSet(const tbGraphics::SpriteMap& spriteMap) :
	mSpriteMap(spriteMap)
{
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

struct TiledTileSetInfo
{
	tbCore::tbString mTileSet;
	tbGame::TileIndex mStartIndex;
	tbGame::TileIndex mFinalIndex;
};

typedef std::vector<TiledTileSetInfo> TiledInfoContainer;

//--------------------------------------------------------------------------------------------------------------------//

tbGame::TileIndex GetTileIndexOf(const TiledInfoContainer& tilesetInfo, const int tiledTileIndex)
{
	if (0 == tiledTileIndex)
	{
		return tbGame::kInvalidTileIndex;
	}

//	for (TiledInfoContainer::const_iterator itr = tilesetInfo.begin(), itrEnd = tilesetInfo.end(); itr != itrEnd; ++itr)
//	{
//		if
//	}

	tb_error_if(true == tilesetInfo.empty(), "tbExternalError: Expected at least one tile set when loading a Tiled map.");
	for (size_t tilesetIndex(tilesetInfo.size() - 1); tilesetIndex > 0; --tilesetIndex)
	{
		int startIndex = tilesetInfo[tilesetIndex].mStartIndex;
		if (startIndex > tiledTileIndex)
		{
			continue;
		}

		int temporaryTileIndex(tiledTileIndex - startIndex);
		tb_error_if(temporaryTileIndex < 0 || temporaryTileIndex > 0xFFFF, "tbDataError: Expected Tiled map, tileset and tile index data to be within bounds 0 < index < max uint16.");
		return static_cast<tbGame::TileIndex>(temporaryTileIndex);
	}

	int startIndex = tilesetInfo[0].mStartIndex;
	int temporaryTileIndex(tiledTileIndex - startIndex);
	tb_error_if(temporaryTileIndex < 0 || temporaryTileIndex > 0xFFFF, "tbDataError: Expected Tiled map, tileset and tile index data to be within bounds 0 < index < max uint16.");
	return static_cast<tbGame::TileIndex>(temporaryTileIndex);

//	tb_error("tbDataError: Was unable to find tileIndex for tileset while loading a Tiled map.");
//	return tbGame::kInvalidTileIndex;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGame::TileSetIndex GetTileSetIndexOf(const TiledInfoContainer& tilesetInfo, const int tiledTileIndex)
{
	if (0 == tiledTileIndex)
	{
		return tbGame::kInvalidTileSetIndex;
	}

	tb_error_if(true == tilesetInfo.empty(), "tbExternalError: Expected at least one tile set when loading a Tiled map.");
	for (size_t tilesetIndex(tilesetInfo.size() - 1); tilesetIndex > 0; --tilesetIndex)
	{
		int startIndex = tilesetInfo[tilesetIndex].mStartIndex;
		if (startIndex > tiledTileIndex)
		{
			continue;
		}

		tb_error_if(tilesetIndex < 0 || tilesetIndex > 0xFF, "tbDataError: Expected Tiled map, tileset index data to be within bounds 0 < index < max uint8.");
		return static_cast<tbGame::TileSetIndex>(tilesetIndex);
	}

	return  static_cast<tbGame::TileSetIndex>(0);
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGame::Extensions::LoadTileSystemFromTiled(tbGame::TileSystem& tileSystem, const tbCore::tbString& tiledFilepath)
{
	tileSystem.ClearMap();

	const tbCore::DynamicStructure& fileData(tbCore::LoadJsonFile(tiledFilepath));
	if (true == fileData.IsNil())
	{
		tb_error("tbDataError: Was unable to load or parse the Tiled json file.");
		return false;
	}

	tb_error_if(1 != fileData["version"], "tbDataError: Expected Tiled map \"%s\" to be \"version\" 1.", tiledFilepath.c_str());

	int temporaryValue(0);

	//TODO: TIM: Implementation: Allow other render by creating an enum type and allowing the render order to change.
	const tbCore::tbString kExpectedRenderOrder(tb_string("right-down"));
	tb_error_if(kExpectedRenderOrder != fileData["renderorder"], "tbDataError: Expected Tiled map \"%s\" to be saved with renderorder: \"%s\"", tiledFilepath.c_str(), kExpectedRenderOrder.c_str());

	//TODO: TIM: Implementation: Allow non-ortho tile system (iso-metric / hex) (Create enum type, offset and render-ordering above.)
	const tbCore::tbString kExpectedOrientation(tb_string("orthogonal"));
	tb_error_if(kExpectedOrientation != fileData["orientation"], "tbDataError: Expected Tiled map \"%s\" to be saved with \"orientation\": \"%s\"", tiledFilepath.c_str(), kExpectedOrientation.c_str());

	const tbCore::uint16 tileWidthPixels(fileData["tilewidth"].AsInteger());
	const tbCore::uint16 tileHeightPixels(fileData["tileheight"].AsInteger());
	tb_error_if(0 == tileWidthPixels || 0 == tileHeightPixels, "tbDataError: Expected Tiled map \"%s\" to have non-zero \"tilewidth\" / \"tileheight\" fields.", tiledFilepath.c_str());

	const tbCore::uint16 mapWidthTiles(fileData["width"].AsInteger());
	const tbCore::uint16 mapHeightTiles(fileData["height"].AsInteger());
	tb_error_if(0 == mapWidthTiles || 0 == mapHeightTiles, "tbDataError: Expected Tiled map \"%s\" to have non-zero \"width\" / \"height\" map size fields.", tiledFilepath.c_str());

	const tbCore::DynamicStructure& layerArray(fileData["layers"]);
	tb_error_if(true == layerArray.IsNil() || false == layerArray.IsArray() || 0 == layerArray.Size(), "tbDataError: Expected Tiled map \"%s\" to have array field \"layers\" with at least one item.", tiledFilepath.c_str());

	const tbCore::DynamicStructure& tileSetArray(fileData["tilesets"]);
	tb_error_if(true == tileSetArray.IsNil() || false == tileSetArray.IsArray() || 0 == tileSetArray.Size(), "tbDataError: Expected Tiled map \"%s\" to have array field \"tilesets\" with at least one item.", tiledFilepath.c_str());
//	tb_error_if(1 < tileSetArray.Size(), "tbInternalError: Currently only a single tileset is allowed per Tiled map \"%s\"", tiledFilepath.c_str());

	const tbCore::DynamicStructure& mapProperties(fileData["properties"]);
	tb_error_if(false == mapProperties.IsNil() && false == mapProperties.IsStructure(), "tbDataError: Expected Tiled map \"%s\" field \"properties\" to be a structure.", tiledFilepath.c_str());
	tileSystem.SetPropertiesForMap(mapProperties);

	TiledInfoContainer tiledTileSets(tileSetArray.Size());


	//
	//Load each tileset into the TileSystem.
	//
	for (size_t tileSetIndex(0); tileSetIndex < tileSetArray.Size(); ++tileSetIndex)
	{
		const tbCore::DynamicStructure& tileSetData(tileSetArray[tileSetIndex]);

		const tbCore::tbString tileSetName(tileSetData["name"]);
		tb_error_if(true == tileSetName.empty(), "tbDataError: Expected Tiled map \"%s\" tileset \"name\" field to contain a valid name.", tiledFilepath.c_str());

		temporaryValue = tileSetData["margin"].AsInteger();
		tb_error_if(temporaryValue < 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"margin\" to be within range of a PixelSpace value, 0 < value < max uint16.", tiledFilepath.c_str(), tileSetName.c_str());
		const tbGraphics::PixelSpace tileSetMargin(static_cast<tbGraphics::PixelSpace>(temporaryValue));

		temporaryValue = tileSetData["spacing"].AsInteger();
		tb_error_if(temporaryValue < 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"spacing\" to be within range of a PixelSpace value, 0 < value < max uint16.", tiledFilepath.c_str(), tileSetName.c_str());
		const tbGraphics::PixelSpace tileSetSpacing(static_cast<tbGraphics::PixelSpace>(temporaryValue));

		temporaryValue = tileSetData["firstgid"].AsInteger();
		tb_error_if(temporaryValue <= 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"firstgid\" to be in range: 0 < id < max uint16", tiledFilepath.c_str(), tileSetName.c_str());
		const TileIndex firstTileIndex(static_cast<TileIndex>(temporaryValue));

		temporaryValue = tileSetData["tilewidth"].AsInteger();
		tb_error_if(temporaryValue < 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"tilewidth\" to be within range of a PixelSpace value, 0 < value < max uint16.", tiledFilepath.c_str(), tileSetName.c_str());
		const tbGraphics::PixelSpace tileSetTileWidthPixels(static_cast<tbGraphics::PixelSpace>(temporaryValue));

		temporaryValue = tileSetData["tileheight"].AsInteger();
		tb_error_if(temporaryValue < 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"tileheight\" to be within range of a PixelSpace value, 0 < value < max uint16.", tiledFilepath.c_str(), tileSetName.c_str());
		const tbGraphics::PixelSpace tileSetTileHeightPixels(static_cast<tbGraphics::PixelSpace>(temporaryValue));

		const tbCore::DynamicStructure& tileSetProperties(tileSetData["properties"]);
		tb_error_if(false == tileSetProperties.IsNil() && false == tileSetProperties.IsStructure(), "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" field \"properties\" to be a structure.", tiledFilepath.c_str(), tileSetName.c_str());

		tbCore::tbString tileSetImage(tileSetData["image"]);
		tbCore::tbString tileSetTexture = tbCore::GetChildFilepath(tiledFilepath, tileSetImage);
		tb_error_if(true == tileSetTexture.empty(), "tbDataError: Expected Tiled map  \"%s\" tileset \"%s\" field \"image\" to be a valid filepath relative to app or map file.", tiledFilepath.c_str(), tileSetName.c_str());

		tbGraphics::TextureHandle textureHandle = tbGraphics::theTextureManager.CreateTextureFromFile(tileSetTexture);

		tbGraphics::SpriteMap spriteMap(textureHandle, tileSetTileWidthPixels, tileSetTileHeightPixels, tileSetSpacing, tileSetSpacing, tileSetMargin, tileSetMargin);
		tileSystem.AddTileSet(tileSetName, spriteMap, tileSetProperties);

		const tbCore::DynamicStructure& tileProperties(tileSetData["tileproperties"]);
		tb_error_if(false == tileProperties.IsNil() && false == tileProperties.IsStructure(), "tbDataError: Expected Tiled tileset \"%s\" field \"tileproperties\" to be a structure if exists.", tiledFilepath.c_str());
		if (false == tileProperties.IsNil())
		{
			for (auto itr = tileProperties.BeginStructure(), itrEnd = tileProperties.EndStructure(); itr != itrEnd; ++itr)
			{
				int tileIndex = tbCore::FromString<int>(itr->first);
				tb_error_if(tileIndex < 0 || tileIndex > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" tileset \"%s\" tile properties to contain valid tile indices.", tiledFilepath.c_str(), tileSetName.c_str());

				//TODO: TIM: Which of these do we want, to account for firstTileIndex or not?
				//tileSystem.SetPropertiesForTile(tileSetName, static_cast<TileIndex>(tileIndex - firstTileIndex), itr->second);
				tileSystem.SetPropertiesForTile(tileSetName, static_cast<TileIndex>(tileIndex), itr->second);
			}
		}
		
		//
		tiledTileSets[tileSetIndex].mTileSet = tileSetName;
		tiledTileSets[tileSetIndex].mStartIndex = static_cast<TileIndex>(firstTileIndex);
		tiledTileSets[tileSetIndex].mFinalIndex = static_cast<TileIndex>(firstTileIndex + spriteMap.GetIndexCount());
	}


	//
	//Load each tile layer into the TileSystem.
	//
	for (size_t layerIndex(0); layerIndex < layerArray.Size(); ++layerIndex)
	{
		const tbCore::DynamicStructure& layerData(layerArray[layerIndex]);

		const tbCore::tbString layerName(layerData["name"]);
		tb_error_if(true == layerName.empty(), "tbDataError: Expected Tiled map \"%s\" layer \"name\" field to contain a valid name.", tiledFilepath.c_str());

		tbCore::tbString layerType(layerData["type"]);
		if (tb_string("tilelayer") != layerType)
		{	//Skip the non-tile layers.
			continue;
		}

		//TODO: TIM: Implementation: Do we want to support layers of different sizes / offsets? (Tiled doesn't seem to? but may in future?)
		const int layerColumnCount(layerData["width"].AsInteger());
		const int layerRowCount(layerData["height"].AsInteger());
		tb_error_if(layerColumnCount < 0 || layerColumnCount > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" layer \"%s\" field \"width\" to be in range: 0 < width < 16bit uvalue.", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(layerRowCount < 0 || layerRowCount > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" layer \"%s\" field \"height\" to be in range: 0 < height < 16bit uvalue.", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(mapWidthTiles != layerColumnCount, "tbInternalError: Currently expecting Tiled map \"%s\" layer \"%s\" field \"width\" to match the with all tile layers and maps.", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(mapHeightTiles != layerRowCount, "tbInternalError: Currently expecting Tiled map \"%s\" layer \"%s\" field \"height\" to match the with all tile layers and maps.", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(0 != layerData["x"], "tbInternalError: Currently expecting Tiled map \"%s\" layer \"%s\" field \"x\" to be 0 on all tile layers and maps.", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(0 != layerData["y"], "tbInternalError: Currently expecting Tiled map \"%s\" layer \"%s\" field \"y\" to be 0 on all tile layers and maps.", tiledFilepath.c_str(), layerName.c_str());

		//TODO: TIM: Maybe add support for this?  SetLayerColor() ?
		tb_error_if(false == tbMath::IsEqual(1.0f, layerData["opacity"].AsFloat()), "tbInternalError: Expected Tiled map \"%s\" layer \"%s\" to have \"opacity\" field set to 1.", tiledFilepath.c_str(), layerName.c_str());

		bool layerIsVisible = layerData["visible"].AsBoolean();

		const TileIndex kNumberOfTiles(mapWidthTiles * mapHeightTiles);
		const tbCore::DynamicStructure& layerTileData(layerData["data"]);
		tb_error_if(true == layerTileData.IsNil() || false == layerTileData.IsArray(), "tbDataError: Expected Tiled map \"%s\" layer \"%s\" to have an array field \"data\"", tiledFilepath.c_str(), layerName.c_str());
		tb_error_if(kNumberOfTiles != layerTileData.Size(), "tbDataError: Expected Tiled map \"%s\" layer \"%s\" \"data\" array to contain width * height elements.", tiledFilepath.c_str(), layerName.c_str());

		std::vector<TileIndex> tileData(kNumberOfTiles);
		std::vector<TileSetIndex> tilesetData(kNumberOfTiles);

		for (TileIndex tileIndex(0); tileIndex < kNumberOfTiles; ++tileIndex)
		{
			temporaryValue = layerTileData[tileIndex].AsInteger();
//			tb_error_if(temporaryValue < 0 || temporaryValue > 0xFFFF, "tbDataError: Expected Tiled map \"%s\" layer \"%s\" to have all tile index data within bounds 0 < index < max uint16.", tiledFilepath.c_str(), layerName.c_str());
//			tileData[tileIndex] = static_cast<TileIndex>(temporaryValue) - 1;
			tileData[tileIndex] = GetTileIndexOf(tiledTileSets, temporaryValue);
			tilesetData[tileIndex] = GetTileSetIndexOf(tiledTileSets, temporaryValue);
		}

		tileSystem.AddTileLayer(layerName, tileData, tilesetData, static_cast<TileLocation>(layerColumnCount), static_cast<TileLocation>(layerRowCount));
		tileSystem.SetLayerVisible(layerName, layerIsVisible);
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//
