///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tbi_renderer.h"
#include "../../core/tb_opengl.h"
#include "../../core/tb_defines.h"
#include "../../core/tb_string.h"
#include "../../core/tb_error.h"
#include "../../core/implementation/tbi_diagnostics.h"
#include "../../debug_tool_set/tb_debug_logger.h"

#include <stack>
#include <iostream>
#include <cstdio>

//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	int tbiBasicVertexShader(0);
	int tbiBasicFragmentShader(0);
	int tbiBasicShaderProgram(0);

#if defined(tb_legacy_gl_support)
	bool tbiLegacyRenderer(false);
#endif /* tb_legacy_gl_support */

	//Note: Can't actually use kIdentity because it may not have been initialized yet.
	tbMath::Matrix4 tbiProjectionMatrix(1.0f, 0.0f, 0.0f, 0.0f,   0.0f, 1.0f, 0.0f, 0.0f,   0.0f, 0.0f, 1.0f, 0.0f,    0.0f, 0.0f, 0.0f, 1.0f);
	tbMath::Matrix4 tbiModelViewMatrix(1.0f, 0.0f, 0.0f, 0.0f,   0.0f, 1.0f, 0.0f, 0.0f,   0.0f, 0.0f, 1.0f, 0.0f,    0.0f, 0.0f, 0.0f, 1.0f);
	tbMath::Matrix4 tbiProjectionModelViewMatrix(1.0f, 0.0f, 0.0f, 0.0f,   0.0f, 1.0f, 0.0f, 0.0f,   0.0f, 0.0f, 1.0f, 0.0f,    0.0f, 0.0f, 0.0f, 1.0f);

	typedef std::stack<tbMath::Matrix4> MatrixStack;
	MatrixStack tbiModelViewMatrixStack;

	const GLenum tbiPrimitiveModes[] = { GL_TRIANGLES, GL_TRIANGLE_STRIP, GL_TRIANGLE_FAN, GL_LINES, GL_LINE_STRIP, GL_LINE_LOOP };

}; /* namespace tbImplementation */

bool tbImplementation::Renderer::tbiIsRendererAvailable(false);

//--------------------------------------------------------------------------------------------------------------------//

#if defined(tb_debug_build)
  #if defined(tb_windows)
void GLAPIENTRY debugOutput(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const void* userParam)
#else
void debugOutput(GLenum source, GLenum type, GLuint id, GLenum severity, GLsizei length, const GLchar* message, const GLvoid* userParam)
  #endif /* tb_windows */
{
	const char* severityString = "";
	switch( severity )
	{
		case GL_DEBUG_SEVERITY_HIGH:         severityString = "GL_DEBUG_SEVERITY_HIGH";          break;
		case GL_DEBUG_SEVERITY_MEDIUM:       severityString = "GL_DEBUG_SEVERITY_MEDIUM";        break;
		case GL_DEBUG_SEVERITY_LOW:          severityString = "GL_DEBUG_SEVERITY_LOW";           break;
		case GL_DEBUG_SEVERITY_NOTIFICATION: severityString = "GL_DEBUG_SEVERITY_NOTIFICATION";  break;
	}

	static bool showNotifications = true;

	if (severity != GL_DEBUG_SEVERITY_NOTIFICATION || true == showNotifications)
	{
		tb_log("Source: %x, Type: %x, Id: %x, Severity: %s\n - %s\n", source, type, id, severityString, message);
	}
}
#endif /* tb_debug_build */

//--------------------------------------------------------------------------------------------------------------------//

char* LoadFileToBuffer(const char* file)
{
#ifdef tb_visual_cpp
	FILE* inputFile(nullptr);
	fopen_s(&inputFile, file, "rb");
#else
	FILE* inputFile = fopen(file, "rb");
#endif

	if (nullptr == inputFile)
	{	//Failed to load the file.
		return NULL;
	}

	fseek(inputFile, 0, SEEK_END); //Seek to the end of the file to get the fileSize.
	long fileSize = ftell(inputFile);

	char* buffer = new char[fileSize + 1];
	fseek(inputFile, 0, SEEK_SET); //Go back to the beginning of the file.
	fread(buffer, fileSize, 1, inputFile);
	buffer[fileSize] = 0;
	fclose(inputFile);

	return buffer;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::Renderer::DidShaderCompile(int glShader)
{
	int isCompiled(0);
	tb_check_gl_errors(glGetShaderiv(glShader, GL_COMPILE_STATUS, &isCompiled));
	if (0 == isCompiled)
	{
		int logSize(0);
		tb_check_gl_errors(glGetShaderiv(glShader, GL_INFO_LOG_LENGTH, &logSize));
		char* log = new char[logSize + 1];
		tb_check_gl_errors(glGetShaderInfoLog(glShader, logSize, &logSize, log));
		tbi_log_error("Shader Compiler Error: %s\n", log);
		std::cout << "Shader Compiler Error: " << log << std::endl;
		printf("Shader Compiler Error: %s\n", log);
		delete [] log;
		return false;
	}
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::Renderer::DidProgramLink(int glProgram)
{
	int isLinked(0);
	tb_check_gl_errors(glGetProgramiv(glProgram, GL_LINK_STATUS, &isLinked));
	if (0 == isLinked)
	{
		int logSize(0);
		tb_check_gl_errors(glGetProgramiv(glProgram, GL_INFO_LOG_LENGTH, &logSize));
		char* log = new char[logSize];
		tb_check_gl_errors(glGetProgramInfoLog(glProgram, logSize, &logSize, log));
		tbi_log_error("Shader Linker Error: %s\n", log);
		delete [] log;
		return false;
	}
	return true;
}

//--------------------------------------------------------------------------------------------------------------------//

#if defined(tb_legacy_gl_support)

bool tbImplementation::Renderer::IsLegacyRenderer(void)
{
	return tbiLegacyRenderer;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::SetLegacyRenderer(bool isLegacyRenderer)
{
	tbiLegacyRenderer = isLegacyRenderer;
}

#endif /* tb_legacy_gl_support */

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::InitializeBasicRenderer(void)
{
#ifdef tb_debug_build
	if (GLEW_ARB_debug_output)
	{
		glDebugMessageCallbackARB(debugOutput, nullptr);
		glGetError();
	}
#endif /* tb_debug_build */

	ClearErrors();

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		glMatrixMode(GL_MODELVIEW);
	}
	else
#endif /* tb_legacy_gl_support */
	{
		char* vertexSource = LoadFileToBuffer("data/vertex_shader.vert");
		tb_error_if(nullptr == vertexSource, "tbInternalError: VertexShader was not loaded, perhaps not shipped?");
		char* fragmentSource = LoadFileToBuffer("data/fragment_shader.frag");
		tb_error_if(nullptr == fragmentSource, "tbInternalError: FragmentShader was not loaded, perhaps not shipped?");

		tbiBasicVertexShader = glCreateShader(GL_VERTEX_SHADER);
		tb_check_gl_errors(glShaderSource(tbiBasicVertexShader, 1, (const GLchar**)&vertexSource, 0));
		tb_check_gl_errors(glCompileShader(tbiBasicVertexShader));
		tb_error_if(false == DidShaderCompile(tbiBasicVertexShader), "tbInternalError: Shader did not compile, what does log say?");

		tbiBasicFragmentShader = glCreateShader(GL_FRAGMENT_SHADER);
		tb_check_gl_errors(glShaderSource(tbiBasicFragmentShader, 1, (const GLchar**)&fragmentSource, 0));
		tb_check_gl_errors(glCompileShader(tbiBasicFragmentShader));
		tb_error_if(false == DidShaderCompile(tbiBasicFragmentShader), "tbInternalError: Shader did not compile, what does log say?");

		tbiBasicShaderProgram = glCreateProgram();
		tb_check_gl_errors(glAttachShader(tbiBasicShaderProgram, tbiBasicVertexShader));
		tb_check_gl_errors(glAttachShader(tbiBasicShaderProgram, tbiBasicFragmentShader));		
		tb_check_gl_errors(glBindAttribLocation(tbiBasicShaderProgram, 0, "vertexPosition"));
		tb_check_gl_errors(glBindAttribLocation(tbiBasicShaderProgram, 1, "vertexColor"));
		tb_check_gl_errors(glBindAttribLocation(tbiBasicShaderProgram, 2, "vertexTextureUV"));
		tb_check_gl_errors(glLinkProgram(tbiBasicShaderProgram));
		tb_error_if(false == DidProgramLink(tbiBasicShaderProgram), "tbInternalError: Program did not link, what does log say?");

		tb_check_gl_errors(glUseProgram(tbiBasicShaderProgram));

		tb_safe_array_delete(vertexSource);
		tb_safe_array_delete(fragmentSource);
	}

	tbiIsRendererAvailable = true;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::CleanupBasicRenderer(void)
{
	tbiIsRendererAvailable = false;

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{

	}
	else
#endif /* tb_legacy_gl_support */
	{
		tb_check_gl_errors(glUseProgram(0));

		//tb_check_gl_errors(glDetachShader(tbiBasicShaderProgram, tbiBasicFragmentShader));
		//tb_check_gl_errors(glDetachShader(tbiBasicShaderProgram, tbiBasicVertexShader));
		tb_check_gl_errors(glDeleteProgram(tbiBasicShaderProgram));
		tbiBasicShaderProgram = 0;

		tb_check_gl_errors(glDeleteShader(tbiBasicFragmentShader));
		tbiBasicFragmentShader = 0;

		tb_check_gl_errors(glDeleteShader(tbiBasicVertexShader));
		tbiBasicVertexShader = 0;
	}

	glFlush();
	glFinish();
}

//--------------------------------------------------------------------------------------------------------------------//

//		RGBA -> RGBA:                                             RGBA -> ABGR:
//		[ 1, 0, 0, 0 ]   [ Ri ]  = [ Ro, Go, Bo, Ao ]             [ 0, 0, 0, 1 ]   [ Ri ]  = [ Ro, Go, Bo, Ao ]
//		[ 0, 1, 0, 0 ] * [ Gi ]                                   [ 0, 0, 1, 0 ] * [ Gi ]
//		[ 0, 0, 1, 0 ]   [ Bi ]                                   [ 0, 1, 0, 0 ]   [ Bi ]
//		[ 0, 0, 0, 1 ]   [ Ai ]                                   [ 1, 0, 0, 0 ]   [ Ai ]
//
//
//		RGBA -> ARGB:                                             R--- -> ---A:
//		[ 0, 0, 0, 1 ]   [ Ri ]  = [ Ro, Go, Bo, Ao ]             [ 0, 0, 0, 0 ]   [ Ri ]  = [ Ro, Go, Bo, Ao ]
//		[ 1, 0, 0, 0 ] * [ Gi ]                                   [ 0, 0, 0, 0 ] * [ Gi ]
//		[ 0, 1, 0, 0 ]   [ Bi ]                                   [ 0, 0, 0, 0 ]   [ Bi ]
//		[ 0, 0, 1, 0 ]   [ Ai ]                                   [ 1, 0, 0, 0 ]   [ Ai ]

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::SetColorMatrix(bool forText)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		return;
	}
#endif /* tb_legacy_gl_support */

	tbMath::Matrix4 colorMatrix(tbMath::Matrix4::kIdentity);
	tbMath::Vector4 colorTint(0.0f, 0.0f, 0.0f, 0.0f);

	if (true == forText)
	{
		colorMatrix(0, 0) = 1.0f;
		colorMatrix(1, 0) = 1.0f;
		colorMatrix(2, 0) = 1.0f;
		colorMatrix(3, 0) = 1.0f;

		colorMatrix(1, 1) = 0.0f;
		colorMatrix(2, 2) = 0.0f;
		colorMatrix(3, 3) = 0.0f;

		colorTint = tbMath::Vector4(1.0f, 1.0f, 1.0f, 0.0f);
	}

	int colorMatrixLocation(0);
	int colorTintLocation(0);

//TODO: TIM: HACK: Provide better support for shaders to be bound outside of TurtleBrains?
	int currentProgram(0);
	tb_check_gl_errors(glGetIntegerv(GL_CURRENT_PROGRAM, &currentProgram));
//END: HACK: Provide better support for shaders to be bound outside of TurtleBrains?

	tb_check_gl_errors(colorMatrixLocation = glGetUniformLocation(currentProgram, "colorMatrix"));
	tb_check_gl_errors(colorTintLocation = glGetUniformLocation(currentProgram, "colorTint"));

	tb_error_if(-1 == colorMatrixLocation, "tbInternalError: GLSL Uniform \"colorMatrix\" was not found in shader program.");
	tb_error_if(-1 == colorTintLocation, "tbInternalError: GLSL Uniform \"colorTint\" was not found in shader program.");
	tb_check_gl_errors(glUniformMatrix4fv(colorMatrixLocation, 1, GL_FALSE, colorMatrix));
	tb_check_gl_errors(glUniform4f(colorTintLocation, colorTint.x, colorTint.y, colorTint.z, colorTint.w));
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::SetProjectionMatrix(const tbMath::Matrix4& projectionMatrix)
{
	tbiProjectionMatrix = projectionMatrix;

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		glMatrixMode(GL_PROJECTION);
		glLoadTransposeMatrixf(projectionMatrix);
		glMatrixMode(GL_MODELVIEW);
		return;
	}
#endif /* tb_legacy_gl_support */

	SetModelViewMatrix(tbiModelViewMatrix); //Multiplies and shoves to shader.
}

//--------------------------------------------------------------------------------------------------------------------//

const tbMath::Matrix4& tbImplementation::Renderer::GetProjectionMatrix(void)
{
	return tbiProjectionMatrix;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::SetModelViewMatrix(const tbMath::Matrix4& modelViewMatrix)
{
	tbiModelViewMatrix = modelViewMatrix;
	tbMath::MatrixMultiply(&tbiProjectionModelViewMatrix, &tbiProjectionMatrix, &tbiModelViewMatrix);

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		glLoadTransposeMatrixf(modelViewMatrix);
		return;
	}
#endif /* tb_legacy_gl_support */

	int mvpMatrixLocation(0);
	int modelViewMatrixLocation(0);

//TODO: TIM: HACK: Provide better support for shaders to be bound outside of TurtleBrains?
	int currentProgram(0);
	tb_check_gl_errors(glGetIntegerv(GL_CURRENT_PROGRAM, &currentProgram));
//END: HACK: Provide better support for shaders to be bound outside of TurtleBrains?

	tb_check_gl_errors(mvpMatrixLocation = glGetUniformLocation(currentProgram, "mvpMatrix"));
	tb_error_if(-1 == mvpMatrixLocation, "tbInternalError: GLSL Uniform \"mvpMatrix\" was not found in shader program.");
	tb_check_gl_errors(glUniformMatrix4fv(mvpMatrixLocation, 1, GL_TRUE, tbiProjectionModelViewMatrix));

	//Pretty sure this was added for ZoomCarWorld3 / InternalCombustion support, like the get GL_CURRENT_PROGRAM hack.
	tb_check_gl_errors(modelViewMatrixLocation = glGetUniformLocation(currentProgram, "modelViewMatrix"));
	if (-1 != modelViewMatrixLocation)
	{
		tb_check_gl_errors(glUniformMatrix4fv(modelViewMatrixLocation, 1, GL_TRUE, tbiModelViewMatrix));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

const tbMath::Matrix4& tbImplementation::Renderer::GetModelViewMatrix(void)
{
	return tbiModelViewMatrix;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::PushMatrix(void)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glPushMatrix());
		return;
	}
#endif /* tb_legacy_gl_support */

	tbiModelViewMatrixStack.push(tbiModelViewMatrix);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::PopMatrix(void)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glPopMatrix());
		return;
	}
#endif /* tb_legacy_gl_support */

	tb_error_if(true == tbiModelViewMatrixStack.empty(), "tbExternalError: Can't pop matrix, there is no matrix to pop.");
	SetModelViewMatrix(tbiModelViewMatrixStack.top());
	tbiModelViewMatrixStack.pop();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::MultiplyMatrix(const tbMath::Matrix4& matrix)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glMultMatrixf(matrix));
		return;
	}
#endif /* tb_legacy_gl_support */

	tbMath::Matrix4 result(tbMath::kSkipInitialization);
	tbMath::MatrixMultiply(&result, &tbiModelViewMatrix, &matrix);
	SetModelViewMatrix(result);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::Translate(float x, float y, float z)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glTranslatef(x, y, z));
		return;
	}
#endif /* tb_legacy_gl_support */

	tbMath::Matrix4 translation(tbMath::kSkipInitialization);
	tbMath::Matrix4 result(tbMath::kSkipInitialization);

	tbMath::MatrixCreateTranslation(&result, x, y, z);
	tbMath::MatrixTranspose(&translation, &result); 

	tbMath::MatrixMultiply(&result, &tbiModelViewMatrix, &translation);
	SetModelViewMatrix(result);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::Rotate(float angleInDegrees, float x, float y, float z)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glRotatef(angleInDegrees, x, y, z));
		return;
	}
#endif /* tb_legacy_gl_support */

	tbMath::Matrix4 rotation(tbMath::kSkipInitialization);
	tbMath::Matrix4 result(tbMath::kSkipInitialization);

	const tbMath::Vector3 rotationAxis(tbMath::Vector3(x, y, z).GetNormalized());
	tbMath::MatrixCreateRotationA(&result, &rotationAxis, angleInDegrees);
	tbMath::MatrixTranspose(&rotation, &result);

	tbMath::MatrixMultiply(&result, &tbiModelViewMatrix, &rotation);
	SetModelViewMatrix(result);	
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::Scale(float x, float y, float z)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glScalef(x, y, z));
		return;
	}
#endif /* tb_legacy_gl_support */

	tbMath::Matrix4 scaling(tbMath::kSkipInitialization);
	tbMath::Matrix4 result(tbMath::kSkipInitialization);

	tbMath::MatrixCreateScale(&result, x, y, z);
	tbMath::MatrixTranspose(&scaling, &result);

	tbMath::MatrixMultiply(&result, &tbiModelViewMatrix, &scaling);
	SetModelViewMatrix(result);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::BeginDraw(void)
{
	GLint viewport[4] = { 0, 0, 1280, 720 };
	tb_check_gl_errors(glGetIntegerv(GL_VIEWPORT, viewport));

	{
		float left = static_cast<float>(viewport[0]);
		float right = static_cast<float>(viewport[0] + viewport[2]);
		float top = static_cast<float>(viewport[1]);
		float bottom = static_cast<float>(viewport[1] + viewport[3]);
		float nearp = -1.0f;
		float farp = 1.0f;

		tbMath::Matrix4 orthographicProjection;
		orthographicProjection(0, 0) = 2.0f / (right - left);
		orthographicProjection(1, 1) = 2.0f / (top - bottom);
		orthographicProjection(2, 2) = -2.0f / (farp - nearp);
		orthographicProjection(3, 3) = 1.0f;
		orthographicProjection(3, 0) = -(right + left) / (right - left);
		orthographicProjection(3, 1) = -(top + bottom) / (top - bottom);
		orthographicProjection(3, 2) = -((farp + nearp) / (farp - nearp));
		SetProjectionMatrix(orthographicProjection);
	}

	SetModelViewMatrix(tbMath::Matrix4::kIdentity);

	//Enable depth testing so we can have layers.  Initially this was disabled.
	tb_check_gl_errors(glEnable(GL_DEPTH_TEST));
	tb_check_gl_errors(glDepthFunc(GL_LEQUAL));

	//Have the pixels blend with what is already behind it.
	tb_check_gl_errors(glEnable(GL_BLEND));
	tb_check_gl_errors(glBlendFunc(GL_SRC_ALPHA, GL_ONE_MINUS_SRC_ALPHA));

	//Only Clockwise wound triangles are to be visible.
	tb_check_gl_errors(glEnable(GL_CULL_FACE));
	tb_check_gl_errors(glCullFace(GL_BACK));
	tb_check_gl_errors(glFrontFace(GL_CW));

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glPushAttrib(GL_DEPTH_BUFFER_BIT | GL_ENABLE_BIT | GL_COLOR_BUFFER_BIT));
		tb_check_gl_errors(glDisable(GL_LIGHTING));
		tb_check_gl_errors(glEnable(GL_TEXTURE_2D));

		//A pixel has to pass this alpha test to write value to the depth buffer.
		tb_check_gl_errors(glEnable(GL_ALPHA_TEST));
		tb_check_gl_errors(glAlphaFunc(GL_GREATER, 0.1f));

		return;
	}
	else
#endif /* tb_legacy_gl_support */
	{
		//TODO: TIM: HACK: Provide better support for shaders to be bound outside of TurtleBrains?
		int currentProgram(0);
		glGetIntegerv(GL_CURRENT_PROGRAM, &currentProgram);
		//END: HACK: Provide better support for shaders to be bound outside of TurtleBrains?

		//Note: Imagine glBindTexture(textureId) does this: textures[activeTexture] = textureId
		tb_check_gl_errors(glActiveTexture(GL_TEXTURE0 + 0));
		tb_check_gl_errors(glUniform1i(glGetUniformLocation(currentProgram, "diffuseTexture"), 0));

		SetColorMatrix(false);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::EndDraw(void)
{
#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		tb_check_gl_errors(glPopAttrib());
	}
#endif /* tb_legacy_gl_support */
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::Render(const PrimitiveType& primitiveType,
	const tbImplementation::Renderer::Vertex2D* vertices, const size_t& vertexCount)
{	
	if (0 == vertexCount)
	{
		return;
	}

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		glBegin(tbiPrimitiveModes[primitiveType]);
		for (size_t v = 0; v < vertexCount; ++v)
		{
			const Vertex2D& vertex(vertices[v]);
			glTexCoord2f(vertex.u, vertex.v);
			glColor4ub(vertex.abgr & 0x000000FF, (vertex.abgr & 0x0000FF00) >> 8, (vertex.abgr & 0x00FF0000) >> 16, (vertex.abgr & 0xFF000000) >> 24);
			glVertex2f(vertex.x, vertex.y);
		}
		glEnd();
		return;
	}
#endif /* tb_legacy_gl_support */

	unsigned int vertexArrayObject(0);
	unsigned int vertexBuffer(0);

	tb_check_gl_errors(glGenVertexArrays(1, &vertexArrayObject));
	tb_check_gl_errors(glBindVertexArray(vertexArrayObject));
		
	{
		tb_check_gl_errors(glGenBuffers(1, &vertexBuffer));
		tb_check_gl_errors(glBindBuffer(GL_ARRAY_BUFFER, vertexBuffer));
		tb_check_gl_errors(glBufferData(GL_ARRAY_BUFFER, vertexCount * sizeof(Vertex2D), vertices, GL_STATIC_DRAW));

		//Set the positional attributes.
		tb_check_gl_errors(glVertexAttribPointer(0, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex2D), 0));
		tb_check_gl_errors(glEnableVertexAttribArray(0));
			
		//Set the color attributes.
		tb_check_gl_errors(glVertexAttribPointer(1, 4, GL_UNSIGNED_BYTE, GL_TRUE, sizeof(Vertex2D), (void*)8));
		tb_check_gl_errors(glEnableVertexAttribArray(1));

		//Set the texture coordinate attributes.
		tb_check_gl_errors(glVertexAttribPointer(2, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex2D), (void*)12));
		tb_check_gl_errors(glEnableVertexAttribArray(2));
	}

	tb_check_gl_errors(glDrawArrays(tbiPrimitiveModes[primitiveType], 0, vertexCount));
	
	{
		tb_check_gl_errors(glDisableVertexAttribArray(0));
		tb_check_gl_errors(glDisableVertexAttribArray(1));
		tb_check_gl_errors(glDisableVertexAttribArray(2));

		tb_check_gl_errors(glBindBuffer(GL_ARRAY_BUFFER, 0));
		tb_check_gl_errors(glDeleteBuffers(1, &vertexBuffer));

		tb_check_gl_errors(glBindVertexArray(0));
		tb_check_gl_errors(glDeleteVertexArrays(1, &vertexArrayObject));
	}
}

void tbImplementation::Renderer::Render(const PrimitiveType& primitiveType, const Vertex3D* vertices, const size_t& vertexCount)
{
	if (0 == vertexCount)
	{
		return;
	}

#if defined(tb_legacy_gl_support)
	if (true == IsLegacyRenderer())
	{
		glBegin(tbiPrimitiveModes[primitiveType]);
		for (size_t v = 0; v < vertexCount; ++v)
		{
			const Vertex3D& vertex(vertices[v]);
			glTexCoord2f(vertex.u, vertex.v);
			glColor4ub(vertex.abgr & 0x000000FF, (vertex.abgr & 0x0000FF00) >> 8, (vertex.abgr & 0x00FF0000) >> 16, (vertex.abgr & 0xFF000000) >> 24);
			glVertex3f(vertex.x, vertex.y, vertex.z);
		}
		glEnd();
		return;
	}
#endif /* tb_legacy_gl_support */

	unsigned int vertexArrayObject(0);
	unsigned int vertexBuffer(0);

	tb_check_gl_errors(glGenVertexArrays(1, &vertexArrayObject));
	tb_check_gl_errors(glBindVertexArray(vertexArrayObject));

	{
		/// Above could be more global initialization ///
		tb_check_gl_errors(glGenBuffers(1, &vertexBuffer));
		tb_check_gl_errors(glBindBuffer(GL_ARRAY_BUFFER, vertexBuffer));
		tb_check_gl_errors(glBufferData(GL_ARRAY_BUFFER, vertexCount * sizeof(Vertex3D), vertices, GL_STATIC_DRAW));

		//Set the positional attributes.
		tb_check_gl_errors(glVertexAttribPointer(0, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex3D), 0));
		tb_check_gl_errors(glEnableVertexAttribArray(0));

		//Set the normal attributes.
		tb_check_gl_errors(glVertexAttribPointer(1, 3, GL_FLOAT, GL_FALSE, sizeof(Vertex3D), (void*)12));
		tb_check_gl_errors(glEnableVertexAttribArray(1));

		//Set the color attributes.
		tb_check_gl_errors(glVertexAttribPointer(2, 4, GL_UNSIGNED_BYTE, GL_TRUE, sizeof(Vertex3D), (void*)24));
		tb_check_gl_errors(glEnableVertexAttribArray(2));

		//Set the texture coordinate attributes.
		tb_check_gl_errors(glVertexAttribPointer(3, 2, GL_FLOAT, GL_FALSE, sizeof(Vertex3D), (void*)28));
		tb_check_gl_errors(glEnableVertexAttribArray(3));
	}

	tb_check_gl_errors(glDrawArrays(tbiPrimitiveModes[primitiveType], 0, vertexCount));

	{
		tb_check_gl_errors(glDisableVertexAttribArray(0));
		tb_check_gl_errors(glDisableVertexAttribArray(1));
		tb_check_gl_errors(glDisableVertexAttribArray(2));
		tb_check_gl_errors(glDisableVertexAttribArray(3));

		tb_check_gl_errors(glBindBuffer(GL_ARRAY_BUFFER, 0));
		tb_check_gl_errors(glDeleteBuffers(1, &vertexBuffer));
		/// Below could be more global cleanup ///
		tb_check_gl_errors(glBindVertexArray(0));
		tb_check_gl_errors(glDeleteVertexArrays(1, &vertexArrayObject));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::RenderSprite(const tbMath::Vector2& topLeft, const tbMath::Vector2& bottomRight, const tbCore::uint32& color)
{
	Renderer::Vertex2D vertices[] =
	{
		{ topLeft.x, bottomRight.y,      color,   0.0f, 1.0f,    },
		{ topLeft.x, topLeft.y,          color,   0.0f, 0.0f,    },
		{ bottomRight.x, bottomRight.y,  color,   1.0f, 1.0f,    },
		{ bottomRight.x, topLeft.y,      color,   1.0f, 0.0f,    },
	};

	Renderer::Render(Renderer::kTriangleStrip, vertices, 4);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::ClearErrors(void)
{
	int ignoredErrorCount(0);
	const int kMaximumErrorCount(100);
	while (GL_NO_ERROR != glGetError() && ignoredErrorCount < kMaximumErrorCount)
	{
		++ignoredErrorCount;
	}

	tb_log_if(ignoredErrorCount > 0, "tbWarning: %d OpenGL errors cleared and ignored.\n", ignoredErrorCount);
	tb_log_if(kMaximumErrorCount <= ignoredErrorCount, "tbWarning: Some OpenGL errors may exist still, maximum reached.\n");
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Renderer::CheckErrors(const char* const fromFile, int onLineNumber, const char* const message)
{
	GLenum errorCode = glGetError();

	while (GL_NO_ERROR != errorCode)
	{
		tbCore::tbString errorString(tb_string("unknown"));
		switch (errorCode)
		{
			case GL_INVALID_ENUM:      errorString = tb_string("GL_INVALID_ENUM"); break;
			case GL_INVALID_VALUE:     errorString = tb_string("GL_INVALID_VALUE"); break;
			case GL_INVALID_OPERATION: errorString = tb_string("GL_INVALID_OPERATION"); break;
			case GL_STACK_OVERFLOW:    errorString = tb_string("GL_STACK_OVERFLOW"); break;
			case GL_STACK_UNDERFLOW:   errorString = tb_string("GL_STACK_UNDERFLOW"); break;
			case GL_OUT_OF_MEMORY:     errorString = tb_string("GL_OUT_OF_MEMORY"); break;
			case GL_INVALID_FRAMEBUFFER_OPERATION:     errorString = tb_string("GL_INVALID_FRAMEBUFFER_OPERATION"); break;
			case GL_TABLE_TOO_LARGE:   errorString = tb_string("GL_TABLE_TOO_LARGE"); break;
		};

		tb_error("tbInternalError: glError[%s] calling: %s on line %d of file: %s\n", errorString.c_str(), message, onLineNumber, fromFile);
		errorCode = glGetError();
	}
}

//--------------------------------------------------------------------------------------------------------------------//
