///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_animated_sprite.h"
#include "tb_sprite_manager.h"

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimationSequence::AnimationSequence(const TextureHandle& textureHandle) :
	mTexture(textureHandle)
{
	tb_error_if(kInvalidTexture == mTexture, "tbExternalError: Expected parameter texture to be a valid texture.");
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimationSequence::AnimationSequence(const SpriteMap& spriteMap, const std::vector<size_t> frames) :
	mTexture(spriteMap.GetTextureHandle())
{
	tb_error_if(kInvalidTexture == mTexture, "tbExternalError: Expected parameter texture to be a valid texture.");

	for (size_t index(0); index < frames.size(); ++index)
	{
		AddFrame(spriteMap.GetSpriteFrameAtIndex(frames[index]));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimationSequence::AnimationSequence(const SpriteMap& spriteMap, const size_t& startFrameIndex, const size_t& frameCount) :
	mTexture(spriteMap.GetTextureHandle())
{
	tb_error_if(kInvalidTexture == mTexture, "tbExternalError: Expected parameter texture to be a valid texture.");

	for (size_t index(0); index < frameCount; ++index)
	{
		AddFrame(spriteMap.GetSpriteFrameAtIndex(startFrameIndex + index));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimationSequence::~AnimationSequence(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimationSequence::AddFrame(const SpriteFrame& frame)
{
	tb_error_if(kInvalidTexture == mTexture, "tbExternalError: Expected texture handle to be valid before a frame is added.");
	tb_error_if(mTexture != frame.mTexture, "tbExternalErro: Expected frame texture to be the same as the texture of sequence.");
	mAnimationFrames.push_back(frame);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimationSequence::AddFrame(const PixelSpace& frameX, const PixelSpace& frameY,
	const PixelSpace& frameWidth, const PixelSpace& frameHeight)
{
	tb_error_if(kInvalidTexture == mTexture, "tbExternalError: Expected texture handle to be valid before a frame is added.");
	SpriteFrame frame = SpriteFrame::CreateWith(mTexture, frameX, frameY, frameWidth, frameHeight);
	mAnimationFrames.push_back(frame);
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbGraphics::AnimationSequence::GetFrameCount(void) const
{
	return mAnimationFrames.size();
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGraphics::SpriteFrame& tbGraphics::AnimationSequence::GetFrame(const size_t& frameIndex) const
{
	tb_error_if(frameIndex >= mAnimationFrames.size(), "tbExternalError: Expected frameIndex to be within range: (0 <= frameIndex < frameCount");
	return mAnimationFrames[frameIndex];
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	float tbiDefaultTimePerFrame(1.0f / 30.0f); //30fps.
};

const float& tbGraphics::AnimatedSprite::kDefaultTimePerFrame(tbImplementation::tbiDefaultTimePerFrame);

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::SetAnimationFrameRate(int framesPerSecond)
{
	tbImplementation::tbiDefaultTimePerFrame = 1.0f / static_cast<float>(framesPerSecond);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite::AnimatedSprite(const SpriteFrame& spriteFrame) :
	Sprite(spriteFrame),	
	mCurrentSequence(nullptr),
	mCurrentFrameIndex(0),
	mTimePerFrame(kDefaultTimePerFrame),
	mFrameTimer(-1.0f),
	mIsPlaying(false),
	mIsLooping(false),
	mIsForward(true)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite::AnimatedSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) :
	Sprite(spriteSheetName, spriteName),
	mCurrentSequence(nullptr),
	mCurrentFrameIndex(0),
	mTimePerFrame(kDefaultTimePerFrame),
	mFrameTimer(-1.0f),
	mIsPlaying(false),
	mIsLooping(false),
	mIsForward(true)
{
	theSpriteManager.AddAnimationSequences(spriteSheetName, spriteName, *this);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite::AnimatedSprite(const AnimatedSprite& other) :
	Sprite(other),
	mAnimationSequences(other.mAnimationSequences),
	mCurrentSequence(nullptr),
	mCurrentFrameIndex(other.mCurrentFrameIndex),
	mTimePerFrame(other.mTimePerFrame),
	mFrameTimer(other.mFrameTimer),
	mIsPlaying(other.mIsPlaying),
	mIsLooping(other.mIsLooping),
	mIsForward(other.mIsForward)
{
	//The following is in the assignement operator and AddSequence(), we could store the current sequence as a string and
	//all of this could go away!
	tbCore::tbString otherSequenceName;
	if (nullptr != other.mCurrentSequence)
	{
		for (AnimationContainer::const_iterator itr = other.mAnimationSequences.begin(), itrEnd = other.mAnimationSequences.end(); itr != itrEnd; ++itr)
		{
			if (&itr->second == other.mCurrentSequence)
			{
				otherSequenceName = itr->first;
				break;
			}
		}

		tb_error_if(true == otherSequenceName.empty(), "tbInternalError: Expected to find the other animation sequence name");

		AnimationContainer::iterator itr = mAnimationSequences.find(otherSequenceName);
		tb_error_if(itr == mAnimationSequences.end(), "tbInternalError: Expected to find the sequence by name to reset current.");
		mCurrentSequence = &itr->second;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite& tbGraphics::AnimatedSprite::operator=(const AnimatedSprite& other)
{
	if (&other != this)
	{
		Sprite::operator=(other);
		mCurrentSequence = nullptr;
		mAnimationSequences = other.mAnimationSequences;
		mCurrentFrameIndex = other.mCurrentFrameIndex;
		mTimePerFrame = other.mTimePerFrame;
		mFrameTimer = other.mFrameTimer;
		mIsPlaying = other.mIsPlaying;
		mIsLooping = other.mIsLooping;
		mIsForward = other.mIsForward;

		//The following is in the copy constructor and AddSequence(), we could store the current sequence as a string and
		//all of this could go away!
		tbCore::tbString otherSequenceName;
		if (nullptr != other.mCurrentSequence)
		{
			for (AnimationContainer::const_iterator itr = other.mAnimationSequences.begin(), itrEnd = other.mAnimationSequences.end(); itr != itrEnd; ++itr)
			{
				if (&itr->second == other.mCurrentSequence)
				{
					otherSequenceName = itr->first;
					break;
				}
			}

			tb_error_if(true == otherSequenceName.empty(), "tbInternalError: Expected to find the other animation sequence name");

			AnimationContainer::iterator itr = mAnimationSequences.find(otherSequenceName);
			tb_error_if(itr == mAnimationSequences.end(), "tbInternalError: Expected to find the sequence by name to reset current.");
			mCurrentSequence = &itr->second;
		}
	}
	return *this;
}


//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::AnimatedSprite::~AnimatedSprite(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::AddSequence(const tbCore::tbString& sequenceName, const AnimationSequence& sequence)
{
	tb_error_if(true == sequenceName.empty(), "tbExternalError: Expected sequenceName to not be an empty string.");
	tb_error_if(mAnimationSequences.end() != mAnimationSequences.find(sequenceName), "tbExternalError: A sequence with name \"%s\" already exists in this AnimatedSprite.", sequenceName.c_str());
	tb_error_if(sequence.GetFrameCount() > 256, "tbExternalError: Expected sequence to contain no more than 256 frames of animation.");

	//The following is in the copy constructor and assignement operator, we could store the current sequence as a string
	//and all of this could go away!
	tbCore::tbString currentSequenceName;
	if (nullptr != mCurrentSequence)
	{
		for (AnimationContainer::const_iterator itr = mAnimationSequences.begin(), itrEnd = mAnimationSequences.end(); itr != itrEnd; ++itr)
		{
			if (&itr->second == mCurrentSequence)
			{
				currentSequenceName = itr->first;
				break;
			}
		}

		tb_error_if(true == currentSequenceName.empty(), "tbInternalError: Expected to find the current animation sequence name");
	}

	tb_error_if(mAnimationSequences.end() != mAnimationSequences.find(sequenceName), "tbExternalError: Expected not to find an AnimationSequence with name \"%s\" already existing.", sequenceName.c_str());

	//Container Changes here, so the mCurrentSequence ptr needs to readjust
	mAnimationSequences.insert(AnimationContainer::value_type(sequenceName, sequence));

	if (false == currentSequenceName.empty())
	{
		AnimationContainer::iterator itr = mAnimationSequences.find(currentSequenceName);
		tb_error_if(itr == mAnimationSequences.end(), "tbInternalError: Expected to find the sequence by name to reset current.");
		mCurrentSequence = &itr->second;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::AddMappedSequence(const tbCore::tbString& sequenceName, const SpriteMap& spriteMap, const std::vector<size_t> frames)
{
	const TextureHandle& textureHandle(GetSpriteFrame().mTexture);
	tb_error_if(textureHandle != spriteMap.GetTextureHandle(), "tbExternalError: SpriteMap Texture is different from AnimatedSprite texture.");

	AnimationSequence mappedSequence(spriteMap, frames);
	AddSequence(sequenceName, mappedSequence);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::AddMappedSequence(const tbCore::tbString& sequenceName, const SpriteMap& spriteMap,
	const size_t& startFrameIndex, const size_t& frameCount)
{
	const TextureHandle& textureHandle(GetSpriteFrame().mTexture);
	tb_error_if(textureHandle != spriteMap.GetTextureHandle(), "tbExternalError: SpriteMap Texture is different from AnimatedSprite texture.");

	AnimationSequence mappedSequence(spriteMap, startFrameIndex, frameCount);
	AddSequence(sequenceName, mappedSequence);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::PlayAnimation(const tbCore::tbString& sequenceName, const bool isLooping,
	const bool isForward, const float timePerFrame)
{
	AnimationContainer::const_iterator animationSequenceItr(mAnimationSequences.find(sequenceName));
	tb_error_if(animationSequenceItr == mAnimationSequences.end(), "tbExternalError: Expected to find an animation named %s to play.", sequenceName.c_str());
	tb_error_if(timePerFrame < 0.0f || timePerFrame > 1.0f, "tbExternalError: Expected timePerFrame to be within range: (0.0f < timePerFrame <= 1.0)");

	mTimePerFrame = timePerFrame;
	mIsLooping = isLooping;
	mIsForward = isForward;

	mCurrentSequence = &animationSequenceItr->second;
	if (true == mIsForward)
	{
		mCurrentFrameIndex = 0;
	}
	else
	{
		mCurrentFrameIndex = mCurrentSequence->GetFrameCount() - 1;
	}

	SetSpriteFrame(mCurrentSequence->GetFrame(mCurrentFrameIndex));
	mIsPlaying = true;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::PlayAnimation(const tbCore::tbString& sequenceName)
{
//	AnimationContainer::const_iterator animationSequenceItr(mAnimationSequences.find(sequenceName));
//	tb_error_if(animationSequenceItr == mAnimationSequences.end(), "tbExternalError: Expected to find an animation named %s to play.", sequenceName.c_str());
//	const AnimationSequence& animationSequence(animationSequenceItr->second);
//	PlayAnimation(sequenceName, animationSequence.IsLooping(), animationSequence.IsForward())
	PlayAnimation(sequenceName, false, true, kDefaultTimePerFrame);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::StopAnimation(void)
{
	mIsPlaying = false;
	mFrameTimer = -1.0f;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbGraphics::AnimatedSprite::IsAnimationPlaying(void) const
{
	return mIsPlaying;
}


//--------------------------------------------------------------------------------------------------------------------//

bool tbGraphics::AnimatedSprite::IsAnimationPlaying(const tbCore::tbString& sequenceName) const
{
	if (nullptr != mCurrentSequence)
	{
		for (AnimationContainer::const_iterator itr = mAnimationSequences.begin(), itrEnd = mAnimationSequences.end(); itr != itrEnd; ++itr)
		{
			if (&itr->second == mCurrentSequence)
			{
				return (itr->first == sequenceName && true == mIsPlaying) ? true : false;
			}
		}
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::AnimatedSprite::OnUpdate(const float deltaTime)
{
	if (true == mIsPlaying)
	{
		tb_error_if(nullptr == mCurrentSequence, "tbInternalError: Expected mCurrentSequence to be a valid sequence when playing.");

		mFrameTimer -= deltaTime;
		if (mFrameTimer < 0.0f)
		{
			if (true == mIsForward)
			{
				if (mCurrentFrameIndex < mCurrentSequence->GetFrameCount() - 1)
				{
					++mCurrentFrameIndex;
				}
				else if (true == mIsLooping)
				{
					mCurrentFrameIndex = 0;
				}
				else
				{
					mIsPlaying = false;
				}
			}
			else /* Play Backwards */
			{
				if (mCurrentFrameIndex > 0)
				{
					--mCurrentFrameIndex;
				}
				else if (true == mIsLooping)
				{
					mCurrentFrameIndex = mCurrentSequence->GetFrameCount() - 1;
				}
				else
				{
					mIsPlaying = false;
				}
			}

			SetSpriteFrame(mCurrentSequence->GetFrame(mCurrentFrameIndex));
			mFrameTimer = mTimePerFrame;
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//
