///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_basic_shapes.h"
#include "tb_texture_manager.h"
#include "../core/tb_opengl.h"
#include "implementation/tbi_renderer.h"

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PolygonShape::PolygonShape(void) :
	mPrimitiveType(tbImplementation::Renderer::kTriangles),
	mMinimumBounds(INFINITY, INFINITY),
	mMaximumBounds(-INFINITY, -INFINITY)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PolygonShape::PolygonShape(const PolygonShape& other) :
	Graphic(other),
	mPrimitiveType(other.mPrimitiveType),
	mVertices(other.mVertices),
	mMinimumBounds(other.mMinimumBounds),
	mMaximumBounds(other.mMaximumBounds)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PolygonShape::~PolygonShape(void)
{
	mVertices.clear();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::PolygonShape::SetColor(const tbGraphics::Color& newColor)
{
	for (size_t index(0); index < mVertices.size(); ++index)
	{
		mVertices[index].abgr = newColor.GetColorABGR();
	}

	Graphic::SetColor(newColor);
}

//--------------------------------------------------------------------------------------------------------------------//

//void tbGraphics::LineContainer::SetLineWidth(const float lineWidth)
//{
//	mLineWidth = lineWidth;
//}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::PolygonShape::OnRender(void) const
{
	if (true == mVertices.empty())
	{
		return;
	}

	tb_error_if(tbImplementation::Renderer::kTriangles == mPrimitiveType && mVertices.size() < 3, "tbExternalError: Expected at least 3 verticies to be added to a PolygonContainer with type Triangles");
	tb_error_if(tbImplementation::Renderer::kTriangles == mPrimitiveType && 0 == (mVertices.size() % 3), "tbExternalError: Expected at least verticies to contain a multiple of 3 for a PolygonContainer with type Triangles");
	tb_error_if(tbImplementation::Renderer::kTriangleStrip == mPrimitiveType && mVertices.size() < 3, "tbExternalError: Expected at least 3 verticies to be added to a PolygonContainer with type TriangleStrip");
	tb_error_if(tbImplementation::Renderer::kTriangleFan == mPrimitiveType && mVertices.size() < 3, "tbExternalError: Expected at least 3 verticies to be added to a PolygonContainer with type TriangleFan");
	tb_error_if(tbImplementation::Renderer::kLines == mPrimitiveType && mVertices.size() < 2, "tbExternalError: Expected at least 2 verticies to be added to a PolygonContainer with type Lines");
	tb_error_if(tbImplementation::Renderer::kLineStrip == mPrimitiveType && mVertices.size() < 2, "tbExternalError: Expected at least 2 verticies to be added to a PolygonContainer with type LineStrip");
	tb_error_if(tbImplementation::Renderer::kLineLoop == mPrimitiveType && mVertices.size() < 3, "tbExternalError: Expected at least 3 verticies to be added to a PolygonContainer with type LineLoop");

	//tb_check_gl_errors(glLineWidth(mLineWidth)); //Seems to only work at 1.0, anything larger causes GL_INVALID_VALUE ??
	//const tbMath::Vector2& position(GetPosition());

	theTextureManager.BindTexture(kBlankTexture);
	//tbImplementation::Renderer::PushMatrix();
	//tbImplementation::Renderer::Translate(position.x, position.y, GetDepth());

	tbImplementation::Renderer::Render(mPrimitiveType, mVertices.data(), mVertices.size());
	//tbImplementation::Renderer::PopMatrix();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::PolygonShape::ClearVertices(void)
{
	mVertices.clear();
	mMinimumBounds = tbMath::Vector2(INFINITY, INFINITY);
	mMaximumBounds = tbMath::Vector2(-INFINITY, -INFINITY);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::PolygonShape::AddVertex(const tbMath::Vector2& position)
{
	tbImplementation::Renderer::Vertex2D vertex = { position.x, position.y, GetColor().GetColorABGR(),   0.0f, 0.0f, };
	mVertices.push_back(vertex);

	if (position.x < mMinimumBounds.x) { mMinimumBounds.x = position.x; }
	if (position.y < mMinimumBounds.y) { mMinimumBounds.y = position.y; }
	if (position.x > mMaximumBounds.x) { mMaximumBounds.x = position.x; }
	if (position.y > mMaximumBounds.y) { mMaximumBounds.y = position.y; }
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::PolygonShape::GetPixelWidth(void) const
{
	return static_cast<PixelSpace>(mMaximumBounds.x - mMinimumBounds.x);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::PolygonShape::GetPixelHeight(void) const
{
	return static_cast<PixelSpace>(mMaximumBounds.y - mMinimumBounds.y);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedPolygonShape::OutlinedPolygonShape(void)
{
	mOutline.SetAsLineLoop();
	mOutline.SetScroll(0.0f); //A bit of a ?HACK? so that the camera doesn't get applied twice for the outline.
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedPolygonShape::OutlinedPolygonShape(const OutlinedPolygonShape& other) :
	PolygonShape(other),
	mOutline(other.mOutline)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedPolygonShape::~OutlinedPolygonShape(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::SetFillColor(const tbGraphics::Color& fillColor)
{
	PolygonShape::SetColor(fillColor);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::SetOutlineColor(const tbGraphics::Color& outlineColor)
{
	mOutline.SetColor(outlineColor);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::SetColor(const tbGraphics::Color& color)
{	//This is overridden to be disabled to avoid confusion.
	PolygonShape::SetColor(color);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::AddOutlineVertex(const tbMath::Vector2& position)
{
	mOutline.AddVertex(position);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::AddShapeVertex(const tbMath::Vector2& position)
{
	AddVertex(position);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::AddShapeAndOutlineVertex(const tbMath::Vector2& position)
{
	mOutline.AddVertex(position);
	AddVertex(position);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::OnRender(void) const
{
	PolygonShape::OnRender();
	mOutline.Render();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::ClearVertices(void)
{
	mOutline.ClearVertices();
	PolygonShape::ClearVertices();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::OutlinedPolygonShape::AddVertex(const tbMath::Vector2& position)
{
	PolygonShape::AddVertex(position);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::BoxShape::BoxShape(const float width, const float height, const tbGraphics::Color& color, const tbMath::Vector2& position)
{
	//const float halfWidth(width / 2.0f);
	//const float halfHeight(height / 2.0f);

	SetPosition(position);
	SetColor(color);

	SetAsTriangleStrip();
	AddVertex(tbMath::Vector2(0.0f, height));
	AddVertex(tbMath::Vector2(0.0f, 0.0f));
	AddVertex(tbMath::Vector2(width, height));
	AddVertex(tbMath::Vector2(width, 0.0f));
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::BoxShape::~BoxShape(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedBoxShape::OutlinedBoxShape(const float width, const float height, const tbGraphics::Color& fillColor,
	const tbGraphics::Color& outlineColor, const tbMath::Vector2& position)
{
	SetPosition(position);
	SetFillColor(fillColor);

	SetAsTriangleStrip();
	AddShapeVertex(tbMath::Vector2(0.0f, height));
	AddShapeVertex(tbMath::Vector2(0.0f, 0.0f));
	AddShapeVertex(tbMath::Vector2(width, height));
	AddShapeVertex(tbMath::Vector2(width, 0.0f));

	SetOutlineColor(outlineColor);
	
	AddOutlineVertex(tbMath::Vector2(0.0f, height));
	AddOutlineVertex(tbMath::Vector2(width, height));
	AddOutlineVertex(tbMath::Vector2(width, 0.0f));
	AddOutlineVertex(tbMath::Vector2(0.0f, 0.0f));
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedBoxShape::~OutlinedBoxShape(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::FullScreenQuad::FullScreenQuad(const tbGraphics::Color& color) :
	BoxShape(tbGraphics::ScreenWidth(), tbGraphics::ScreenHeight(), color, tbMath::Vector2::kZero)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::FullScreenQuad::~FullScreenQuad(void)
{
}


//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::CircleShape::CircleShape(const float radius, const tbGraphics::Color& color, const tbMath::Vector2& position, const int sectionCount)
{
	SetPosition(position);
	SetColor(color);

	const tbMath::Vector2 circleCenter(radius, radius);
	SetAsTriangleFan();
	AddVertex(circleCenter);

	for (int section = 0; section <= sectionCount; ++section)
	{
		const float percentage(static_cast<float>(section) / static_cast<float>(sectionCount));
		tbMath::Vector2 direction(cos(percentage * tbMath::kTwoPi), sin(percentage * tbMath::kTwoPi));
		AddVertex(circleCenter + (direction * radius));
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::CircleShape::~CircleShape(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedCircleShape::OutlinedCircleShape(const float radius, const tbGraphics::Color& fillColor,
	const tbGraphics::Color& outlineColor, const tbMath::Vector2& position, const int sectionCount)
{
	SetPosition(position);
	SetFillColor(fillColor);
	SetOutlineColor(outlineColor);

	const tbMath::Vector2 circleCenter(radius, radius);
	SetAsTriangleFan();
	AddShapeVertex(circleCenter);

	for (int section = 0; section <= sectionCount; ++section)
	{
		const float percentage(static_cast<float>(section) / static_cast<float>(sectionCount));
		tbMath::Vector2 vertexPosition(cos(percentage * tbMath::kTwoPi) * radius, sin(percentage * tbMath::kTwoPi) * radius);
		AddShapeAndOutlineVertex(circleCenter + vertexPosition);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::OutlinedCircleShape::~OutlinedCircleShape(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//
