///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_BasicShapes_h_
#define _TurtleBrains_BasicShapes_h_

#include "../math/tb_vector.h"
#include "implementation/tbi_renderer.h"
#include "tb_graphic.h"
//#include "tb_line.h" //For line container? which may become a PolygonShape ??
#include <vector>

namespace TurtleBrains
{
	namespace Graphics
	{
		class OutlinedPolygonShape;

		///
		///	@details Allow creation of simple and complex shapes by supplying several points that get shaped into triangles.
		///
		class PolygonShape : public Graphic
		{
		public:

			///
			///	@details Constructs an empty PolygonShape object that contains no vertices.
			///
			PolygonShape(void);

			///
			///	@details Constructs a PolygonShape object with the same properties as the other object being copied.
			///
			PolygonShape(const PolygonShape& other);

			///
			///	@details Cleans up after a PolygonShape, clearing the vertex container in the process.
			///
			virtual ~PolygonShape(void);

			///
			///	@details Sets the color of all the vertices in the PolygonShape.
			///
			virtual void SetColor(const tbGraphics::Color& newColor) override;

		protected:
			///
			///	@details Renders the PolygonShape object vertices as Triangles, TriangleStrip, TriangleFan, Lines, LineStrip
			///   or LineLoop depending on how the object is configured to treat the vertices. Depending on the configuration
			///   this may trigger an error condition based on number of vertices.
			///
			virtual void OnRender(void) const override;

			///
			///	@details Clears all the vertices in the PolygonShape.
			///
			virtual void ClearVertices(void);

			///
			///	@details Adds a vertex to a container of vertices for the PolygonShape. How it will be rendered depends on
			///   how the PolygonShape object is configured.
			///
			virtual void AddVertex(const tbMath::Vector2& position);

			///
			/// @details Computes and returns the width of the PolygonShape in pixels.
			///
			virtual PixelSpace GetPixelWidth(void) const override;

			///
			/// @details Computes and returns the height of the PolygonShape in pixels.
			///
			virtual PixelSpace GetPixelHeight(void) const override;

			///
			///	@details Configures the object to treat the vertices as a list of Triangles. In this mode the number of
			///   vertices must be equal-to or greater-than 3, and must also be divisible by 3.
			///
			void SetAsTriangles(void) { mPrimitiveType = tbImplementation::Renderer::kTriangles; }

			///
			///	@details Configures the object to treat the vertices as a fan of Triangles. In this mode the number of
			///   vertices must be equal-to or greater-than 3. The first vertex added will be the center of the fan and used
			///   for each of the triangles.
			///
			void SetAsTriangleFan(void) { mPrimitiveType = tbImplementation::Renderer::kTriangleFan; }

			///
			/// @details Configures the object to treat the vertices as a strip of Triangles. In this mode the number of
			///   vertices must be at least 3. Each vertex after the third will create a triangle using the two vertices
			///   added immediately before it.
			///
			void SetAsTriangleStrip(void) { mPrimitiveType = tbImplementation::Renderer::kTriangleStrip; }

			///
			/// @details Configures the object to treat the vertices as a list of lines. In this mode the number of vertices
			///   must be equal-to or greater-than 2, and must also be divisible by 2. Each pair of vertices added will form
			///   a line segment.
			///
			void SetAsLines(void) { mPrimitiveType = tbImplementation::Renderer::kLines; }

			///
			/// @details Configures the object to treat the vertices as a line strip. In this mode the number of vertices must
			///   be at least 2. Each vertex added after the first will create a line segment to the vertex added previously.
			///
			void SetAsLineStrip(void) { mPrimitiveType = tbImplementation::Renderer::kLineStrip; }

			///
			/// @details Configures the object to treat the vertices as a line loop, which is identical in behavior to a line
			///   strip in that at least 2 vertices must be added, (really 3 for a visible loop). Each vertex added after the
			///   first will create a line segment to the vertex added previously. A final line segment will be created
			///   automatically from the last added vertex to the very first vertex which will complete the loop.
			///
			void SetAsLineLoop(void) { mPrimitiveType = tbImplementation::Renderer::kLineLoop; }

		private:
			friend class OutlinedPolygonShape;

			std::vector<tbImplementation::Renderer::Vertex2D> mVertices;
			tbImplementation::Renderer::PrimitiveType mPrimitiveType;
			tbMath::Vector2 mMinimumBounds;
			tbMath::Vector2 mMaximumBounds;
		};


		///
		///	@details The OutlinedPolygonShape class extends the Polygon shape to include an outline around it along with the
		///   filled shape.
		///
		class OutlinedPolygonShape : public PolygonShape
		{
		public:
			///
			///	@details Constructs an empty shape with no vertices.
			///
			OutlinedPolygonShape(void);

			///
			///	@details Constructs a polygon shape by copying the vertices and information from an existing shape.
			///
			OutlinedPolygonShape(const OutlinedPolygonShape& other);

			///
			///	@details Cleans up all vertex data.
			///
			virtual ~OutlinedPolygonShape(void);

			///
			/// @details Clears all vertices from both the filled shape and the outline.
			///
			virtual void ClearVertices(void) override;

			///
			///	@details This will change the color of the filled portion of the OutlinedPolygonShape, behaves exactly like
			///   PolygonShape::SetColor().
			///
			/// @param fillColor The desired fill color.
			///
			void SetFillColor(const tbGraphics::Color& fillColor);

			///
			///	@details Change the color of the outline surrounding the shape.
			///
			/// @param outlineColor The desired color of the outline.
			///
			void SetOutlineColor(const tbGraphics::Color& outlineColor);

		protected:
			///
			///	@details Adds a vertex only to the outline line loop and not the filled shape. The outline is always a line
			///   list that loops back to the first vertex.
			///
			/// @param position The position of the vertex to add to the outlined line loop.
			///
			void AddOutlineVertex(const tbMath::Vector2& position);

			///
			/// @details Adds a vertex only to the filled shape triangles and not the outline list.
			///
			/// @param position The position of the vertex to add to the filled shape, behavior may depend on the mode of the
			///   polygon shape, Triangles, TriangleStrip or TriangleFan.
			///
			void AddShapeVertex(const tbMath::Vector2& position);

			///
			///	@details Adds a vertex to both the outlined line loop and the filled shape. See AddOutlineVertex and
			///   AddShapeVertex() for more information.
			///
			/// @param position The position of the vertex to add to both the outlined line loop and the filled shape.
			///
			void AddShapeAndOutlineVertex(const tbMath::Vector2& position);

			///
			/// @details Renders the filled polygon shape first, exactly as described in PolygonShape::OnRender() then renders
			///   the outline line loop.
			///
			virtual void OnRender(void) const override;

		private:
			//Privately overriden to protect from ambiguous SetColor() mistakes, use SetFillColor() or SetOutlineColor().
			virtual void SetColor(const tbGraphics::Color& color) override;

			//Privately overridden to protect from ambiguous AddVertex() mistakes, use AddOutlinedVertex(), AddShapeVertex()
			//  or AddShapeAndOutlineVertex() for clarity.
			virtual void AddVertex(const tbMath::Vector2& position) override;

			PolygonShape mOutline;
		};
		

		///
		///	@details The BoxShape is a special case of the PolygonShape that creates a box with a specified width and height
		///   of a specific color.
		///
		class BoxShape : public PolygonShape
		{
		public:
			///
			///	@details Constructs a PolygonShape in the form of a box with the given parameters, position, width, height.
			///
			/// @param width The width of the box, the units are in pixels unless projection/view matrix changed.
			/// @param height The vertical size of the box, in pixel units unless projection/view matrix changed.
			/// @param color What color the box should be when rendered, can be changed later with SetColor().
			/// @param position The position of the top-left corner of the box... TODO: TIM: Documentation: Check this fact,
			///   it may be different depending on anchor point.
			///
			BoxShape(const float width, const float height, const tbGraphics::Color& color = tbGraphics::Color::kWhite, const tbMath::Vector2& position = tbMath::Vector2::kZero);

			///
			///	@details Destroys a box object.
			///
			virtual ~BoxShape(void);
		};
		

		///
		///	@details The OutlinedBoxShape is a special case of the OutlinedPolygonShape that creates a box with a specified
		///   width and height and color, which also has an outline.
		///
		class OutlinedBoxShape : public OutlinedPolygonShape
		{
		public:
			///
			///	@details Constructs an OutlinedPolygonShape in the form of a box with the given parameters, position, width, height.
			///
			/// @param width The width of the box, the units are in pixels unless projection/view matrix changed.
			/// @param height The vertical size of the box, in pixel units unless projection/view matrix changed.
			/// @param fillColor What color the box should be when rendered, can be changed later with SetColor(), defaults to White.
			/// @param outlineColor What color the outline of the box should be when rendered, defaults to black.
			/// @param position The position of the top-left corner of the box... TODO: TIM: Documentation: Check this fact,
			///   it may be different depending on anchor point.
			///
			OutlinedBoxShape(const float width, const float height, const tbGraphics::Color& fillColor = tbGraphics::Color::kWhite,
				const tbGraphics::Color& outlineColor = tbGraphics::Color::kBlack, const tbMath::Vector2& position = tbMath::Vector2::kZero);

			///
			///	@details Destroys an outlined box object.
			///
			virtual ~OutlinedBoxShape(void);
		};

		
		///
		///	@details Special case of a BoxShape that covers the full screen. Useful to create a screen fade or other effect
		///   across the entire screen.
		///
		class FullScreenQuad : public BoxShape
		{
		public:
			///
			///	@details Constructs a FullScreenQuad object of a specified color.
			///
			FullScreenQuad(const tbGraphics::Color& color);

			///
			///	@details Destroys the fullscreen quad object.
			///
			virtual ~FullScreenQuad(void);
		};


		///
		///	@details The CircleShape is a special case of the PolygonShape that creates a circle with a specified radius
		///   and color to be displayed in a scene.
		///
		class CircleShape : public PolygonShape
		{
		public:
			///
			///	@details Constructs a PolygonShape in the form of a circle with specified size, position and color.
			///
			/// @param radius The size of the circle, in pixels unless the projection or view matrix changes.
			/// @param color The desired color of the circle when it renders.
			/// @param position The position of the center of the circle. //TODO: TIM: Documentation: Check this fact and make sure
			///   it works as intended with anchor points.
			/// @param sectionCount How many triangle sections the circle should be broken down to, more sections results in
			///   a smoother curve but also takes more rendering time and memory. In general smaller circles can reduce the
			///   number of sections while larger circles need more sections to be visually smooth.
			///
			CircleShape(const float radius, const tbGraphics::Color& color = tbGraphics::Color::kWhite, const tbMath::Vector2& position = tbMath::Vector2::kZero,
				const int sectionCount = 30);

			///
			///	@details Cleans up after the CircleShape.
			///
			virtual ~CircleShape(void);
		};


		///
		///	@details The OutlinedCircleShape is a special case of the OutlinedPolygonShape that creates a circle with a
		///   specified radius and color to be displayed in a scene.
		///
		class OutlinedCircleShape : public OutlinedPolygonShape
		{
		public:
			///
			///	@details Constructs a PolygonShape in the form of a circle with specified size, position and color.
			///
			/// @param radius The size of the circle, in pixels unless the projection or view matrix changes.
			/// @param fillColor The desired color of the circle when it renders.
			/// @param outlineColor The desired color of the outline when rendering.
			/// @param position The position of the center of the circle. //TODO: TIM: Documentation: Check this fact and make sure
			///   it works as intended with anchor points.
			/// @param sectionCount How many triangle sections the circle should be broken down to, more sections results in
			///   a smoother curve but also takes more rendering time and memory. In general smaller circles can reduce the
			///   number of sections while larger circles need more sections to be visually smooth.
			///
			OutlinedCircleShape(const float radius, const tbGraphics::Color& fillColor = tbGraphics::Color::kWhite,
				const tbGraphics::Color& outlineColor = tbGraphics::Color::kBlack, const tbMath::Vector2& position = tbMath::Vector2::kZero,
				const int sectionCount = 30);

			///
			///	@details Cleans up after the OutlinedCircleShape
			///
			virtual ~OutlinedCircleShape(void);
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_BasicShapes_h_ */
