///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Camera_h_
#define _TurtleBrains_Camera_h_

#include "../math/tb_vector.h"

namespace TurtleBrains
{
	namespace Graphics
	{

		class Camera;

		///
		/// @details Sets the current camera to be inactive and sets the provided camera as an active camera that
		///   will not be cleaned up. It is expected the camera object provided will be in existance until it
		///   is no longer the active camera.
		///
		/// @param camera The newly activated camera ready to apply to the view.
		///
		void SetActiveCamera(Camera& camera);

		///
		/// @details Sets the current camera to be inactive and sets the provided camera as an active camera that
		///   will automatically be cleaned up by calling delete on the object once a new camera becomes active.
		///
		/// @param camera A pointer to a camera object that was allocated with new. This object will become the active
		///   camera ready to apply to the view, and will automatically get deleted once a new camera is activated.
		///
		void SetActiveCamera(Camera* camera);

		///
		/// @details Retrieves the currently active camera, which there can only ever be one active camera and there
		///   is always an active camera.
		///
		Camera& GetActiveCamera(void);
		
		///
		///	@details Defines what the camera object can do and how it does it. The camera can move around in the world
		///   by using SetPosition(), and CameraShake can be applied using ShakeCamera().
		///
		class Camera
		{
		public:
			///
			/// @details Provides different camera shaking intensity levels.
			///
			enum IntensityLevel
			{
				kNoShaking,     ///< A shake intensity level that cancels out any current shaking effects.
				kLightShaking,  ///< A shake intensity level like some slight vibration from movement.
				kMediumShaking, ///< A shake intensity level like a small explosion.
				kHeavyShaking   ///< A shake intensity level like an earthquake is happening.
			};

			///
			/// @details Creates a camera object that places the top-left of the screen at 0,0 with no shake started.
			///
			Camera(void);

			///
			/// @details Cleans up after the camera object, which doesn't actually have any resources to clean.
			///
			virtual ~Camera(void);

			///
			/// @details Change the position of the camera, which moves the top-left (0,0) of the screen, to some other
			///   coordinate in the world based on the supplied cameraPosition parameter.
			///
			/// @param cameraPosition The world-space position where the top-left of the screen will be placed.
			///
			void SetPosition(const tbMath::Vector2& cameraPosition);

			///
			/// @details Returns the current position of the camera in world-space coordinates, representing the
			///   top-left (0,0) position of the screen if there is no shake, but if there is shake the screen
			///   (0,0) will move around slightly based on shake intensity.
			///
			const tbMath::Vector2& GetPosition(void) const;

			///
			/// @details Returns the current position of the camera in world-space coordinates, representing the
			///   top-left (0,0) position of the screen including the added shake offset which gets randomly set
			///   each frame. This is the position of the screen (0,0) if the camera is shaking or not.
			///
			tbMath::Vector2 GetPositionWithShake(void) const;

			///
			/// @details This must be called once per frame, every frame, and only once during the frame to update the
			///   camera shake and other camera features. When using tbGame::GameScene this is automatically called by
			///   the game scene loop update.
			///
			void Update(const float deltaTime);

			///
			/// @details Based on the shakeIntensity supplied the camera will start shaking various amounts and for
			///   different durations. Light and quick, to heavy and long.
			///
			/// @param shakeIntensity How vigorously the camera is shaking, slight rumble to earthquake rumbles.
			/// @param shakeDuration How long the camera will shake for, if less-than or equal-to 0, a default
			///   internal value will be used for duration. If shakeIntensity is kNoShaking the shakeDuration will
			///   have no effect.
			///
			void ShakeCamera(const IntensityLevel& shakeIntensity, const float shakeDuration = -1.0f);

			///
			/// @details Sets the rotation value of the camera so that the scene can rotate. <b>This is completely
			///   experimental</b> and may be removed from the interface completely. The camera will rotate the screen
			///   around the center of the screen, useful if you want the player to always face a single direction.
			///
			/// @param cameraRotation The amount of rotation to apply, in degrees. Positive values spins the camera
			///   clockwise, this will cause the screen to spin counter clockwise.
			///
			/// @note: Use this at your own risk, as it is not fully supported and may
			///   get removed from future versions.
			///
			void SetRotation(const float cameraRotation);

			///
			/// @details Retrieves the current rotation value of the camera. <b>This is completely experimental</b>
			///   and may be removed from the interface completely.
			///
			/// @note: Use this at your own risk, as it is not fully supported and may
			///   get removed from future versions.
			///
			float GetRotation(void) const;

		private:
			tbMath::Vector2 mPosition;
			tbMath::Vector2 mShakeOffset;
			float mRotation;

			float mShakeIntensity;
			float mShakeDuration;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Camera_h_ */
