///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_GraphicList_h_
#define _TurtleBrains_GraphicList_h_

#include "tb_graphic.h"
#include <list>

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		///	@details A GraphicList is a container of Graphic objects that get updated and rendered when the GraphicList
		///   gets updated or rendered. The contained Graphic objects are sorted and updated/rendered based on their depth
		///   in the scene. Depending on how the Graphic object is added to the GraphicList, the memory may also be deleted.
		///   See the difference between AddGraphic(Graphic&) and AddGraphic(Graphic*).
		///
		class GraphicList : public Graphic
		{
		public:
			///
			///	@details Constructs an empty list of Graphic objects.  Use AddGraphic() to add Graphic objects to be
			///   updated and rendered by the GraphicList object.
			///
			GraphicList(void);

			///
			///	@details Destructs the GraphicList object, which does not clean up any resources associated with the
			///   Graphic objects that were added to the list with AddGraphic.
			///
			virtual ~GraphicList(void);

			///
			///	@details Adds a Graphic to the list so it will be updated, if active, and rendered, if visible, whenever
			///   the object is updated or rendered.  The memory for the graphic is managed and delete graphic; will be called
			///   when either the GraphicList is destroyed or the Graphic is removed with RemoveGraphic().
			///
			void AddGraphic(Graphic* graphic);

			///
			///	@details Adds a Graphic to the list so it will be updated, if active, and rendered, if visible, whenever
			///   the object is updated or rendered.  The memory for the graphic is NOT managed and must remain available and
			///   in scope until it is removed with RemoveGraphic(). The GraphicList will not 'own' a Graphic added this way.
			///
			void AddGraphic(Graphic& graphic);

			///
			///	@details Removes a Graphic from the list so it will no longer be updated or rendered. This will only
			///   clean up memory or resources used by the graphic if added with AddGraphic(Graphic*), but those added with
			///   AddGraphic(Graphic&) are expected that the graphic remained available until it was removed.
			///
			void RemoveGraphic(Graphic* graphic);

			///
			///	@details Changes the color of all the Graphic objects in the GraphicList.
			///
			virtual void SetColor(const tbGraphics::Color& newColor) override;

		protected:
			///
			/// @details Calls Update() on each Graphic object contained in the list which will then call OnUpdate()
			///   if the object is active.
			///
			virtual void OnUpdate(const float deltaTime) override;

			///
			/// @details Calls Render() on each Graphic object contained in the list which will then call OnRender()
			///   if the object is visible.
			///
			/// @note This will currently trigger an error condition if the Graphic object is not relative to the
			///   parent location as this feature is currently unimplemented.  TODO: TIM: Implementation: Needs to
			///   be implemented.
			///
			virtual void OnRender(void) const override;

		private:
			void ReallyAddGraphic(Graphic* graphic, const bool managed);

			//These are all disabled from GraphicList, maybe they shouldn't be in the base Graphic
			//TODO: TIM: Refactor: Remove this from base Graphic and create a Transform or some other class.
			void SetOrigin(const AnchorLocation& anchor) { Graphic::SetOrigin(anchor); }
			void SetOrigin(const float originX, const float originY) { Graphic::SetOrigin(originX, originY); }
			void SetScale(const float scale) { Graphic::SetScale(scale);  }
			void SetScale(const float scaleX, const float scaleY) { Graphic::SetScale(scaleX, scaleY); }
			void SetRotation(const float rotation) { Graphic::SetRotation(rotation); }
			float GetRotation(void) const { return Graphic::GetRotation(); }

			void SetScroll(const float scroll) { SetScroll(scroll, scroll); }
			void SetScroll(const float scrollX, const float scrollY)
			{
				for (Graphic* graphic : mGraphics)
				{
					graphic->SetScroll(scrollX, scrollY);
				}
				Graphic::SetScroll(0.0f, 0.0f);
			}

			typedef std::list<Graphic*> GraphicContainer;
			GraphicContainer mGraphics;
			GraphicContainer mManagedGraphics;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Graphic_h_ */
