///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_sprite.h"
#include "tb_sprite_manager.h"
#include "../core/tb_opengl.h"
#include "../math/tb_math.h"
#include "implementation/tbi_renderer.h"

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteFrame tbGraphics::SpriteFrame::CreateWith(const TextureHandle& textureHandle)
{
	if (kInvalidTexture != textureHandle)
	{	//Create a SpriteFrame taking up the entire Texture.
		const unsigned int textureWidth(theTextureManager.GetTextureWidth(textureHandle));
		const unsigned int textureHeight(theTextureManager.GetTextureHeight(textureHandle));
		return CreateWith(textureHandle, 0, 0, textureWidth, textureHeight);
	}

	SpriteFrame frame = { 0.0f, 0.0f, 64.0f, 64.0f, 0.0f, 0.0f, 1.0f, 1.0f, kInvalidTexture };
	return frame;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::SpriteFrame tbGraphics::SpriteFrame::CreateWith(const TextureHandle& textureHandle,
	const PixelSpace& frameX, const PixelSpace& frameY, const PixelSpace& frameWidth, const PixelSpace& frameHeight)
{
	tb_error_if(kInvalidTexture == textureHandle, "tbExternalError: Expected texture handle to be valid when creating SpriteFrame.");
	const PixelSpace& textureWidth(theTextureManager.GetTextureWidth(textureHandle));
	const PixelSpace& textureHeight(theTextureManager.GetTextureHeight(textureHandle));

	const PixelSpace realFrameWidth((0 == frameWidth) ? textureWidth : frameWidth);
	const PixelSpace realFrameHeight((0 == frameHeight) ? textureHeight : frameHeight);

	tb_error_if(realFrameWidth > textureWidth, "tbExternalError: Expected frameWidth to be less than or equal to textureWidth");
	tb_error_if(realFrameWidth > textureWidth, "tbExternalError: Expected frameHeight to be less than or equal to textureHeight");
	tb_error_if(frameX > textureWidth - realFrameWidth, "tbExternalError: Expected frameX location to allow frame (width) to fit on texture.");
	tb_error_if(frameY > textureHeight - realFrameHeight, "tbExternalError: Expected frameY location to allow frame (height) to fit on texture.");

	SpriteFrame frame;
	frame.mTexture = textureHandle;
	frame.x = static_cast<float>(frameX);
	frame.y = static_cast<float>(frameY);
	frame.w = static_cast<float>(realFrameWidth);
	frame.h = static_cast<float>(realFrameHeight);

	frame.uvx = frame.x / textureWidth;
	frame.uvy = frame.y / textureHeight;
	frame.uvw = frame.w / textureWidth;
	frame.uvh = frame.h / textureHeight;

	return frame;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelTop(void) const
{
	return static_cast<PixelSpace>(y);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelLeft(void) const
{
	return static_cast<PixelSpace>(x);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelRight(void) const
{
	return static_cast<PixelSpace>(x + w);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelBottom(void) const
{
	return static_cast<PixelSpace>(y + h);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelWidth(void) const
{
	return static_cast<PixelSpace>(w);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::SpriteFrame::GetPixelHeight(void) const
{
	return static_cast<PixelSpace>(h);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelTop(void) const
{
	return uvy;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelLeft(void) const
{
	return uvx;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelRight(void) const
{
	return uvx + uvw;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelBottom(void) const
{
	return uvy + uvh;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelWidth(void) const
{
	return uvw;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::TexelSpace tbGraphics::SpriteFrame::GetTexelHeight(void) const
{
	return uvh;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::Sprite(const SpriteFrame& spriteFrame) :
	mSpriteFrame(spriteFrame)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::Sprite(const tbCore::tbString& textureFile, const PixelSpace& frameX, const PixelSpace& frameY,
	const PixelSpace& frameWidth, const PixelSpace& frameHeight) :
	mSpriteFrame(SpriteFrame::CreateWith(theTextureManager.CreateTextureFromFile(textureFile), frameX, frameY, frameWidth, frameHeight))
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::Sprite(const TextureHandle& textureHandle, const PixelSpace& frameX, const PixelSpace& frameY,
	const PixelSpace& frameWidth, const PixelSpace& frameHeight) :
	mSpriteFrame(SpriteFrame::CreateWith(textureHandle, frameX, frameY, frameWidth, frameHeight))
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::Sprite(const tbCore::tbString& spriteSheet, const tbCore::tbString& spriteName) :
	mSpriteFrame(theSpriteManager.GetSpriteFrame(spriteSheet, spriteName))
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::Sprite(const Sprite& other) :
	mSpriteFrame(other.mSpriteFrame)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite::~Sprite(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Sprite& tbGraphics::Sprite::operator=(const Sprite& other)
{
	if (&other != this)
	{
		mSpriteFrame = other.mSpriteFrame;
	}
	return *this;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::Sprite::GetPixelWidth(void) const
{
	return mSpriteFrame.GetPixelWidth();
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::Sprite::GetPixelHeight(void) const
{
	return mSpriteFrame.GetPixelHeight();
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::Sprite::OnRender(void) const
{
	theTextureManager.BindTexture(mSpriteFrame.mTexture);

	tbMath::Vector2 topLeft(0.0f, 0.0f);
	tbMath::Vector2 bottomRight(mSpriteFrame.GetPixelWidth(), mSpriteFrame.GetPixelHeight());

	tbImplementation::Renderer::Vertex2D vertices[] =
	{
		{ topLeft.x, bottomRight.y,      GetColor().GetColorABGR(),   mSpriteFrame.GetTexelLeft(), mSpriteFrame.GetTexelBottom(),    },
		{ topLeft.x, topLeft.y,          GetColor().GetColorABGR(),   mSpriteFrame.GetTexelLeft(), mSpriteFrame.GetTexelTop(),    },
		{ bottomRight.x, bottomRight.y,  GetColor().GetColorABGR(),   mSpriteFrame.GetTexelRight(), mSpriteFrame.GetTexelBottom(),    },
		{ bottomRight.x, topLeft.y,      GetColor().GetColorABGR(),   mSpriteFrame.GetTexelRight(), mSpriteFrame.GetTexelTop(),    },
	};

	tbImplementation::Renderer::Render(tbImplementation::Renderer::kTriangleStrip, vertices, 4);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::Sprite::SetSpriteFrame(const SpriteFrame& spriteFrame)
{
	tb_error_if(spriteFrame.mTexture != mSpriteFrame.mTexture, "tbError: Expected the new sprite frame to have the same texture handle.");
	mSpriteFrame = spriteFrame;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbGraphics::SpriteFrame& tbGraphics::Sprite::GetSpriteFrame(void) const
{
	return mSpriteFrame;
}

//--------------------------------------------------------------------------------------------------------------------//
