///
/// @file
/// @details This is the start of the sprite manager that will eventually become the TurtleBrains sprite manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_SpriteManager_h_
#define _TurtleBrains_SpriteManager_h_

#include "tb_sprite.h"
#include "tb_animated_sprite.h"
#include "tb_sprite_sheet.h"
#include "../core/tb_noncopyable.h"
#include "../core/tb_string.h"

#include <map>

namespace TurtleBrains
{
	namespace Core { class DynamicStructure; }

	namespace Graphics
	{

		///
		///	@details The SpriteManager holds each of the SpriteSheets and all the Sprites and Animations that have been
		///   loaded into the manager.  It provides an easy method of create a Sprite or AnimatedSprite object from data.
		///
		class SpriteManager : public tbCore::Noncopyable
		{
		public:
			///
			/// @details The SpriteManager should be treated like a singleton object and should only ever be accessed through
			///   the tbGraphics::theSpriteManager 'global' variable.
			///
			SpriteManager(void);

			///
			///	@details The SpriteManager should be treated like a singleton object and should only ever be accessed through
			///   the tbGraphics::theSpriteManager 'global' variable.
			///
			~SpriteManager(void);

			///
			///	@details Loads a json formatted file and processes it into a DynamicStructure which is then processed with
			///   LoadSpriteSheetFromData().  If the file was not found or loaded an error condition is triggered.  If the
			///   json data did not contain all required information about a sprite or animation sequence an error condition
			///   will be triggered from within LoadSpriteSheetFromData().
			///
			///	@param spriteSheetName The name of the SpriteSheet where all the sprite and animation data will be stored.
			/// @param spriteSheetFile The path to the json formatted file that contains the sprites and animation data to load.
			///
			/// Ex. SpriteSheet:
			///
			/// { "texture" : "pathToTexture",
			///   "sprites" : [
			///     { "name" : "frame_name", "x" : 0, "y" : 0, "width" : 128, "height" : 128, "animations" : [ "idle", "jump", "death" ] },
			///   ],
			///   "animations" : [
			///     { "name" : "sequence_name", "frames" : [ "sprite_frame_1", "sprite_frame_2" ... ] },
			///     { "name" : "sequence_mapp", "frames" :
			///       { "startIndex" : 0, "frameCount" : 3, "width" : 32, "height" : 32, "locationX" : 0, "locationY" : 0 } },
			///     { "name" : "sequence_mapc", "frames" :
			///       { "width" : 32, "height" : 32, "locationX" : 0, "locationY" : 0, "frames" : [ 0, 1, 5, 3 ] } }
			///   ]
			/// }
			///
			///
			bool LoadSpriteSheetFromFile(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteSheetFile);

			///
			/// @details Finds the sprite sheet previously loaded with spriteSheetName and returns the associated texture handle.
			///
			/// @param spriteSheetName The name of the SpriteSheet to retrieve the texture handle of.
			///
			/// @note If no SpriteSheet with the name spriteSheetName is found, and error condition will be triggered.
			///
			TextureHandle GetSpriteSheetTextureHandle(const tbCore::tbString& spriteSheetName) const;

			///
			///	@details Creates just the SpriteFrame  data provided on the SpriteSheet for the given spriteName.  If no
			///   SpriteSheet is found with the spriteSheetName, or if no sprite data is found with the spriteName an error
			///   condition will be triggered.
			///
			/// @param spriteSheetName A name to a SpriteSheet as loaded during a call to LoadSpriteSheetFromFile() or
			///   LoadSpriteSheetFromData().  If no object is found with the sheet name, an error condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded.  If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			///
			SpriteFrame GetSpriteFrame(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const;

			///
			///	@details By default this will creates a sprite that is the entire width and height of the texture loaded from
			///   the textureFile provided.  If the file is unable to load from disk an error condition will be triggered.
			///   The remaining parameters are optional to setup a frame of specified location and size.
			///
			/// @param textureFile The filepath of the image to load using theTextureManager.  Will use an already loaded
			///   texture if it has been loaded previously, otherwise loads the file (or triggers an error condition if the
			///   texture file cannot be loaded.
			/// @param frameX An optional parameter to set the horizontal location of the frame on the texture, in pixels.
			/// @param frameY An optional parameter to set the vertical location of the frame on the texture, in pixels.
			/// @param frameWidth An option parameter to set the width in pixels of the sprites frame.
			/// @param frameHeight An option parameter to set the height in pixels of the sprites frame.
			///
			/// @note This can trigger an error condition if the texture cannot be loaded.
			/// @note If any of the optional frame parameters are used, all must be set.
			/// @note This may be deprecated in favor of the Sprite() ctor version.
			///
			Sprite GetSpriteFromFile(const tbCore::tbString& textureFile, const PixelSpace& frameX = 0, const PixelSpace& frameY = 0,
				const PixelSpace& frameWidth = 0, const PixelSpace& frameHeight = 0) const;
			
			///
			///	@details Creates a Sprite object from the data provided on the SpriteSheet for the given spriteName.  If no
			///   SpriteSheet is found with the spriteSheetName, or if no sprite data is found with the spriteName an error
			///   condition will be triggered.
			///
			/// @param spriteSheetName A name to a SpriteSheet as loaded during a call to LoadSpriteSheetFromFile() or
			///   LoadSpriteSheetFromData().  If no object is found with the sheet name, an error condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded.  If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			///
			Sprite GetSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const;

			///
			///	@details Behaves identical to GetSprite() except an AnimatedSprite object is returned with AnimationSequences
			///   added to the AnimatedSprite, ready for action.  An error condition is triggered if the sprite sheet is not
			///   found or the sprite data is not found.  This can be called even if the sprite data does not contain any
			///   animation sequences, but as it is expected the sequences will be added upon initialization.
			///
			/// @param spriteSheetName A name to a SpriteSheet as loaded during a call to LoadSpriteSheetFromFile() or
			///   LoadSpriteSheetFromData().  If no object is found with the sheet name, an error condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded.  If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			///
			AnimatedSprite GetAnimatedSprite(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName) const;

			///
			/// @details Adds all the AnimationSequences found on the spriteName to the animatedSprite provided.
			///
			/// @param spriteSheetName A name to a SpriteSheet as loaded during a call to LoadSpriteSheetFromFile() or
			///   LoadSpriteSheetFromData(). If no object is found with the sheet name, an error condition will be triggered.
			/// @param spriteName A name to the sprite data on the specified SpriteSheet which was loaded. If no sprite data
			///   is found with the provided spriteName and error condition will be triggered.
			/// @param animatedSprite The AnimatedSprite object to add the sequences to so they can be played back.
			///
			void AddAnimationSequences(const tbCore::tbString& spriteSheetName, const tbCore::tbString& spriteName, AnimatedSprite& animatedSprite) const;

		private:
			void AddAnimationSequences(const SpriteSheet& spriteSheet, const tbCore::tbString& spriteName, AnimatedSprite& animatedSprite) const;

			typedef std::map<tbCore::tbString, SpriteSheet> SpriteSheetContainer;
			SpriteSheetContainer mSpriteSheets;
		};

		///
		/// @details Instead of creating your own SpriteManager object, this is a way to access it as a singleton.  This
		///   _may_ turn into a pointer to the object, although the SpriteManager api will remain the same.
		///
		extern SpriteManager theSpriteManager;

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_SpriteManager_h_ */
