///
/// @file
/// @details The Text object provides an easy, minimal, interface to creating and visualizing Text graphics in a game 
///   scene.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// Credits: Implementation makes heavy use of stb_trutype.h (v1.02) by Sean Barrett which is in Public Domain
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_text.h"

#define STB_TRUETYPE_IMPLEMENTATION
#include "implementation/stb_truetype.h"
#include "implementation/tbi_default_font.h"
#include "implementation/tbi_renderer.h"

#include "../core/tb_opengl.h"
#include "../math/tb_math.h"

#include <fstream>
#include <vector>
#include <map>
#include <cassert>
#include <cmath>
#include <string>

namespace tbImplementation
{
	
	struct TextData
	{
		unsigned int mTextureIndex;
		tbGraphics::PixelSpace mImageWidth;
		tbGraphics::PixelSpace mImageHeight;
	};

	///
	/// @details Font is heavily inspired, and/or taken from a C-Wrapper for stb_truetype that rxi shared and is a
	///   TurtleBrains implementation detail.
	///
	class Font : public TurtleBrains::Core::Noncopyable
	{
	public:

		static Font* CreateFontFromFile(const tbCore::tbString& fontFilepath);
		static Font* CreateFontFromData(const unsigned char* fontData, const size_t& size);
		static void DestroyFont(Font* font);

		///
		///
		///
		void SetPointSize(const float pointSize);

		///
		///
		///
		tbGraphics::PixelSpace GetTextHeight(void) const;

		///
		///
		///
		tbGraphics::PixelSpace GetTextWidth(const std::string& text) const;

		///
		/// @details This is the function that does the heavy lifting and creates the data necessary for the TextObject
		///   to create an OpenGL texture and render as a quad.  Changing the fonts pointSize will change the size of
		///   the texture that gets created!
		///
		unsigned char* CreateTextImage(const std::string& text, tbGraphics::PixelSpace& width, tbGraphics::PixelSpace& height) const;

	private:
		Font(void);
		~Font(void);

		float GetCharacterWidth(int character, int lastCharacter) const;

		stbtt_fontinfo mFontInfo;
		char* mFontData;
		float mPointSize;
		float mScale;
		int mBaseLine;
	};

	///
	/// @details Loads fonts as they are created.
	///
	class FontManager : public TurtleBrains::Core::Noncopyable
	{
	public:
		//void LoadFont(const tbCore::tbString& fontFilepath);

		static FontManager& GetInstance(void);
		Font& GetFont(const tbCore::tbString& fontFilepath);

	private:
		FontManager(void);
		~FontManager(void);
	};

	tbImplementation::FontManager* theFontManager(nullptr);
	typedef std::map<tbCore::tbString, tbImplementation::Font*> FontContainer;
	FontContainer theFontCache;

}; /* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

const tbCore::tbString tbGraphics::Text::kDefaultFontFile(tb_string(""));
const float tbGraphics::Text::kDefaultFontSize(25.0f);

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Text::Text(const tbCore::tbString& text, const float pointSize, const tbCore::tbString& fontFilepath) :
	mTextData(nullptr),
	mTextFont(fontFilepath),
	mTextSize(pointSize)
{
	SetText(text, pointSize, fontFilepath);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::Text::~Text(void)
{
	if (nullptr != mTextData)
	{

		if (true == tbImplementation::Renderer::tbiIsRendererAvailable)
		{
			//tb_log("tbInternalWarning: TODO: TIM: Skipping the unloading of text, need refactor.");
			tb_check_gl_errors(glDeleteTextures(1, &mTextData->mTextureIndex));
		}
		
		delete mTextData;
		mTextData = nullptr;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::Text::OnRender(void) const
{
	if (nullptr == mTextData)
	{
		return;
	}

	const float imageWidth(GetWidth());
	const float imageHeight(GetHeight());

	tb_check_gl_errors(glBindTexture(GL_TEXTURE_2D, mTextData->mTextureIndex));
	tbImplementation::Renderer::SetColorMatrix(true);
	tbImplementation::Renderer::RenderSprite(tbMath::Vector2::kZero, tbMath::Vector2(imageWidth, imageHeight), GetColor().GetColorABGR());
	tbImplementation::Renderer::SetColorMatrix(false);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::Text::SetText(const tbCore::tbString& text, const float pointSize, const tbCore::tbString& fontFilepath)
{
	if (nullptr != mTextData)
	{	//First clean up the old text object if it exists.
		tb_check_gl_errors(glDeleteTextures(1, &mTextData->mTextureIndex));
		delete mTextData;
		mTextData = nullptr;
	}

	if (false == fontFilepath.empty())
	{	//New text font is not default and could be changed from the old font.
		mTextFont = fontFilepath;
	}

	if (pointSize > 0.0f)
	{	//New text size is not default and could be changed from the old size.
		mTextSize = pointSize;
	}

	mTextString = text;

	if (true == text.empty())
	{	//No need to create an image or anything, no text!
		return;
	}
	
	//
	// Finally create the new text object, load font if needed etc.
	mTextData = new tbImplementation::TextData();
	tb_error_if(nullptr == mTextData, "tbError: Failed to allocate memory for TextObject data.");

	tbImplementation::FontManager& fontManager(tbImplementation::FontManager::GetInstance());
	tbImplementation::Font& font(fontManager.GetFont(mTextFont));
	font.SetPointSize(mTextSize);

	unsigned char* imageData = font.CreateTextImage(tbCore::ToStdString(text), mTextData->mImageWidth, mTextData->mImageHeight);
	tb_error_if(nullptr == imageData, "tbError: Failed to create image for TextObject.");

	tb_check_gl_errors(glGenTextures(1, &mTextData->mTextureIndex));
	tb_check_gl_errors(glBindTexture(GL_TEXTURE_2D, mTextData->mTextureIndex));

	//Technically we don't want this seen, so we use bright pink.
	float borderColor[] = { 1.0f, 0.0f, 1.0f, 1.0f };
	tb_check_gl_errors(glTexParameterfv(GL_TEXTURE_2D, GL_TEXTURE_BORDER_COLOR, borderColor));
	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_LINEAR));
	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_LINEAR));
	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT));
	tb_check_gl_errors(glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT));

#if defined(tb_legacy_gl_support)
	if (true == tbImplementation::Renderer::IsLegacyRenderer())
	{
		tb_check_gl_errors(glPixelStorei(GL_UNPACK_ALIGNMENT, 1));
		tb_check_gl_errors(glTexImage2D(GL_TEXTURE_2D, 0, GL_ALPHA, mTextData->mImageWidth, mTextData->mImageHeight, 0, GL_ALPHA, GL_UNSIGNED_BYTE, imageData));
		tb_check_gl_errors(glPixelStorei(GL_UNPACK_ALIGNMENT, 4));	//Set back to default value.
	}
	else
#endif /* tb_legacy_gl_support */
	{
		tb_check_gl_errors(glPixelStorei(GL_UNPACK_ALIGNMENT, 1));
		tb_check_gl_errors(glTexImage2D(GL_TEXTURE_2D, 0, GL_RED, mTextData->mImageWidth, mTextData->mImageHeight, 0, GL_RED, GL_UNSIGNED_BYTE, imageData));
		tb_check_gl_errors(glPixelStorei(GL_UNPACK_ALIGNMENT, 4));	//Set back to default value.
	}

	delete [] imageData;
	imageData = nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbGraphics::Text::SetText(const tbCore::tbString& text)
{
	SetText(text, mTextSize, mTextFont);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::Text::GetPixelWidth(void) const
{
	return (nullptr == mTextData) ? 0 : mTextData->mImageWidth;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbGraphics::Text::GetPixelHeight(void) const
{
	return (nullptr == mTextData) ? 0 : mTextData->mImageHeight;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::FontManager::FontManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::FontManager::~FontManager(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::FontManager& tbImplementation::FontManager::GetInstance(void)
{
	if (nullptr == theFontManager)
	{
		theFontManager = new FontManager();
		theFontCache[""] = Font::CreateFontFromData(tbImplementation::theDefaultFontSourceData, tbImplementation::theDefaultFontSourceSize);
	}
	return *theFontManager;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::Font& tbImplementation::FontManager::GetFont(const tbCore::tbString& fontFilepath)
{
	FontContainer::iterator fontIterator(theFontCache.find(fontFilepath));
	if (fontIterator == theFontCache.end())
	{
		Font* createdFont(Font::CreateFontFromFile(fontFilepath));
		tb_error_if(nullptr == createdFont, "tbExternalError: Was unable to load font from file: \"%s\"", fontFilepath.c_str());
//		theFontCache->insert(FontContainer::value_type(fontFilepath, createdFont));
//		return (theFontCache.insert(FontContainer::value_type(fontFilepath, createdFont)).first)->second;
		theFontCache[fontFilepath] = createdFont;
		return *theFontCache[fontFilepath];
	}
	return *fontIterator->second;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::Font::Font(void) :
	mFontData(nullptr),
	mPointSize(25.0f)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::Font::~Font(void)
{
	if (nullptr != mFontData)
	{
		delete [] mFontData;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::Font* tbImplementation::Font::CreateFontFromFile(const tbCore::tbString &fontFilepath)
{
	Font* newFont = new Font();
	if (nullptr == newFont)
	{
		return nullptr;
	}

	std::ifstream inputFile(tbCore::ToStdString(fontFilepath).c_str(), std::ios_base::binary);
	if (false == inputFile.is_open())
	{
		delete newFont;
		return nullptr;
	}

	inputFile.seekg(0, std::ios_base::end);
	size_t fileLength = static_cast<size_t>(inputFile.tellg());
	inputFile.seekg(0, std::ios_base::beg);

	newFont->mFontData = new char[fileLength];
	if (nullptr == newFont->mFontData)
	{
		delete newFont;
		return nullptr;
	}

	inputFile.read(newFont->mFontData, fileLength);
	
	if (!stbtt_InitFont(&newFont->mFontInfo, reinterpret_cast<unsigned char*>(newFont->mFontData), 0))
	{
		delete newFont;
		return nullptr;
	}

	newFont->SetPointSize(25.0f);

	return newFont;
}

//--------------------------------------------------------------------------------------------------------------------//

tbImplementation::Font* tbImplementation::Font::CreateFontFromData(const unsigned char* fontData, const size_t& size)
{
	Font* newFont = new Font();
	if (nullptr == newFont)
	{
		return nullptr;
	}

	newFont->mFontData = new char[size];
	if (nullptr == newFont->mFontData)
	{
		delete newFont;
		return nullptr;
	}

	memcpy((void*)newFont->mFontData, (const void*)fontData, size);

	if (!stbtt_InitFont(&newFont->mFontInfo, reinterpret_cast<unsigned char*>(newFont->mFontData), 0))
	{
		delete newFont;
		return nullptr;
	}

	newFont->SetPointSize(25.0f);

	return newFont;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Font::DestroyFont(tbImplementation::Font* font)
{
	if (nullptr != font)
	{
		delete font;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::Font::SetPointSize(const float pointSize)
{
	int ascent(0);
	int descent(0);
	int lineGap(0);

	mPointSize = pointSize;
	mScale = stbtt_ScaleForMappingEmToPixels(&mFontInfo, mPointSize);
	stbtt_GetFontVMetrics(&mFontInfo, &ascent, &descent, &lineGap);
	mBaseLine = static_cast<int>(ascent * mScale + 1);
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbImplementation::Font::GetTextHeight(void) const
{
	int ascent(0);
	int descent(0);
	int lineGap(0);

	stbtt_GetFontVMetrics(&mFontInfo, &ascent, &descent, &lineGap);
	const int textHeight = static_cast<int>(ceil((ascent - descent + lineGap) * mScale)) + 1;
	tb_error_if(textHeight < 0 || textHeight > 0xFFFF, "tbInternalError: Text height is out of range of PixelSpace value, cast below.");
	return static_cast<tbGraphics::PixelSpace>(textHeight);
}

//--------------------------------------------------------------------------------------------------------------------//

static const char* ttf_utf8toCodepoint(const char* p, unsigned* result)
{
	unsigned x(0);
	unsigned mask(0);
	unsigned shift(0);

	switch (*p & 0xF0)
	{
		case 0xF0: mask = 0x07; shift = 18; break;
		case 0xE0: mask = 0x0F; shift = 12; break;
		case 0xC0: /* fall through to 0xD0 case */
		case 0xD0: mask = 0x1F; shift = 6; break;
		default: *result = *p; return p + 1;
	};

	x = (*p & mask) << shift;
	do {
		/* return early if we reach an unexpected NULL */
		if (*(++p) == '\0')
		{
			*result = x;
			return p;
		}
		shift -= 6;
		x |= (*p & 0x3F) << shift;
	} while (shift);

	*result = x;
	return p + 1;
}

//--------------------------------------------------------------------------------------------------------------------//

tbGraphics::PixelSpace tbImplementation::Font::GetTextWidth(const std::string& text) const
{
	float result(0.0f);
	int lastCharacter(0);
	const char* p = text.c_str();
	while (*p)
	{
		unsigned character;
		p = ttf_utf8toCodepoint(p, &character);
		result += GetCharacterWidth(character, lastCharacter);
		lastCharacter = character;
	}

	const int textWidth = static_cast<int>(ceil(result));
	tb_error_if(textWidth < 0 || textWidth > 0xFFFF, "tbInternalError: Text width is out of range of PixelSpace value, cast below.");
	return static_cast<tbGraphics::PixelSpace>(textWidth);
}

//--------------------------------------------------------------------------------------------------------------------//

unsigned char* tbImplementation::Font::CreateTextImage(const std::string& text, tbGraphics::PixelSpace& width, tbGraphics::PixelSpace& height) const
{
	width = GetTextWidth(text);
	height = GetTextHeight();

	unsigned char* pixels = new unsigned char[width * height];
	if (nullptr == pixels)
	{
		return nullptr;
	}

	memset(pixels, 0, width * height);

	const char *p = text.c_str();
	float xoffset(0.0f);
	float xfract(0.0f);
	int lastCharacter(0);

	while (*p)
	{
		unsigned character;
		p = ttf_utf8toCodepoint(p, &character);

		int x0, y0, x1, y1;
		stbtt_GetCodepointBitmapBoxSubpixel(&mFontInfo, character, mScale, mScale, xfract, 0, &x0, &y0, &x1, &y1);

		/* work out position / max size */
		int x = static_cast<int>(xoffset) + x0;
		int y = mBaseLine + y0;
		if (x < 0) { x = 0; }
		if (y < 0) { y = 0; }

		/* Render Character */
		stbtt_MakeCodepointBitmapSubpixel(&mFontInfo, &pixels[x + (y * width)], width - x, height - y,
																			width, mScale, mScale, xfract, 0, character);

		/* Next */
		xoffset += GetCharacterWidth(character, lastCharacter);
		xfract = xoffset - static_cast<int>(xoffset);
		lastCharacter = character;
	}

	return pixels;
}

//--------------------------------------------------------------------------------------------------------------------//

float tbImplementation::Font::GetCharacterWidth(int character, int lastCharacter) const
{
	int result(0);
	int width(0);
	int leftSideBearing(0);

	stbtt_GetCodepointHMetrics(&mFontInfo, character, &width, &leftSideBearing);
	result = width;
	if (lastCharacter)
	{
		int kerning = stbtt_GetCodepointKernAdvance(&mFontInfo, lastCharacter, character);
		result += kerning;
	}
	return result * mScale;
}

//--------------------------------------------------------------------------------------------------------------------//
