///
/// @file
/// @details The Text object provides an easy, minimal, interface to creating and visualizing Text graphics in a game 
///   scene.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// Credits: Implementation makes heavy use of stb_trutype.h (v1.02) by Sean Barrett which is in Public Domain
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Text_h_
#define _TurtleBrains_Text_h_

#include "tb_graphic.h"
#include "tb_color.h"
#include "tb_texture_manager.h" //for PixelSpace definition.
#include "../core/tb_noncopyable.h"
#include "../core/tb_string.h"

namespace tbImplementation
{
	struct TextData;
};

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		/// @details The basic TextObject provides a minimalistic, but powerful true-type font api, each instance of the
		///   TextObject creates a GL texture that will be rendered to the screen as appropriate.  It does take a little
		///   overhead to create the TextObject, so don't create a bunch of them over and over each frame.
		///
		/// @note the Constructor seems to take a lot of parameters, but the primary use case in mind with the particular
		///   design is for the TextObject to be derived and changed so only the first parameter, text, is actually used.
		///
		///  class MySpecialTextObject : public TextObject {
		///    public: MySpecialText(const tbString& text) : TextObject(text, "Path/To/Some/Font.ttf", 20.0f) { }
		///  }
		///
		class Text : public Graphic, public tbCore::Noncopyable
		{
		public:
			///
			/// @details the default font size for a Text object created without specifying a size.
			///
			static const float kDefaultFontSize;

			///
			/// @details The default font file automatically created by TurtleBrains if a Text object is created with default.
			///
			static const tbCore::tbString kDefaultFontFile;

			///
			/// @details This constructor does all the heavy lifting for the TextObject.  If the font specified by 
			///   fontFilepath is not loaded, it will be loaded, and then a small GL texture will be rasterized and 
			///   stored to be used by the Draw() method.  There are several ways the constructor can fail, and if it
			///   does fail the tb_error_if behavior will be invoked, either throwing exceptions, asserts, or other.
			///
			/// @param text          A string representing the actual letters/symbols to be displayed by the object.
			/// @param fontFilepath  The filepath to the truetype-font to use for this text object, file must exist or 
			///   an error condition will be triggered.  Set as kDefaultFontFile, the font will be the default font as
			///   supplied by TurtleBrains.
			/// @param pointSize     How large this TextObject should be rasterized at.
			///
			explicit Text(const tbCore::tbString& text = tb_string(""), const float pointSize = kDefaultFontSize, const tbCore::tbString& fontFilepath = kDefaultFontFile);

			///
			/// @details Cleans up any resources used by the Text object, with the current exception that a Font will 
			///   remain loaded for other Text objects to use.  This will likely change in the future to unload Fonts,
			///   except for the default font, that have no Text objects referencing.
			///
			virtual ~Text(void);

			///
			///	@details SetText is the heavy lifter of the object, it actually creates the small GL texture for of the
			///   rasterized letters/symbols from the true-type font and prepares everything to be displayed.  It can be
			///   a fairly expensive call, so avoid changing the text, size or font when possible.
			///
			/// @param text the string of letters to display when rendered.
			/// @param pointSize is optional to change the size of the text should when rasterized.  This will effect the
			///   width and height of the displayed text.
			/// @param fontFilepath is optional to change the font of the text object.
			///
			/// @note Rasterizing the text is the heavy lifter of the Text object and should only be called when needed.
			///
			void SetText(const tbCore::tbString& text, const float pointSize, const tbCore::tbString& fontFilepath);

			///
			///	@details SetText is the heavy lifter of the object, it actually creates the small GL texture for of the
			///   rasterized letters/symbols from the true-type font and prepares everything to be displayed.  It can be
			///   a fairly expensive call, so avoid changing the text, size or font when possible.
			///
			/// @param text the string of letters to display when rendered.
			///
			/// @note Rasterizing the text is the heavy lifter of the Text object and should only be called when needed.
			///
			void SetText(const tbCore::tbString& text);

			///
			///	@details Returns the width of the text in PixelSpace which may include some padding around the edges.
			///
			virtual PixelSpace GetPixelWidth(void) const override;

			///
			///	@details Returns the height of the text in PixelSpace which may include some padding around the edges.
			///
			virtual PixelSpace GetPixelHeight(void) const override;

		protected:
			///
			/// @details Draws the precomputed image from the constructor to the screen.  The location of the TextObject is
			///   determined by the last call to SetPosition(), as is the color with SetColor().
			///
			virtual void OnRender(void) const override;
			
		private:
			tbImplementation::TextData* mTextData;
			tbCore::tbString mTextFont;
			tbCore::tbString mTextString;
			float mTextSize;
		};

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBrains_Text_h_ */
