///
/// @file
/// @details This is the start of the texture manager that will eventually become the TurtleBrains texture manager.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// Credits: Implementation makes heavy use of stb_image.h (v1.33) by Sean Barrett which is in Public Domain
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBraines_TextureManager_h_
#define _TurtleBraines_TextureManager_h_

#include "../core/tb_noncopyable.h"
#include "../core/tb_string.h"
#include "../core/tb_types.h"

#include <cstddef>

namespace TurtleBrains
{
	namespace Graphics
	{

		///
		/// @details Represents a value from 0 to 1 in texture space for the width/height or location on a texture.
		///
		typedef float TexelSpace;

		///
		/// @details Represents a measurement in pixel space be it the width/height of an image in pixels, or the location
		///   on the screen or texture in pixels.
		///
		typedef tbCore::uint16 PixelSpace;

		///
		///	@details A TextureHandle is a unique value for each unique texture created with the TextureManager.  Use the
		///   handle to access the textures properties and to destroy any resources associated.
		///
		/// TODO: TIM: Implementation: Maybe this should be a tiny class/object that automatically cleans up resources?
		///
		typedef unsigned int TextureHandle;

		///
		/// @details This is a constant value that represents a null/invalid texture handle.
		///
		extern const TextureHandle kInvalidTexture;

		///
		/// @details This is a small, white texture that is automatically loaded for rendering Lines without a separate shader.
		///
		extern const TextureHandle& kBlankTexture;

		///
		/// @details A basic texture manager to load images from disk or file/pixel data in memory while using a caching
		///   technique so using the same filepath will not create more textures.  TurtleBrains makes use of Sean Barrett's
		///		stb_image api to process the file, which supports the following file formats with some limitations:
		///
		///   - JPEG baseline (no JPEG progressive)
		///		- PNG 8-bit-per-channel only
		///		-	TGA (not sure what subset, if a subset)
		///		-	BMP non-1bpp, non-RLE
		///		-	PSD (composited view only, no extra channels)
		///		-	GIF (*comp always reports as 4-channel)
		///		-	HDR (radiance rgbE format)
		///		-	PIC (Softimage PIC)
		///
		class TextureManager : public tbCore::Noncopyable
		{
		public:
			///
			///	@details The TextureManager should be treated like a singleton object and should only ever be accessed through
			///   the tbGraphics::theTextureManager 'global' variable.
			///
			TextureManager(void);

			///
			///	@details The TextureManager should be treated like a singleton object and should only ever be accessed through
			///   the tbGraphics::theTextureManager 'global' variable.
			///
			~TextureManager(void);

			///
			/// @details Checks the cache if the file has already been opened, if not, the file will be opened from disk and
			///   processed creating an ARGB texture with OpenGL.  Any texture created must be cleaned up by calling
			///   DestroyTexture.  Although the texture may not actually be destroyed if there are still others referencing
			///		the texture object (via other calls to CreateTextureFromFile).
			///
			/// @param filename the path of the file to open and process.
			///
			TextureHandle CreateTextureFromFile(const tbCore::tbString& filename);

			///
			/// @details Creates a texture from the data of a file already in memory, this is not the uncompressed pixel data.
			///   Textures created from file data do not get cached, so be aware that calling this several times with the same
			///   file data will create new textures.  Any texture created must be cleaned up by calling DestroyTexture().
			///
			///	@param fileDataInMemory should contain an entire supported file in memory which can be removed after the call.
			/// @param fileSizeInBytes is how many bytes make up the file.
			///
			/// @note to create multiple references to the texture created by file data you can use CreateTextureReference().
			///
			TextureHandle CreateTextureFromFileData(const unsigned char* fileDataInMemory, const size_t& fileSizeInBytes);

			///
			/// @details Creates a texture from the data of each pixel color, this is the uncompressed 32bit ARGB pixel data
			///   format.  A 1x1 texture would be 4 bytes, first byte Alpha, second Red, then Green and final byte Blue.
			///   Textures created from pixel data do not get cached, so be aware that calling this several times with the same
			///   pixel data will create new textures.  Any texture created must be cleaned up by calling DestroyTexture().
			///
			///	@param pixelData must contain at least (textureWidth * textureHeight * 4) bytes of data for each byte.
			/// @param textureWidth is the width of the supplied pixelData in pixels wide, must be greater than 0.
			/// @param textureHeight is the height of the supplied pixelData in pixels tall, must be greater than 0.
			///
			/// @note to create multiple references to the texture created by pixel data you can use CreateTextureReference().
			///
			TextureHandle CreateTextureFromPixelData(const unsigned char* pixelData, const PixelSpace& textureWidth, const PixelSpace& textureHeight);

			///
			/// @details This does not actually create a texture but will increase the reference count so it does not get
			///   unloaded/cleaned on the call to DestroyTexture.  This is meant to allow textured created by file data or
			///   pixel data to be shared more easily in the reference counted manner.  Any texture created must be cleaned up
			///   by calling DestroyTexture().
			///
			///	@param textureHandle The texture associated with this handle will increment the reference count and be
			///   returned back.  Must be a valid TextureHandle or an error condition will be triggered.
			///
			TextureHandle CreateTextureReference(const TextureHandle& textureHandle);

			///
			///	@details Decrements the textures reference count and if needed will destroys the texture, removing it from
			///   the OpenGL pipeline and cleaning any other associated resources. Each texture handle is reference counted,
			///   so every call to Create needs a matching call to destroy.
			///
			/// @param textureHandle of the texture to destroy and remove completely from the GL context and TextureManager.
			///
			void DestroyTexture(const TextureHandle& textureHandle);

			///
			///	@details Binds the texture associated with the textureHandle to the GL context for rendering calls.  An error
			///   condition will be triggered if textureHandle is the handle was not initialized or the associated texture has
			///   been destroyed.
			///
			/// @param textureHandle of the texture to apply to the GL contect.
			///
			void BindTexture(const TextureHandle& textureHandle);

			///
			///	@details Returns the width of the texture asscioued with the textureHandle provided.  If kInvalidTexture is
			///   provided 0 will be returned.  An error condition will be triggered if textureHandle is the handle was not
			///   initialized or the associated texture has been destroyed.
			///
			PixelSpace GetTextureWidth(const TextureHandle& textureHandle) const;

			///
			///	@details Returns the width of the texture asscioued with the textureHandle provided.  If kInvalidTexture is
			///   provided 0 will be returned.  An error condition will be triggered if textureHandle is the handle was not
			///   initialized or the associated texture has been destroyed.
			///
			PixelSpace GetTextureHeight(const TextureHandle& textureHandle) const;

		private:
			TextureHandle mBoundTexture;
		};

		///
		/// @details Instead of creating your own TextureManager object, this is a way to access it as a singleton.  This
		///   _may_ turn into a pointer to the object, although the TextureManager api will remain the same.
		///
		extern TextureManager theTextureManager;

	}; /* namespace Graphics */
}; /* namespace TurtleBrains */

namespace tbGraphics = TurtleBrains::Graphics;

#endif /* _TurtleBraines_TextureManager_h_ */
