///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Math_h_
#define _TurtleBrains_Math_h_

#include "tb_constants.h"
#include <cmath>

namespace TurtleBrains
{
	namespace Math
	{
		///
		///	@details Compares two floating point values returning true if the difference between the two values is less-than
		///   or equal to the tolerance allowed.
		///
		/// @param leftValue The value that would be on the left side of the comparison operator.
		/// @param rightValue The value that would be on the right side of the comparison operator.
		/// @param tolerance An optional value to change the tolerance of the comparison, smaller value means the floats
		///   need to be closer together, defaults to kTolerance.
		///
		inline bool IsEqual(const float leftValue, const float rightValue, const float tolerance = tbMath::kTolerance)
		{
			return fabs(leftValue - rightValue) <= tolerance;
		}

		///
		///	@details Compares a floating point values returning true if the difference between the value and zero is less-than
		///   or equal to the tolerance allowed.
		///
		/// @param value The value to check if it is within the tolerance range of zero.
		/// @param tolerance An optional value to change the tolerance of the comparison, smaller value means the floats
		///   need to be closer together, defaults to kTolerance.
		///
		inline bool IsZero(const float value, const float tolerance = tbMath::kTolerance)
		{
			return (fabs(value)) <= tolerance;
		}

		///
		///	@details Compares the value of two similar types using the less-than operator returning the greater of the values.
		///
		/// @param leftValue The value that would be on the left side of the comparison operator.
		/// @param rightValue The value that would be on the right side of the comparison operator.
		///
		/// @note There will be compile issues if called on a type that does not have an operator< to use.
		///
		template <typename T> const T& Maximum(const T& leftValue, const T& rightValue)
		{
			return (leftValue < rightValue) ? rightValue : leftValue;
		}

		///
		///	@details Compares the value of two similar types using the less-than operator returning the lesser of the values.
		///
		/// @param leftValue The value that would be on the left side of the comparison operator.
		/// @param rightValue The value that would be on the right side of the comparison operator.
		///
		/// @note There will be compile issues if called on a type that does not have an operator< to use.
		///
		template <typename T> const T& Minimum(const T& leftValue, const T& rightValue)
		{
			return (leftValue < rightValue) ? leftValue : rightValue;
		}

		///
		///	@details Returns a value that falls within the given range of minimumValue to maximumValue using only the
		///   less-than operator for the type.
		///
		/// @param value The value that will be returned if it is within the range provided.
		/// @param minimumValue The lower end of the range to clamp the value at, will be returned if the value is less-than
		///   the range.
		/// @param maximumValue The upper end of the range to clamp the value at, will be returned if the value is greater-than
		///   the range.
		///
		/// @note There will be compile issues if called on a type that does not have an operator< to use.
		///
		template <typename T> const T& Clamp(const T& value, const T& minimumValue, const T& maximumValue)
		{
			if (value < minimumValue)	return minimumValue;
			if (maximumValue < value)	return maximumValue;
			return value;
		}

	}; /* namespace Math */
}; /* namespace TurtleBrains */

namespace tbMath = TurtleBrains::Math;

#endif /* _TurtleBrains_Math_h_ */
