///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_Vector_h_
#define _TurtleBrains_Vector_h_

#include "../core/tb_configuration.h"
#include "../core/tb_error.h"
#include "../core/tb_defines.h" //For tb_unsused
#include "tb_math.h"

namespace TurtleBrains
{
	namespace Math
	{

		///
		/// @details The SkipInitialization type is used to provide a non-constuctor to a few objects, essentially Vector2,
		///   Vector3, Vector4 objects.  The non-constructor essentially leaves the members of the object uninitialized.
		///   This is unsafe but allows for slightly faster creation of the object with the premise that the values will be
		///   set immediately after.
		///
		/// @note Should be used only if the consequences are understood, and even then only sparingly.
		///
		enum SkipInitialization { kSkipInitialization = 0, };

		///
		/// @details Provides constant values for each component of the Vector2, Vector3 and Vector4 objects to get the
		///   component values by index using operator[].
		///
		enum VectorComponent
		{
			kComponentX = 0, ///< @details The index of the x component of the Vector2, Vector3 and Vector4 objects.
			kComponentY,     ///< @details The index of the y component of the Vector2, Vector3 and Vector4 objects.
			kComponentZ,     ///< @details The index of the z component of the Vector3, Vector4 objects.
			kComponentW      ///< @details The index of the w component of the Vector4 object.
		};

		///
		/// @details The Vector2 object is a two component mathmatical vector of floats packed together with some
		///   functionality to perform basic vector math.
		///
		class Vector2
		{
		public:
			///
			/// @details A constant vector representing 0.0f, 0.0f, 0.0f.
			///
			static const Vector2 kZero;

			union
			{
				float mComponents[2];

#if defined(tb_visual_cpp)
#pragma warning(push)
#pragma warning(disable: 4201)
				struct { float x, y; };
#pragma warning(pop)
#else
				struct { float x, y; };
#endif
			};

			///
			/// @details Constructs a Vector2 object leaving the members uninitialized and must be initialized manually before
			///   the object is used.
			///
			/// @note Should be used only if the consequences are understood, and even then only sparingly.
			///
			inline explicit Vector2(const SkipInitialization& fastAndStupid)
			{
				tb_unused(fastAndStupid);
			}

			///
			/// @details Constructs a default Vector2 object with all components set to 0.0f.
			///
			inline Vector2(void) :
				x(0.0f),
				y(0.0f)
			{
			}

			///
			/// @details Constructs a Vector2 object with each component initialized with the values provided.
			///
			/// @param valueX The value to initialize the x component as.
			/// @param valueY The value to initialize the y component as.
			///
			inline Vector2(const float valueX, const float valueY) :
				x(valueX),
				y(valueY)
			{
			}

			///
			/// @details Constructs a Vector2 object by copying another Vector2 object, each component will be initialized to
			///   the value of the other components.
			///
			/// @param other The Vector2 object to initialize component values to.
			///
			inline Vector2(const Vector2& other) :
				x(other.x),
				y(other.y)
			{
			}

			///
			/// @details Nothing needs to happen to destroy a Vector2 object.
			///
			inline ~Vector2(void)
			{
			}

			///
			/// @details Assignment operator, sets each component of the Vector2 object to the values of the other object.
			///
			/// @param other The Vector2 object to set the components to.
			///
			inline Vector2& operator=(const Vector2& other)
			{
				if (&other != this)
				{
					x = other.x;
					y = other.y;
				}
				return *this;
			}

			///
			/// @details Equality operator, compares each component of the Vector2 object to the values of another Vector2
			///   returning true if all components are equal, or false if any of the components are not equal.
			///
			/// @param other The Vector2 object to compare against.
			///
			/// @note Because the components are floating point values tbMath::IsEqual is used to compare the components for
			///   equality using the default tolerance: kTolerance.
			///
			inline bool operator==(const Vector2& other) const
			{
				return (true == IsEqual(x, other.x) && true == IsEqual(y, other.y)) ? true : false;
			}

			///
			/// @details Inquality operator, returns the opposite of the equality operator, which compares each value in one
			///   Vector2 object to another using IsEqual to compare the floating point component values.
			///
			inline bool operator!=(const Vector2& other) const
			{
				return (true == operator==(other)) ? false : true;
			}

			///
			/// @details Returns a const float array of the Vector2 components.
			///
			inline operator const float*(void) const { return mComponents; }

			///
			/// @details Returns a float array of the Vector2 components.  If modified the internal components will also be
			///   modified as well.
			///
			inline operator float*(void) { return mComponents; }

			///
			/// @details Returns a reference to one of the Vector2 components by an index provided by the VectorComponent enum.
			///
			inline const float& operator[](const size_t index) const { return mComponents[index]; }

			///
			/// @details Returns a reference to one of the Vector2 components by an index provided by the VectorComponent enum.
			///   If the reference returned is modified, the internal components will also be modified as well.
			///
			inline float& operator[](const size_t index) { return mComponents[index]; }

#ifndef tb_math_less_operators
			///
			/// @details Adds two Vector2 objects (this + rightSide) and returns a Vector2, by value, with the result.
			///
			inline Vector2 operator+(const Vector2& rightSide) const	{	return Vector2(x + rightSide.x, y + rightSide.y);	}

			///
			/// @details Adds each component of a Vector2 object to the component values of the left side object
			///   modifying it and returning the result.
			///
			inline Vector2& operator+=(const Vector2& rightSide)		{	x += rightSide.x, y += rightSide.y;	return *this;	}

			///
			/// @details Subtracts two Vector2 objects (this - rightSide) and returns a Vector2, by value, with the result.
			///
			inline Vector2 operator-(const Vector2& rightSide) const	{	return Vector2(x - rightSide.x, y - rightSide.y);	}

			///
			/// @details Subtracts two Vector2 objects (this -= rightSide) and modifies this object and returns the result.
			///
			inline Vector2& operator-=(const Vector2& rightSide)		{	x -= rightSide.x, y -= rightSide.y;	return *this;	}

			///
			/// @details Performs a scalar multiplication on the Vector2 object, returning the resulting Vector2.
			///
			inline Vector2 operator*(float scalar) const				{	return Vector2(x * scalar, y * scalar);	}

			///
			/// @details Performs a scalar multiplication on the Vector2 object with the scalar on the left-hand side
			///  for simple usage.  Returns the resulting Vector2.
			///
			friend Vector2 operator*(float scalar, const Vector2& rightSide) { return Vector2(scalar * rightSide.x, scalar * rightSide.y); }

			///
			/// @details Performa a scalar multiplication on the Vector2 object modifying the object and returning it
			///   as the resulting Vector2.
			///
			inline Vector2& operator*=(float scalar)				{	x *= scalar;	y *= scalar;	return *this;	}

			///
			/// @details Performs a scalar division operation on each component of the Vector2 object returning the result.
			///
			inline Vector2 operator/(float scalar) const			{	return Vector2(x / scalar, y / scalar);	}

			///
			/// @details Performs a scalar division operation on each component of the Vector2 object modifying the 
			///   componens of the Vector2 and returning the object as the result.
			///
			inline Vector2& operator/=(float scalar)				{	x /= scalar;	y /= scalar;	return *this;	}

			///
			/// @details Returns a Vector2 object with each component being the negative of the object used.
			///
			inline Vector2 operator-(void) const				{	return Vector2(-x, -y);		}

			///
			/// @details Perform a dot product on two Vector2 objects, returning the scalar result.
			///
			inline float operator*(const Vector2 &rhs) const	{	return (x * rhs.x) + (y * rhs.y);	}

			///
			/// @details Computes the length / magnitude of the Vector object which involves a square root, then
			///   returns the resulting scalar value.
			///
			inline float Magnitude(void) const { return sqrt((x * x) + (y * y)); }

			///
			/// @details Computes the length / magnitude of the Vector object left squared so the square root is not
			///   taken, then returns the resulting scalar value.
			///
			inline float MagnitudeSquared(void) const { return (x * x) + (y * y); }

			///
			/// @details Computes the magnitude and returns a unit length vector in the same direction.  Since this
			///   could result in a divide-by-zero, if the magnitude is zero, the zero vector is returned.
			///
			inline Vector2 GetNormalized(void) const
			{
				const float magnitude(Magnitude()); 
				if (true == IsZero(magnitude)) { return kZero; }
				return Vector2(x / magnitude, y / magnitude);
			}

			///
			/// @details Computes the magnitude of the Vector object and modifies it to represent the unit length 
			///   vector in the same direction.  If the magnitude is zero, the vector remains unmodified...  After 
			///   modifying the object the old magnitude is returned.
			///
			inline float Normalize(void)
			{
				const float magnitude(Magnitude()); 
				if (false == IsZero(magnitude))
				{
					x /= magnitude;
					y /= magnitude;
				}
				return magnitude;
			}

			///
			/// @details Performs scalar multiplication on the Vector object, exactly the same as *= with a scalar.
			///
			inline void Scale(float scalar) { *this *= scalar; }

			///
			/// @details This is a convenience function that will normalize then scale the Vector object, setting the
			///   length as desried.
			///
			/// @param length  The desired length of the vector in the same direction.  1.0 is the unit vector and has
			///   the same vector as if you had just normalized.
			///
			inline void SetLength(float length) { Normalize(); *this *= length; }
#endif
		};

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details The Vector3 object is a three component mathmatical vector of floats packed together with some
		///   functionality to perform basic vector math.
		///
		class Vector3
		{
		public:
			///
			/// @details A constant vector representing 0.0f, 0.0f, 0.0f.
			///
			static const Vector3 kZero;

			union
			{
				float mComponents[3];

#if defined(tb_visual_cpp)
#pragma warning(push)
#pragma warning(disable: 4201)
				struct { float x, y, z; };
#pragma warning(pop)
#else
				struct { float x, y, z; };
#endif
			};

			///
			/// @details Constructs a Vector3 object leaving the members uninitialized and must be initialized manually before
			///   the object is used.
			///
			/// @note Should be used only if the consequences are understood, and even then only sparingly.
			///
			inline explicit Vector3(const SkipInitialization& fastAndStupid)
			{
				tb_unused(fastAndStupid);
			}

			///
			/// @details Constructs a default Vector3 object with all components set to 0.0f.
			///
			Vector3(void) :
				x(0.0f),
				y(0.0f),
				z(0.0f)
			{
			}

			///
			/// @details Constructs a Vector4 object with each component initialized with the values provided.
			///
			/// @param valueX The value to initialize the x component as.
			/// @param valueY The value to initialize the y component as.
			/// @param valueZ The value to initialize the z component as.
			///
			inline Vector3(const float valueX, const float valueY, const float valueZ) :
				x(valueX),
				y(valueY),
				z(valueZ)
			{
			}

			///
			/// @details Constructs a Vector3 object by copying a Vector2 object and providing a value for the Z component.
			///
			/// @param other The Vector2 object to initialize the x and y component values to.
			/// @param valueZ The value to initialize the z component as.
			///
			inline explicit Vector3(const Vector2& other, const float valueZ) :
				x(other.x),
				y(other.y),
				z(valueZ)
			{
			}

			///
			/// @details Constructs a Vector3 object by copying another Vector3 object, each component will be initialized to
			///   the value of the other components.
			///
			/// @param other The Vector3 object to initialize component values to.
			///
			Vector3(const Vector3& other) :
				x(other.x),
				y(other.y),
				z(other.z)
			{
			}

			///
			/// @details Nothing needs to happen to destroy a Vector3 object.
			///
			~Vector3(void)
			{
			}

			///
			/// @details Assignment operator, sets each component of the Vector3 object to the values of the other object.
			///
			/// @param other The Vector3 object to set the components to.
			///
			inline Vector3& operator=(const Vector3& other)
			{
				if (&other != this)
				{
					x = other.x;
					y = other.y;
					z = other.z;
				}
				return *this;
			}

			//
			// TODO: TIM: Planning: Would this be useful to have, or just dangerous?
			//
			//inline Vector3& operator=(const Vector2 &v)
			//{
			//	x = v.x;	y = v.y; /* z = z; */
			//	return (*this);
			//}

			///
			/// @details Equality operator, compares each component of the Vector3 object to the values of another Vector3
			///   returning true if all components are equal, or false if any of the components are not equal.
			///
			/// @param other The Vector3 object to compare against.
			///
			/// @note Because the components are floating point values tbMath::IsEqual is used to compare the components for
			///   equality using the default tolerance: kTolerance.
			///
			inline bool operator==(const Vector3& other) const
			{
				return (true == IsEqual(x, other.x) && true == IsEqual(y, other.y) && true == IsEqual(z, other.z)) ? true : false;
			}

			///
			/// @details Inquality operator, returns the opposite of the equality operator, which compares each value in one
			///   Vector3 object to another using IsEqual to compare the floating point component values.
			///
			inline bool operator!=(const Vector3& other) const
			{
				return (true == operator==(other)) ? false : true;
			}

			///
			/// @details Returns a float array of the Vector3 components.
			///
			inline operator const float*(void) const { return mComponents; }

			///
			/// @details Returns a float array of the Vector3 components.  If modified the internal components will also be
			///   modified as well.
			///
			inline operator float*(void) { return mComponents; }

			///
			/// @details Returns a reference to one of the Vector3 components by an index provided by the VectorComponent enum.
			///
			inline const float& operator[](const size_t index) const { return mComponents[index]; }

			///
			/// @details Returns a reference to one of the Vector3 components by an index provided by the VectorComponent enum.
			///   If the reference returned is modified, the internal components will also be modified as well.
			///
			inline float& operator[](const size_t index) { return mComponents[index]; }

			//TODO: TIM: Reconsider: This could be dangerous, do we want to support it?
			//inline operator Vector2(void) { return Vector2(x, y);	}

	#ifndef tb_math_less_operators
			///
			/// @details Adds two Vector3 objects (this + rightSide) and returns a Vector3, by value, with the result.
			///
			inline Vector3 operator+(const Vector3& rightSide) const	{	return Vector3(x + rightSide.x, y + rightSide.y, z + rightSide.z);	}

			///
			/// @details Adds each component of a Vector3 object to the component values of the left side object
			///   modifying it and returning the result.
			///
			inline Vector3& operator+=(const Vector3& rightSide)		{	x += rightSide.x, y += rightSide.y;	z += rightSide.z;	return *this;	}

			///
			/// @details Subtracts two Vector3 objects (this - rightSide) and returns a Vector3, by value, with the result.
			///
			inline Vector3 operator-(const Vector3& rightSide) const	{	return Vector3(x - rightSide.x, y - rightSide.y, z - rightSide.z);	}

			///
			/// @details Subtracts two Vector3 objects (this -= rightSide) and modifies this object and returns the result.
			///
			inline Vector3& operator-=(const Vector3& rightSide)		{	x -= rightSide.x, y -= rightSide.y;	z -= rightSide.z;	return *this;	}


			///
			///@details Performs a scalar multiplication on the Vector3 object returning the resulting Vector3.
			///
			inline Vector3 operator*(float scalar) const			{	return Vector3(x * scalar, y * scalar, z * scalar);	}

			///
			///@details Performs a scalar multiplication on the Vector2 object with the scalar on the left-hand side
			///  for simple usage.  Returns the resulting Vector2.
			///
			friend Vector3 operator*(float scalar, const Vector3& rightSide) { return Vector3(scalar * rightSide.x, scalar * rightSide.y, scalar * rightSide.z); }

			///
			/// @details Performa a scalar multiplication on the Vector3 object modifying the object and returning it
			///   as the resulting Vector3.
			///
			inline Vector3& operator*=(float scalar)				{	x *= scalar;	y *= scalar;	z *= scalar;	return *this;	}

			///
			/// @details Performs a scalar division operation on each component of the Vector3 object returning the result.
			///
			inline Vector3 operator/(float scalar) const			{	return Vector3(x / scalar, y / scalar, z / scalar);	}

			///
			/// @details Performs a scalar division operation on each component of the Vector3 object modifying the 
			///   componens of the Vector3 and returning the object as the result.
			///
			inline Vector3& operator/=(float scalar)				{	x /= scalar;	y /= scalar;	z /= scalar;	return *this;	}


			///
			/// @details Returns a Vector3 object with each component being the negative of the object used.
			///
			inline Vector3 operator-(void) const				{	return Vector3(-x, -y, -z);		}

			///
			/// @details Perform a dot product on two Vector3 objects, returning the scalar result.
			///
			inline float operator*(const Vector3 &rhs) const	{	return (x * rhs.x) + (y * rhs.y) + (z * rhs.z);		}

			///
			/// @details Perform a cross product on two Vector3 objects, returning the Vector3 result that is 
			///   perpendicular two the two given vectors.
			///
			inline Vector3 operator^(const Vector3& rightSide) const
			{	
				return Vector3((y * rightSide.z) - (rightSide.y * z), -((x * rightSide.z) - (rightSide.x * z)), (x * rightSide.y) - (rightSide.x * y));
			}


			///
			/// @details Computes the length / magnitude of the Vector object which involves a square root, then
			///   returns the resulting scalar value.
			///
			inline float Magnitude(void) const { return sqrt((x * x) + (y * y) + (z * z)); }

			///
			/// @details Computes the length / magnitude of the Vector object left squared so the square root is not
			///   taken, then returns the resulting scalar value.
			///
			inline float MagnitudeSquared(void) const { return (x * x) + (y * y) + (z * z); }

			///
			/// @details Computes the magnitude and returns a unit length vector in the same direction.  Since this
			///   could result in a divide-by-zero, if the magnitude is zero, the zero vector is returned.
			///
			inline Vector3 GetNormalized(void) const
			{ 
				const float magnitude(Magnitude());
				if (true == IsZero(magnitude)) { return kZero; }
				return Vector3(x / magnitude, y / magnitude, z / magnitude);
			}

			///
			/// @details Computes the magnitude of the Vector object and modifies it to represent the unit length 
			///   vector in the same direction.  If the magnitude is zero, the vector remains unmodified...  After 
			///   modifying the object the old magnitude is returned.
			///
			inline float Normalize(void)
			{
				const float magnitude(Magnitude());
				if (false == IsZero(magnitude))
				{
					x /= magnitude;
					y /= magnitude;
					z /= magnitude;
				}
				return magnitude;
			}

			///
			/// @details Performs scalar multiplication on the Vector object, exactly the same as *= with a scalar.
			///
			inline void Scale(float scalar) { *this *= scalar; }

			///
			/// @details This is a convenience function that will normalize then scale the Vector object, setting the
			///   length as desried.
			///
			/// @param length  The desired length of the vector in the same direction.  1.0 is the unit vector and has
			///   the same vector as if you had just normalized.
			///
			inline void SetLength(float length) { Normalize(); *this *= length; }
	#endif
		};

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details The Vector4 object is a four component mathmatical vector of floats packed together with some
		///   functionality to perform basic vector math.
		///
		class Vector4
		{
		public:
			///
			/// @details A constant vector representing 0.0f, 0.0f, 0.0f.
			///
			static const Vector4 kZero;

			union
			{
				float mComponents[4];

#if defined(tb_visual_cpp)
#pragma warning(push)
#pragma warning(disable: 4201)
				struct { float x, y, z, w; };
#pragma warning(pop)
#else
				struct { float x, y, z, w; };
#endif
			};

			///
			/// @details Constructs a Vector4 object leaving the members uninitialized and must be initialized manually before
			///   the object is used.
			///
			/// @note Should be used only if the consequences are understood, and even then only sparingly.
			///
			inline explicit Vector4(const SkipInitialization& fastAndStupid)
			{
				tb_unused(fastAndStupid);
			}

			///
			/// @details Constructs a default Vector4 object with all components set to 0.0f.
			///
			inline Vector4(void) :
				x(0.0f),
				y(0.0f),
				z(0.0f),
				w(0.0f)
			{
			}

			///
			/// @details Constructs a Vector4 object with each component initialized with the values provided.
			///
			/// @param valueX The value to initialize the x component as.
			/// @param valueY The value to initialize the y component as.
			/// @param valueZ The value to initialize the z component as.
			/// @param valueW The value to initialize the w component as.
			///
			inline Vector4(const float valueX, const float valueY, const float valueZ, const float valueW) :
				x(valueX),
				y(valueY),
				z(valueZ),
				w(valueW)
			{
			}

			///
			/// @details Constructs a Vector4 object by copying a Vector2 object and providing a value for the z and w components.
			///
			/// @param other The Vector2 object to initialize the x and y component values to.
			/// @param valueZ The value to initialize the z component as.
			/// @param valueW The value to initialize the w component as.
			///
			inline explicit Vector4(const Vector2& other, const float valueZ, const float valueW) :
				x(other.x),
				y(other.y),
				z(valueZ),
				w(valueW)
			{
			}

			///
			/// @details Constructs a Vector4 object by copying a Vector3 object and providing a value for the w component.
			///
			/// @param other The Vector3 object to initialize the x, y and z component values to.
			/// @param valueW The value to initialize the w component as.
			///
			inline explicit Vector4(const Vector3& other, const float valueW) :
				x(other.x),
				y(other.y),
				z(other.z),
				w(valueW)
			{
			}

			///
			/// @details Constructs a Vector4 object by copying another Vector4 object, each component will be initialized to
			///   the value of the other components.
			///
			/// @param other The Vector4 object to initialize component values to.
			///
			inline Vector4(const Vector4& other) :
				x(other.x),
				y(other.y),
				z(other.z),
				w(other.w)
			{
			}

			///
			/// @details Nothing needs to happen to destroy a Vector4 object.
			///
			~Vector4(void)
			{
			}

			///
			/// @details Assignment operator, sets each component of the Vector4 object to the values of the other object.
			///
			/// @param other The Vector4 object to set the components to.
			///
			inline Vector4& operator=(const Vector4& other)
			{
				if (&other != this)
				{
					x = other.x;
					y = other.y;
					z = other.z;
					w = other.w;
				}
				return *this;
			}

			///
			/// @details Equality operator, compares each component of the Vector4 object to the values of another Vector4
			///   returning true if all components are equal, or false if any of the components are not equal.
			///
			/// @param other The Vector4 object to compare against.
			///
			/// @note Because the components are floating point values tbMath::IsEqual is used to compare the components for
			///   equality using the default tolerance: kTolerance.
			///
			inline bool operator==(const Vector4& other) const
			{
				return (true == IsEqual(x, other.x) && true == IsEqual(y, other.y) &&
					true == IsEqual(z, other.z) && true == IsEqual(w, other.w)) ? true : false;
			}

			///
			/// @details Inquality operator, returns the opposite of the equality operator, which compares each value in one
			///   Vector4 object to another using IsEqual to compare the floating point component values.
			///
			inline bool operator!=(const Vector4& other) const
			{
				return (true == operator==(other)) ? false : true;
			}

			///
			/// @details Returns a float array of the Vector4 components.
			///
			inline operator const float*(void) const { return mComponents; }

			///
			/// @details Returns a float array of the Vector4 components.  If modified the internal components will also be
			///   modified as well.
			///
			inline operator float*(void) { return mComponents; }

			///
			/// @details Returns a reference to one of the Vector4 components by an index provided by the VectorComponent enum.
			///
			inline const float& operator[](const size_t index) const { return mComponents[index]; }

			///
			/// @details Returns a reference to one of the Vector4 components by an index provided by the VectorComponent enum.
			///   If the reference returned is modified, the internal components will also be modified as well.
			///
			inline float& operator[](const size_t index) { return mComponents[index]; }

			//TODO: TIM: Reconsider: This could be dangerous, do we want to support it?
			//inline operator Vector2(void) { return Vector2(x, y); }
			//inline operator Vector3(void) { return Vector3(x, y, z); }

#ifndef tb_math_less_operators
			///
			/// @details Adds two vectors together and returns a Vector4 object, by value, with the result.
			///
			inline Vector4 operator+(const Vector4& rightSide) const	{	return Vector4(x + rightSide.x, y + rightSide.y, z + rightSide.z, w + rightSide.w);	}

			///
			/// @details Adds each component of a Vector4 object to the component values of the left side object
			///   modifying it and returning the result.
			///
			inline Vector4& operator+=(const Vector4& rightSide)		{	x += rightSide.x, y += rightSide.y;	z += rightSide.z;	w += rightSide.w; return *this;	}

			///
			/// @details Subtracts two vectors (this - rightSide) and returns a Vector4 object, by value, with the result.
			///
			inline Vector4 operator-(const Vector4& rightSide) const	{	return Vector4(x - rightSide.x, y - rightSide.y, z - rightSide.z, w - rightSide.w);	}

			///
			/// @details Subtracts two Vector2 objects (this -= rightSide) and modifies this object and returns the result.
			///
			inline Vector4& operator-=(const Vector4& rightSide)		{	x -= rightSide.x, y -= rightSide.y;	z -= rightSide.z; w -= rightSide.w;	return *this;	}


			///
			/// @details Performs a scalar multiplication on the Vector4 object, returning the resulting Vector4.
			///
			inline Vector4 operator*(float scalar) const				{	return Vector4(x * scalar, y * scalar, z * scalar, w * scalar);	}

			///
			/// @details Performs a scalar multiplication on the Vector4 object with the scalar on the left-hand side
			///  for simple usage.  Returns the resulting Vector4.
			///
			friend Vector4 operator*(float scalar, const Vector4& rightSide) { return Vector4(scalar * rightSide.x, scalar * rightSide.y, scalar * rightSide.z, scalar * rightSide.w); }

			///
			/// @details Performa a scalar multiplication on the Vector4 object modifying the object and returning it
			///   as the resulting Vector4.
			///
			inline Vector4& operator*=(float scalar)					{	x *= scalar;	y *= scalar;	z *= scalar; w *= scalar;	return *this;	}

			///
			/// @details Performs a scalar division operation on each component of the Vector4 object returning the result.
			///
			inline Vector4 operator/(float scalar) const			{	return Vector4(x / scalar, y / scalar, z / scalar, w / scalar);	}

			///
			/// @details Performs a scalar division operation on each component of the Vector4 object modifying the 
			///   componens of the Vector4 and returning the object as the result.
			///
			inline Vector4& operator/=(float scalar)				{	x /= scalar;	y /= scalar;	z /= scalar; w /= scalar;	return *this;	}


			///
			/// @details Returns a Vector4 object with each component being the negative of the object used.
			///
			inline Vector4 operator-(void) const				{	return Vector4(-x, -y, -z, -w);		}

			///
			/// @details Perform a dot product on two Vector4 objects, returning the scalar result.
			///
			inline float operator*(const Vector4& rightSide) const	{	return (x * rightSide.x) + (y * rightSide.y) + (z * rightSide.z) + (w * rightSide.w);	}

			//TODO: TIM: Reconsider: Does this work, and if it does, how do we want to support it?  Document if needed.
			//inline Vector4 operator^(const Vector4 &rhs) const	{	return Vector4((y * rhs.z) - (rhs.y * z), -((x * rhs.z) - (rhs.x * z)), (x * rhs.y) - (rhs.x * y));	 }

			///
			/// @details Computes the length / magnitude of the Vector object which involves a square root, then
			///   returns the resulting scalar value.
			///
			inline float Magnitude(void) const { return sqrt((x * x) + (y * y) + (z * z) + (w * w)); }

			///
			/// @details Computes the length / magnitude of the Vector object left squared so the square root is not
			///   taken, then returns the resulting scalar value.
			///
			inline float MagnitudeSquared(void) const { return (x * x) + (y * y) + (z * z) + (w * w); }

			///
			/// @details Computes the magnitude and returns a unit length vector in the same direction.  Since this
			///   could result in a divide-by-zero, if the magnitude is zero, the zero vector is returned.
			///
			inline Vector4 GetNormalized(void) const
			{
				const float magnitude(Magnitude());
				if (true == IsZero(magnitude)) { return kZero; }
				return Vector4(x / magnitude, y / magnitude, z / magnitude, w / magnitude);
			}

			///
			/// @details Computes the magnitude of the Vector object and modifies it to represent the unit length 
			///   vector in the same direction.  If the magnitude is zero, the vector remains unmodified...  After 
			///   modifying the object the old magnitude is returned.
			///
			inline float Normalize(void)
			{
				const float magnitude(Magnitude());
				if (false == IsZero(magnitude))
				{
					x /= magnitude;
					y /= magnitude;
					z /= magnitude;
					w /= magnitude;
				}
				return magnitude;
			}

			///
			/// @details Performs scalar multiplication on the Vector object, exactly the same as *= with a scalar.
			///
			inline void Scale(float scalar) { *this *= scalar; }

			///
			/// @details This is a convenience function that will normalize then scale the Vector object, setting the
			///   length as desried.
			///
			/// @param length  The desired length of the vector in the same direction.  1.0 is the unit vector and has
			///   the same vector as if you had just normalized.
			///
			inline void SetLength(float length) { Normalize(); *this *= length; }
#endif
		};

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @details Add two the components of one Vector to another Vector and return the resulting Vector.  Behavior is
		///   the same as Vector::operator+
		///
		/// @param leftSide The Vector object to take the left hand side of the addition operation.
		/// @param rightSide The Vector object to take the right hand side of the addition operation.
		/// @param result is the result of leftSide + rightSide.
		///
		/// @note None of the inputs can be null.
		///
		inline Vector2* Vector2Add(Vector2* result, const Vector2* leftSide, const Vector2* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x + rightSide->x;
			result->y = leftSide->y + rightSide->y;
			return result;
		}

		///
		/// @copydoc Vector2Add()
		///
		inline Vector3* Vector3Add(Vector3* result, const Vector3* leftSide, const Vector3* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x + rightSide->x;
			result->y = leftSide->y + rightSide->y;
			result->z = leftSide->z + rightSide->z;
			return result;
		}

		///
		/// @copydoc Vector2Add()
		///
		inline Vector4* Vector4Add(Vector4* result, const Vector4* leftSide, const Vector4* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x + rightSide->x;
			result->y = leftSide->y + rightSide->y;
			result->z = leftSide->z + rightSide->z;
			result->w = leftSide->w + rightSide->w;
			return result;
		}

		///
		/// @details Subtracts the components of one Vector from another Vector and returns the resulting Vector.  Behavior
		///   is the same as using Vector::operator-
		///
		/// @param leftSide The Vector object to take the left hand side of the subtraction operation.
		/// @param rightSide The Vector object to take the right hand side of the subtraction operation.
		/// @param result is the result of leftSide - rightSide.
		///
		/// @note None of the inputs can be null.
		///
		inline Vector2* Vector2Subtract(Vector2* result, const Vector2* leftSide, const Vector2* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x - rightSide->x;
			result->y = leftSide->y - rightSide->y;
			return result;
		}

		///
		/// @copydoc Vector2Subtract()
		///
		inline Vector3* Vector3Subtract(Vector3* result, const Vector3* leftSide, const Vector3* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x - rightSide->x;
			result->y = leftSide->y - rightSide->y;
			result->z = leftSide->z - rightSide->z;
			return result;
		}

		///
		/// @copydoc Vector2Subtract()
		///
		inline Vector4* Vector4Subtract(Vector4* result, const Vector4* leftSide, const Vector4* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");

			result->x = leftSide->x - rightSide->x;
			result->y = leftSide->y - rightSide->y;
			result->z = leftSide->z - rightSide->z;
			result->w = leftSide->w - rightSide->w;
			return result;
		}

		///
		/// @details Scales the components of one Vector and returns the resulting Vector.
		///
		/// @param result The result of input * scalar.
		/// @param input The Vector object to scale by the scalar value.
		/// @param scalar The amount to scale the input Vector by.  2.0f would double the Vectors magnitude.
		///
		/// @note None of the inputs can be null.
		///
		inline Vector2* Vector2Scale(Vector2* result, const Vector2* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x * scalar;
			result->y = input->y * scalar;
			return result;
		}

		///
		/// @copydoc Vector2Scale()
		///
		inline Vector3* Vector3Scale(Vector3* result, const Vector3* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x * scalar;
			result->y = input->y * scalar;
			result->z = input->z * scalar;
			return result;
		}

		///
		/// @copydoc Vector2Scale()
		///
		inline Vector4* Vector4Scale(Vector4* result, const Vector4* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x * scalar;
			result->y = input->y * scalar;
			result->z = input->z * scalar;
			result->w = input->w * scalar;
			return result;
		}

		///
		/// @details Divides the components of one Vector by the scalar and returns the resulting Vector.
		///
		/// @param result The result of input * scalar.
		/// @param input The Vector object to scale by the scalar value.
		/// @param scalar The amount to inverse scale the input Vector by.  2.0f would half the Vectors magnitude.
		///
		/// @note None of the inputs can be null.
		///
		inline Vector2* Vector2ScaleDivide(Vector2* result, const Vector2* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x / scalar;
			result->y = input->y / scalar;
			return result;
		}

		///
		/// @copydoc Vector2ScaleDivide()
		///
		inline Vector3* Vector3ScaleDivide(Vector3* result, const Vector3* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x / scalar;
			result->y = input->y / scalar;
			result->z = input->z / scalar;
			return result;
		}

		///
		/// @copydoc Vector2ScaleDivide()
		///
		inline Vector4* Vector4ScaleDivide(Vector4* result, const Vector4* input, const float scalar)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = input->x / scalar;
			result->y = input->y / scalar;
			result->z = input->z / scalar;
			result->w = input->w / scalar;
			return result;
		}

		///
		/// @details Negates each of the components of a Vector and returns the resulting Vector.
		///
		/// @param result The result of negating each component in input.
		/// @param input The Vector object to negate.
		///
		/// @note None of the inputs can be null.
		///
		inline Vector2* Vector2Negate(Vector2* result, const Vector2* input)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = -input->x;
			result->y = -input->y;
			return result;
		}

		///
		/// @copydoc Vector2Negate()
		///
		inline Vector3* Vector3Negate(Vector3* result, const Vector3* input)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = -input->x;
			result->y = -input->y;
			result->z = -input->z;
			return result;
		}

		///
		/// @copydoc Vector2Negate()
		///
		inline Vector4* Vector4Negate(Vector4* result, const Vector4* input)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			result->x = -input->x;
			result->y = -input->y;
			result->z = -input->z;
			result->w = -input->w;
			return result;
		}

		///
		/// @details Performs the DotProduct on two Vector objects returning resulting scalar value.
		///
		/// @param leftSide The Vector object to take the left hand side of the dot product operation.
		/// @param rightSide The Vector object to take the right hand side of the dot product operation.
		///
		/// @note None of the inputs can be null.
		///
		inline float Vector2DotProduct(const Vector2* leftSide, const Vector2* rightSide)
		{
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");
			return (leftSide->x * rightSide->x) + (leftSide->y * rightSide->y);
		}

		///
		/// @copydoc Vector2DotProduct()
		///
		inline float Vector3DotProduct(const Vector3* leftSide, const Vector3* rightSide)
		{
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");
			return (leftSide->x * rightSide->x) + (leftSide->y * rightSide->y) + (leftSide->z * rightSide->z);
		}

		///
		/// @copydoc Vector2DotProduct()
		///
		inline float Vector4DotProduct(const Vector4* leftSide, const Vector4* rightSide)
		{
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");
			return (leftSide->x * rightSide->x) + (leftSide->y * rightSide->y) + (leftSide->z * rightSide->z) + (leftSide->w * rightSide->w);
		}

		///
		/// @details Computes the CrossProduct of two Vector objects. The resulting Vector will be perpendicular to the
		///   input vectors, or become a zero vector if those are parallel.
		///
		/// @param result The result of leftSide cross rightSide, also performed with operator^
		/// @param leftSide The Vector object to take the left hand side of the cross product operation.
		/// @param rightSide The Vector object to take the right hand side of the cross product operation.
		///
		/// @note None of the input parameters can be null, and result must be different than leftSide and rightSide.
		///
		inline Vector3* Vector3CrossProduct(Vector3* result, const Vector3* leftSide, const Vector3* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");
			tb_error_if(leftSide == rightSide, "tbExternalError: Invalid parameter; expected leftSide to be different from rightSide.");
			tb_error_if(result == leftSide || result == rightSide, "Invalid parameter; expected result to be different than leftSide and rightSide");

			result->x = ((leftSide->y * rightSide->z) - (rightSide->y * leftSide->z));
			result->y = -(((leftSide->x * rightSide->z) - (rightSide->x * leftSide->z)));
			result->z = ((leftSide->x * rightSide->y) - (rightSide->x * leftSide->y));
			return result;
		}

		///
		/// @copydoc Vector3CrossProduct()
		/// @note Currently unsure if this is accurate, so it is implemented but will throw an error upon use.
		///
		inline Vector4* Vector4CrossProduct(Vector4* result, const Vector4* leftSide, const Vector4* rightSide)
		{
			tb_error_if(nullptr == result, "tbExternalError: Invalid parameter for result, expected valid pointer.");
			tb_error_if(nullptr == leftSide, "tbExternalError: Invalid parameter for leftSide, expected valid pointer.");
			tb_error_if(nullptr == rightSide, "tbExternalError: Invalid parameter for rightSide, expected valid pointer.");
			tb_error_if(result == leftSide || result == rightSide, "Invalid parameter; expected result to be different than leftSide and rightSide");

			tb_error_if(true, "Not sure if this is an accurate Vector4 CrossProduct");
			result->x = ((leftSide->y * rightSide->z) - (rightSide->y * leftSide->z));
			result->y = -(((leftSide->x * rightSide->z) - (rightSide->x * leftSide->z)));
			result->z = ((leftSide->x * rightSide->y) - (rightSide->x * leftSide->y));
			result->w = 0.0f;
			return result;
		}

		///
		/// @details Computes the length (magnitude) of a Vector object returning resulting scalar value.
		///
		/// @param input The Vector object to compute the length of.
		///
		/// @note The input vector cannot be null.
		///
		inline float Vector2Magnitude(const Vector2* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return sqrt((input->x * input->x) + (input->y * input->y));
		}

		///
		/// @copydoc Vector2Magnitude()
		///
		inline float Vector3Magnitude(const Vector3* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return sqrt((input->x * input->x) + (input->y * input->y) + (input->z * input->z));
		}

		///
		/// @copydoc Vector2Magnitude()
		///
		inline float Vector4Magnitude(const Vector4* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return sqrt((input->x * input->x) + (input->y * input->y) + (input->z * input->z) + (input->w * input->w));
		}

		///
		/// @details Computes the length (magnitude) of a Vector object without performing the squareroot, so the resulting
		///   scalar is actually the squared length of the Vector.
		///
		/// @param input The Vector object to get the squared length of.
		///
		/// @note The input vector cannot be null.
		///
		inline float Vector2MagnitudeSquared(const Vector2* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return (input->x * input->x) + (input->y * input->y);
		}

		///
		/// @copydoc Vector2MagnitudeSquared()
		///
		inline float Vector3MagnitudeSquared(const Vector3* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return (input->x * input->x) + (input->y * input->y) + (input->z * input->z);
		}

		///
		/// @copydoc Vector2MagnitudeSquared()
		///
		inline float Vector4MagnitudeSquared(const Vector4* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");
			return (input->x * input->x) + (input->y * input->y) + (input->z * input->z) + (input->w * input->w);
		}

		///
		/// @details Calculates the unit-length vector of the input Vector and returns the resulting Vector.
		///
		/// @param input The Vector object to get the unit-length of.
		/// @param result The unit length vector from the input, or a zero vector if the input had no length.
		///
		/// @note The input vector cannot be null.
		///
		inline Vector2* Vector2Normalize(Vector2* result, const Vector2* input)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			const float magnitude = Vector2Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
			}
			return result;
		}

		///
		/// @copydoc Vector2Normalize()
		///
		inline Vector3* Vector3Normalize(Vector3* result, const Vector3* input)
		{
			const float magnitude = Vector3Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
				result->z = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
				result->z = input->z / magnitude;
			}
			return result;
		}

		///
		/// @copydoc Vector2Normalize()
		///
		inline Vector4* Vector4Normalize(Vector4* result, const Vector4* input)
		{
			const float magnitude = Vector4Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
				result->z = 0.0f;
				result->w = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
				result->z = input->z / magnitude;
				result->w = input->w / magnitude;
			}
			return result;
		}

		///
		/// @details Calculates the unit-length vector of the input Vector and returns the resulting Vector and to save
		///   computing the magnitude elsewhere, this will return the value in the magnitude parameter.
		///
		/// @param result The unit length vector from the input, or a zero vector if the input had no length.
		/// @param input The Vector object to get the unit-length of.
		/// @param magnitude The length of the input vector as it must be computed.
		///
		/// @note The input vector cannot be null.
		///
		inline Vector2* Vector2NormalizeMagnitude(Vector2* result, const Vector2* input, float& magnitude)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			magnitude = Vector2Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
			}
			return result;
		}

		///
		/// @copydoc Vector2NormalizeMagnitude()
		///
		inline Vector3* Vector3NormalizeMagnitude(Vector3* result, const Vector3* input, float &magnitude)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			magnitude = Vector3Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
				result->z = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
				result->z = input->z / magnitude;
			}
			return result;
		}

		///
		/// @copydoc Vector2NormalizeMagnitude()
		///
		inline Vector4* Vector4NormalizeMag(Vector4* result, const Vector4* input, float &magnitude)
		{
			tb_error_if(nullptr == input, "tbExternalError: Invalid parameter for input, expected valid pointer.");

			magnitude = Vector4Magnitude(input);
			if (true == IsZero(magnitude))
			{
				result->x = 0.0f;
				result->y = 0.0f;
				result->z = 0.0f;
				result->w = 0.0f;
			}
			else
			{
				result->x = input->x / magnitude;
				result->y = input->y / magnitude;
				result->z = input->z / magnitude;
				result->w = input->w / magnitude;
			}
			return result;
		}

		///
		/// @details Returns angle between vectors in radians, 0 is both pointing same direction, kPi if both vectors
		///   are pointing in opposite directions.
		///
		inline float Vector3AngleBetween(const Vector3* left, const Vector3* right)
		{
			const float productOfMagnitudes(Vector3Magnitude(left) * Vector3Magnitude(right));
			if (true == IsZero(productOfMagnitudes)) { return 0.0f; }
			const float value(Vector3DotProduct(left, right) / productOfMagnitudes);
			const float clampedValue((value < -1.0f) ? -1.0f : (value > 1.0f) ? 1.0f  : value); //Clamp: -1.0f <= value <= 1.0f
			return acos(clampedValue);
		}

		///
		/// @details Converts an orientation value from 2D (XZ plane) into a forward vector X, 0, Z.  Orientation should be
		///   in radians, positive for clockwise rotation around the Y axis, 0 pointing along the -Z axis.
		///
		static inline Vector3* OrientationToForwardVector3(Vector3 *result, float orientation)
		{
			result->x = sin(orientation);
			result->y = 0.0f;
			result->z = -cos(orientation);

			return result;
		}

		///
		/// @details Converts an orientation value from 2D into a forward vector.  Orientation should be
		///   in radians, positive for clockwise rotation around the Z axis, 0 pointing along the -Y axis.
		///
		static inline Vector2& OrientationToForwardVector2(Vector2& result, float orientation)
		{
			result.x = sin(orientation);
			result.y = -cos(orientation);
			return result;
		}

		///
		/// @details Converts a normalized 'forward' vector to an orientation value for 2D games.  The orientation returned
		///   will be the radians in a clockwise (TODO: TIM: Doublecheck this directional comment) from the ZAxis, meaning
		///   the input vector should be something like forwardX, 0, forwardZ instead of the forwardX, forwardY.
		///
		/// @param forward should be a vector on the XZ plane.
		///
		/// @note the orientation is in radians, not degrees.
		///
		static inline float ForwardVector3ToOrientation(const Vector3& forward)
		{
			Vector3 vZAxis(0.0f, 0.0f, -1.0f);
			float orientation = acos((vZAxis.x * forward.x) + (vZAxis.y * forward.y) + (vZAxis.z * forward.z));
			if (forward.x < 0.0f)
			{
				orientation = fabs(orientation - kTwoPi);
			}
			return orientation;
		}

		///
		/// @details Converts a normalized 'forward' vector to an orientation value for 2D games.  The orientation returned
		///   will be the radians in a clockwise (TODO: TIM: Doublecheck this directional comment) from the negative Y-Axis,
		///   meaning that something pointing up the screen (-Y) will return 0.
		///
		/// @param forward should be a normalized vector.
		///
		/// @note the orientation is in radians, not degrees.
		///
		static inline float ForwardVector2ToOrientation(const Vector2& forward)
		{
			Vector2 yAxis(0.0f, -1.0f);
			float orientation = acos((yAxis.x * forward.x) + (yAxis.y * forward.y));
			if (forward.x < 0.0f)
			{
				orientation = fabs(orientation - kTwoPi);
			}
			return orientation;
		}

	}; /* namespace Math */
}; /* namespace TurtleBrains */

namespace tbMath = TurtleBrains::Math;

#endif /* _TurtleBrains_Vector_h_ */
