///
/// @file
/// @details An entity within the LudumDare35 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#include "fighter_entity.h"
#include "game_results.h"
#include "arena_entity.h"
#include "pickup_entity.h"
#include "score_entity.h"

const tbGame::GameTimer LudumDare35::FighterEntity::kPowerAttackTime(400);
const tbGame::GameTimer LudumDare35::FighterEntity::kAttackDelayTime(250);

const float LudumDare35::FighterEntity::kMaximumAttackDistance[] = {
	90.0f,  // for Swing Attack
	110.0f, // for Stab Attack
};

const float LudumDare35::FighterEntity::kMaximumForwardAttackRange(65.0f);

static const LudumDare35::VersusInfo kGreatVersus = { 40, tb_string("great"), tbGame::GameTimer(750), 50.0f };
static const LudumDare35::VersusInfo kGoodVersus = { 25, tb_string("good"), tbGame::GameTimer(400), 25.0f };
static const LudumDare35::VersusInfo kWeakVersus = { 15, tb_string("weak"), tbGame::GameTimer(150), 5.0f };

const LudumDare35::VersusInfo* LudumDare35::kVersusTable[3][3] = {
	{ &kGoodVersus, &kGreatVersus, &kWeakVersus, },
	{ &kWeakVersus, &kGoodVersus, &kGreatVersus, },
	{ &kGreatVersus, &kWeakVersus, &kGoodVersus, },
};

const LudumDare35::FighterShape LudumDare35::kFighterTable[3][3] = {
	{ LudumDare35::kFighterWater, LudumDare35::kFighterEarth, LudumDare35::kFighterFire, },
	{ LudumDare35::kFighterFire, LudumDare35::kFighterWater, LudumDare35::kFighterEarth, },
	{ LudumDare35::kFighterEarth, LudumDare35::kFighterFire, LudumDare35::kFighterWater, },
};

const int kLootDropPercentage(30); //0 to 100 for % chance

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::FighterEntity::FighterEntity(const tbMath::Vector2& spawnPosition, const FighterShape& fighterShape) :
	tbGame::Entity("FighterEntity"),
	mFighterSprite("concept_art", "solid_square"),
	mSwordSprite("sword", "sword"),

	mIsLockedOnTarget(false),
	mTargetPosition(0.0f, 0.0f),
	mIsWalkingForward(false),
	mIsWalkingBackward(false),
	mIsStrafingLeft(false),
	mIsStrafingRight(false),
	mIsAttacking(false),
	mAttemptShapeShift(false),

	mIsMovingNorth(false),
	mIsMovingEast(false),
	mIsMovingSouth(false),
	mIsMovingWest(false),

	mWalkingSpeed(160.0f),
	mTurningSpeed(200.0f),

	mFighterShape(fighterShape),
	mCanAttack(true),
	mCanShapeShift(true),
	mIsOnArena(true),
	mIsStunned(false),
	mAttackTimer(0),
	mOnAttackDelayTimer(0),
	mShapeShiftTimer(0),


	mMoveForwardTimer(0),
	mMoveBackwardTimer(0),
	mMovementSpeed(250.0f),

	mHealthBar(80)
{
	SetPosition(spawnPosition);
	
	SetupSprite();
	AddGraphic(mFighterSprite);
	AddGraphic(mSwordSprite);

	mHealthBar.SetPosition(-32.0f, -52.0f);
	AddGraphic(mHealthBar);

	tbGraphics::AnimatedSprite::SetAnimationFrameRate(12);
	
	mFighterSprite.PlayAnimation("idle", true);
	mSwordSprite.PlayAnimation("idle", true);

	//const float collisionRadius(kMaximumAttackDistance[0]);
	//tbGraphics::Sprite* collisionCheck = new tbGraphics::Sprite("concept_art", "solid_circle");
	//collisionCheck->SetOrigin(tbGraphics::kAnchorCenter);
	//collisionCheck->SetColor(tbGraphics::Color(0x80FF8080));
	//collisionCheck->SetScale(collisionRadius / 32.0f);
	//AddGraphic(collisionCheck);

	AddBoundingCircle(32.0f);
}

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::FighterEntity::~FighterEntity(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

bool LudumDare35::FighterEntity::IsAlive(void) const
{
	return mHealthBar.IsAlive();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnAdded(void)
{
	tbGame::Entity::OnAdded();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnRemoved(void)
{
	tbGame::Entity::OnRemoved();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnSimulate(void)
{
	tbGame::Entity::OnSimulate();

	//
	// Do nothing below this if the fighter is dead.
	//
	if (false == IsAlive() || true == IsSpawning())
	{
		mFighterSprite.SetColor(tbGraphics::Color(0x80FFFFFF));
		mSwordSprite.SetColor(tbGraphics::Color(0x80FFFFFF));

		if (true == mSpawnTimer.DecrementStep())
		{
			mFighterSprite.SetColor(tbGraphics::Color(0xFFFFFFFF));
			mSwordSprite.SetColor(tbGraphics::Color(0xFFFFFFFF));
		}
		return;
	}

	tbMath::Vector2 currentPosition(GetPosition());
	float currentRotation(mFighterSprite.GetRotation());

	if (false == mMoveBackwardTimer.IsZero())
	{
		mMoveBackwardTimer.DecrementStep();
		currentPosition -= GetForwardDirection() * (mMovementSpeed * kFixedTime);
	}

	//
	// Do nothing if the fighter is stunned!
	//
	if (false == mStunnedTimer.IsZero())
	{
		if (false == mFighterSprite.IsAnimationPlaying("stun"))
		{
			mFighterSprite.PlayAnimation("stun");
		}

		if (true == mStunnedTimer.DecrementStep())
		{
			mFighterSprite.PlayAnimation("idle", true);
		}

		//Can not attack if hit while trying to attack.
		mAttackTimer = 0;
		return;
	}


	if (false == mMoveForwardTimer.IsZero())
	{
		mMoveForwardTimer.DecrementStep();
		currentPosition += GetForwardDirection() * (mMovementSpeed * kFixedTime);
	}

	//
	//Hard lock on TargetPosition atm.
	//
	if (true == mIsLockedOnTarget)
	{
		tbMath::Vector2 toTarget(mTargetPosition - currentPosition);
		currentRotation = tbMath::Convert::RadiansToDegrees(tbMath::ForwardVector2ToOrientation(toTarget.GetNormalized()));
	}

	//
	//Movement
	//
	const float movementSpeed(mWalkingSpeed * kFixedTime);
	const float rotationSpeed(mTurningSpeed * kFixedTime);

	{
		tbMath::Vector2 forwardDirection(GetForwardDirection());
		tbMath::Vector2 rightDirection(GetRightDirection());

		if (true == mIsWalkingForward)
		{
			currentPosition += forwardDirection * movementSpeed;
		}
		if (true == mIsWalkingBackward)
		{
			currentPosition -= forwardDirection * movementSpeed;
		}

		if (true == mIsLockedOnTarget)
		{	//Strafing actually strafes.
			if (true == mIsStrafingRight)
			{
				currentPosition += rightDirection * movementSpeed;
			}
			if (true == mIsStrafingLeft)
			{
				currentPosition -= rightDirection * movementSpeed;
			}
		}
		else
		{	//Strafing actually rotates.

			if (true == mIsStrafingRight)
			{
				currentRotation += rotationSpeed;
			}
			if (true == mIsStrafingLeft)
			{
				currentRotation -= rotationSpeed;
			}
		}
	}

	{
		tbMath::Vector2 northDirection(0.0f, -1.0f);
		tbMath::Vector2 eastDirection(1.0f, 0.0f);

		if (true == mIsMovingNorth)
		{
			currentPosition += northDirection * movementSpeed;
		}
		if (true == mIsMovingSouth)
		{
			currentPosition -= northDirection * movementSpeed;
		}
		if (true == mIsMovingEast)
		{
			currentPosition += eastDirection * movementSpeed;
		}
		if (true == mIsMovingWest)
		{
			currentPosition -= eastDirection * movementSpeed;
		}
	}


	mFighterSprite.SetRotation(currentRotation);

	tbMath::Vector2 towardFighter(currentPosition);
	float distanceFromCenter = towardFighter.Normalize();
	if (distanceFromCenter > ArenaEntity::kArenaRadius)
	{
		currentPosition = towardFighter * ArenaEntity::kArenaRadius;
	}

	SetPosition(currentPosition);

	//
	//Attacks
	//
	if (true == mCanAttack)
	{
		if (true == mIsAttacking)
		{
			mAttackTimer.IncrementStep();

			if (mAttackTimer.GetElapsedTime() > kPowerAttackTime.GetElapsedTime())
			{	//Then Power Attack
				mSwordSprite.PlayAnimation("stab_attack");
				mOnAttackDelayTimer = kAttackDelayTime;

				mAttackTimer = 0;
				mCanAttack = false;
			}
		}
		else
		{
			if (false == mAttackTimer.IsZero())
			{	//Then Quick Attack
				mSwordSprite.PlayAnimation("swing_attack");
				mOnAttackDelayTimer = kAttackDelayTime;

				mCanAttack = false;
			}
		}
	}
	else
	{
		if (false == mIsAttacking)
		{
			mAttackTimer = 0;
		}
	}

	if (false == mOnAttackDelayTimer.IsZero())
	{
		//TODO: TIM: TurtleBrains: DecrementStep should only return true if not already at 0.
		if (mOnAttackDelayTimer.DecrementStep())
		{
			bool powerAttack = (mSwordSprite.IsAnimationPlaying("stab_attack")) ? true : false;
			OnAttackDamage(powerAttack);
		}
	}
	else
	{
		if (false == mIsAttacking)
		{
			mCanAttack = true;
		}
	}

	//
	// ShapeShift
	//
	if (true == mCanShapeShift)
	{
		if (true == mAttemptShapeShift)
		{
			OnBeginShapeShift();
		}

		mAttemptShapeShift = false;
	}
	else
	{
		mAttemptShapeShift = false;
	}

	if (false == mFighterSprite.IsAnimationPlaying())
	{
		mFighterSprite.PlayAnimation("idle", true);
	}
	if (false == mSwordSprite.IsAnimationPlaying())
	{
		mSwordSprite.PlayAnimation("idle", true);
	}

	//These get reset during collisions.
	mCanShapeShift = false;
	mIsOnArena = false;
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnUpdate(const float deltaTime)
{
	tbGame::Entity::OnUpdate(deltaTime);

	mSwordSprite.SetRotation(mFighterSprite.GetRotation());

	tbMath::Vector2 hpPosition(GetForwardDirection() * -50.0f);
	mHealthBar.SetPosition(-32.0f + hpPosition.x, hpPosition.y);
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnRender(void) const
{
	tbGame::Entity::OnRender();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnCollideWith(tbGame::Entity& otherEntity)
{
	tbGame::Entity::OnCollideWith(otherEntity);

	if (otherEntity.IsEntityOfType("ShifterEntity"))
	{
		mCanShapeShift = true;
	}

	if (otherEntity.IsEntityOfType("ArenaEntity"))
	{
		mIsOnArena = true;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::AutoLockOnTarget(void)
{
	FighterEntity* nearestEnemy = GetNearestEnemy(nullptr);
	if (nullptr != nearestEnemy)
	{
		mTargetPosition = nearestEnemy->GetPosition();
		mIsLockedOnTarget = true;
	}
	else
	{
		mTargetPosition = tbMath::Vector2::kZero;
		mIsLockedOnTarget = true;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

LudumDare35::FighterEntity* LudumDare35::FighterEntity::GetNearestEnemy(float *returnDistance, float degreesFromForward) const
{
	tbGame::EntityType enemyType = "Enemy";
	if (IsEntityOfType(enemyType))
	{
		enemyType = "Hero";
	}

	///////////////////
	const tbMath::Vector2 position(GetPosition());
	const tbMath::Vector2 forward(GetForwardDirection());
	const tbMath::Vector3 forward3(forward.x, 0.0f, forward.y);

	FighterEntity* nearestEnemy = nullptr;
	float nearestEnemyDistance = INFINITY;

	std::list<Entity*> enemies = GetEntityManager()->GetEntitiesByType(enemyType);
	for (Entity* enemy : enemies)
	{
		tbMath::Vector2 enemyPosition(enemy->GetPosition());
		tbMath::Vector2 toEnemy(enemyPosition - position);
		float distanceToEnemy = toEnemy.Normalize();

		tbMath::Vector3 toEnemy3(toEnemy.x, 0.0f, toEnemy.y);
		float angle = tbMath::Convert::RadiansToDegrees(tbMath::Vector3AngleBetween(&forward3, &toEnemy3));

		if (distanceToEnemy < nearestEnemyDistance && (degreesFromForward < 0.0f || angle < degreesFromForward))
		{
			FighterEntity* thisEnemy = dynamic_cast<FighterEntity*>(enemy);
			if (nullptr != thisEnemy && true == thisEnemy->IsAlive() && false == thisEnemy->IsSpawning())
			{
				nearestEnemy = thisEnemy;
				nearestEnemyDistance = distanceToEnemy;
			}
		}
	}

	if (nullptr != returnDistance)
	{
		*returnDistance = nearestEnemyDistance;
	}
	return nearestEnemy;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbMath::Vector2 LudumDare35::FighterEntity::GetForwardDirection(void) const
{
	tbMath::Vector2 forwardDirection;
	tbMath::OrientationToForwardVector2(forwardDirection, tbMath::Convert::DegreesToRadians(mFighterSprite.GetRotation()));
	return forwardDirection;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbMath::Vector2 LudumDare35::FighterEntity::GetRightDirection(void) const
{
	tbMath::Vector2 rightDirection;
	tbMath::OrientationToForwardVector2(rightDirection, tbMath::Convert::DegreesToRadians(mFighterSprite.GetRotation() + 90.0f));
	return rightDirection;
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::PrepareAttack(bool powerAttack)
{
	const tbCore::tbString animationName((true == powerAttack) ? "ready_stab_attack" : "ready_swing_attack");

	if (false == mSwordSprite.IsAnimationPlaying(animationName))
	{
		mSwordSprite.PlayAnimation(animationName);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnBeginShapeShift(void)
{
	tbAudio::theAudioManager.PlayEvent("audio_events", "shape_shift");

	switch (mFighterShape)
	{
	case kFighterSquare: mFighterShape = kFighterCircle; break;
	case kFighterCircle: mFighterShape = kFighterTriangle; break;
	case kFighterTriangle: mFighterShape = kFighterSquare; break;
	}

	SetupSprite();
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::OnAttackDamage(bool powerAttack)
{
	float nearestEnemyDistance;
	FighterEntity* nearestEnemy = GetNearestEnemy(&nearestEnemyDistance, kMaximumForwardAttackRange);
	
	const float maximumAttackDistance(kMaximumAttackDistance[(true == powerAttack) ? 1 : 0]);

	if (nullptr != nearestEnemy && nearestEnemyDistance < maximumAttackDistance + nearestEnemy->GetRadius())
	{
		int damage = kVersusTable[mFighterShape][nearestEnemy->mFighterShape]->mDamage;
		int finalDamage((true == powerAttack) ? damage + (damage / 2) : damage);
		nearestEnemy->ApplyDamage(finalDamage);

		const float kickBackAmount(kVersusTable[mFighterShape][nearestEnemy->mFighterShape]->mPushBack);
		const float finalKickBackAmount((true == powerAttack) ? kickBackAmount + (kickBackAmount / 4) : kickBackAmount);
		
		tbGame::GameTimer stunTimer(kVersusTable[mFighterShape][nearestEnemy->mFighterShape]->mStunTimer);
		nearestEnemy->mStunnedTimer = (true == powerAttack) ? (stunTimer.GetElapsedTime() + (stunTimer.GetElapsedTime() / 2)) : stunTimer;
		
		tbMath::Vector2 towardNearest(nearestEnemy->GetPosition() - GetPosition());
		nearestEnemy->SetPosition(nearestEnemy->GetPosition() + (towardNearest.GetNormalized() * finalKickBackAmount));

		tbCore::tbString audioEvent = kVersusTable[mFighterShape][nearestEnemy->mFighterShape]->mHitName + "_hit";
		tbAudio::theAudioManager.PlayEvent("audio_events", audioEvent);
	}
	else
	{
		tbAudio::theAudioManager.PlayEvent("audio_events", "miss");
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::ApplyDamage(int damageAmount)
{
	mHealthBar.Hurt(damageAmount);

	if (false == IsEntityOfType("Player"))
	{
		const tbMath::Vector2 randomPosition = ArenaEntity::RandomPositionNear(GetPosition(), 64.0f);
		GetEntityManager()->AddEntity(new ScoreEntity(randomPosition, damageAmount));
		GameResults::IncrementCurrentScore(damageAmount);
	}
	else
	{
		tbGraphics::GetActiveCamera().ShakeCamera(tbGraphics::Camera::kLightShaking);
	}

	if (false == mHealthBar.IsAlive())
	{
		mFighterSprite.PlayAnimation("death");
		mSwordSprite.PlayAnimation("death");

		if (false == IsEntityOfType("Player"))
		{
			PushBehavior(new tbGame::KillBehavior(*this));
			PushBehavior(new tbGame::DelayBehavior(*this, tbGame::GameTimer(2500)));

			tbGraphics::GetActiveCamera().ShakeCamera(tbGraphics::Camera::kLightShaking);

			//Random Drop
			if (tbMath::RandomInt() % 100 >= (100 - kLootDropPercentage))
			{
				GetEntityManager()->AddEntity(new PickupEntity(GetPosition(), false));
			}
		}
		else
		{
			tbGraphics::GetActiveCamera().ShakeCamera(tbGraphics::Camera::kHeavyShaking);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void LudumDare35::FighterEntity::SetupSprite(void)
{
	static std::vector<tbGraphics::AnimatedSprite> shapeSprites;
	shapeSprites.push_back(tbGraphics::AnimatedSprite("earth_fighter_sheet", "earth_fighter"));
	shapeSprites.push_back(tbGraphics::AnimatedSprite("water_fighter_sheet", "water_fighter"));
	shapeSprites.push_back(tbGraphics::AnimatedSprite("fire_fighter_sheet", "fire_fighter"));

	mFighterSprite = shapeSprites[mFighterShape];
	mFighterSprite.SetOrigin(tbGraphics::kAnchorCenter);

	mSwordSprite.SetOrigin(tbGraphics::kAnchorCenter);
}

//--------------------------------------------------------------------------------------------------------------------//
