///
/// @file
/// @details An entity within the LudumDare35 project.
///
/// <!-- Copyright (c) 2016 Tim Beaudet - All Rights Reserved -->
///-----------------------------------------------------------------------------------------------------------------///

#ifndef _LudumDare35_FighterEntity_h_
#define _LudumDare35_FighterEntity_h_

#include "ludumdare35.h"
#include "health_bar.h"

namespace LudumDare35
{

	enum FighterShape
	{
		kFighterSquare,
		kFighterEarth = kFighterSquare,

		kFighterCircle,
		kFighterWater = kFighterCircle,

		kFighterTriangle,
		kFighterFire = kFighterTriangle,
	};

	struct VersusInfo
	{
		int mDamage;
		tbCore::tbString mHitName;
		tbGame::GameTimer mStunTimer;
		float mPushBack;
	};

	extern const VersusInfo* kVersusTable[3][3];

	//To find easy/hard enemy do: table[fighter][0] for easy and table[fighter][2] for hard.
	extern const FighterShape kFighterTable[3][3];

	class FighterEntity : public tbGame::Entity
	{
	public:
		static const tbGame::GameTimer kPowerAttackTime;
		static const float kMaximumAttackDistance[2];
		static const float kMaximumForwardAttackRange; //Angle in degrees left or right of forward vector.

		FighterEntity(const tbMath::Vector2& spawnPosition, const FighterShape& fighterShape = kFighterSquare);
		virtual ~FighterEntity(void) = 0;

		bool IsSpawning(void) const { return mSpawnTimer.IsZero() ? false : true; }
		bool IsAlive(void) const;
		
		float GetRadius(void) const { return 32.0f; }

		void ArtificialHackSetSpawnTimer(const tbGame::GameTimer& timer) { mSpawnTimer = timer; }
		void ArtificialHackMoveForward(const tbGame::GameTimer& timer, float speed) { mMoveBackwardTimer = tbGame::GameTimer::kZero; mMoveForwardTimer = timer; mMovementSpeed = speed; }
		void ArtificialHackMoveBackward(const tbGame::GameTimer& timer, float speed) { mMoveForwardTimer = tbGame::GameTimer::kZero; mMoveBackwardTimer = timer; mMovementSpeed = speed; }

		HealthBar& GetHealthBar(void) { return mHealthBar; }
		const tbMath::Vector2 GetForwardDirection(void) const;
		const tbMath::Vector2 GetRightDirection(void) const;
		const FighterShape& GetFighterShape(void) const { return mFighterShape; }

	protected:
		bool mIsLockedOnTarget;
		tbMath::Vector2 mTargetPosition;
		bool mIsWalkingForward;
		bool mIsWalkingBackward;
		bool mIsStrafingLeft;
		bool mIsStrafingRight;
		bool mIsAttacking;
		bool mAttemptShapeShift;

		bool mIsMovingNorth;
		bool mIsMovingEast;
		bool mIsMovingSouth;
		bool mIsMovingWest;

		float mWalkingSpeed;
		float mTurningSpeed;

		virtual void OnAdded(void) override;
		virtual void OnRemoved(void) override;
		virtual void OnSimulate(void) override;
		virtual void OnUpdate(const float deltaTime) override;
		virtual void OnRender(void) const override;
		virtual void OnCollideWith(tbGame::Entity& otherEntity) override;

		void AutoLockOnTarget(void);
		FighterEntity* GetNearestEnemy(float *returnDistance, float degreesFromForward = -1.0f) const;

		void PrepareAttack(bool powerAttack);

	private:
		static const tbGame::GameTimer kAttackDelayTime;

		void OnBeginShapeShift(void);
		void OnAttackDamage(bool powerAttack);
		void SetupSprite(void);

		void ApplyDamage(int damageAmount);

		tbGraphics::AnimatedSprite mFighterSprite;
		tbGraphics::AnimatedSprite mSwordSprite;

		FighterShape mFighterShape;
		bool mCanAttack;
		bool mCanShapeShift;
		bool mIsOnArena;
		bool mIsStunned;
		tbGame::GameTimer mStunnedTimer;
		tbGame::GameTimer mAttackTimer;
		tbGame::GameTimer mOnAttackDelayTimer;
		tbGame::GameTimer mShapeShiftTimer;
		tbGame::GameTimer mSpawnTimer;

		tbGame::GameTimer mMoveForwardTimer;
		tbGame::GameTimer mMoveBackwardTimer;
		float mMovementSpeed;
		HealthBar mHealthBar;
	};

}; /* namespace LudumDare35 */

#endif /* _LudumDare35_FighterEntity_h_ */
