///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_macosx

#include "../tbi_system_application_dialog.h"
#include "../../tb_application_dialog.h"
#include "../../tb_application_handler_interface.h"
#include "../../../core/tb_defines.h"
#include "../../../core/tb_error.h"

#import <AppKit/NSPanel.h>
#import <AppKit/NSButton.h>
#import <Appkit/NSTextField.h>
#import <AppKit/NSTextView.h>
#import <AppKit/NSComboBox.h>
#import <AppKit/NSScrollView.h>

//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	extern tbApplication::ApplicationHandlerInterface* tbiApplicationHandler;	//This exists in tbi_realtime_mac_application.cpp

	struct DialogWindowInformation
	{
		tbApplication::DialogIdentifier mDialogIdentifier;
		NSPanel* mDialogWindow;
	};

	typedef std::map<tbApplication::DialogIdentifier, DialogWindowInformation> DialogTable;
	DialogTable tbiDialogTable;

	NSPanel* CreateDialogWindow(const tbApplication::DialogIdentifier& dialogIdentifier);
	void CreateDialogControl(const DialogControl& dialogControl, const tbApplication::DialogIdentifier& dialogIdentifier,
													 NSPanel* dialogWindow);

	NSString* ToCocoaString(const std::wstring& input);
	NSString* ToCocoaString(const std::string& input);
	std::wstring ToWideString(NSString* input);

	//To safely dynamic-cast objective-c objects.
	template<typename T> inline T* objc_dynamic_cast(id object)
	{
    return ([object isKindOfClass:[T class]]) ? static_cast<T*>(object) : nil;
	}
	
};	/* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

NSString* tbImplementation::ToCocoaString( const std::wstring& input )
{
	return [[[NSString alloc] initWithBytes:input.c_str()
																	 length:input.size() * sizeof(wchar_t)
																 encoding:NSUTF32LittleEndianStringEncoding] autorelease];
}

//--------------------------------------------------------------------------------------------------------------------//

NSString* tbImplementation::ToCocoaString(const std::string& input)
{
	return ToCocoaString( tbCore::ToWideString(input) );
}

//--------------------------------------------------------------------------------------------------------------------//

std::wstring tbImplementation::ToWideString(NSString* input) {
  NSStringEncoding pEncode = CFStringConvertEncodingToNSStringEncoding(kCFStringEncodingUTF32LE);
  NSData* stringData = [input dataUsingEncoding : pEncode];
  return std::wstring(static_cast<const wchar_t*>([stringData bytes]), [stringData length] / sizeof(wchar_t));
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

@interface ControlActionDelegator : NSObject <NSTextFieldDelegate, NSComboBoxDelegate>
{
	int mControlIdentifier;
  int mDialogIdentifier;

	NSControl* mControl;
}
- (void)SetIdentifier:(int)dialogIdentifier :(int)controlIdentifier;
- (void)SetControl:(NSControl*)control;
- (void)PerformButtonAction:(id)sender;
@end

@implementation ControlActionDelegator

//--------------------------------------------------------------------------------------------------------------------//

- (void) SetIdentifier:(int)dialogIdentifier :(int)controlIdentifier
{
  mDialogIdentifier = dialogIdentifier;
	mControlIdentifier = controlIdentifier;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)SetControl:(NSControl*)control
{
	mControl = control;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)controlTextDidEndEditing:(NSNotification *)notification
{
	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mDialogIdentifier, mControlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbInternalError: Expected to find a dialog control (%d) for dialog (%d)", mControlIdentifier, mDialogIdentifier);

	if (tbImplementation::kTextControl == dialogControl->mType)
	{
		NSTextField *textField = [notification object];
		dialogControl->mValue = tbCore::ToString(tbImplementation::ToWideString([textField stringValue]));
		
		tbImplementation::tbiApplicationHandler->OnDialogAction(mDialogIdentifier, mControlIdentifier);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) PerformButtonAction:(id)sender
{
	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mDialogIdentifier, mControlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbInternalError: Expected to find a dialog control (%d) for dialog (%d)", mControlIdentifier, mDialogIdentifier);

	if (tbImplementation::kCheckboxControl == dialogControl->mType)
	{	//Mac seems to already change the state of the button.
		NSButton* checkboxControl = tbImplementation::objc_dynamic_cast<NSButton>(mControl);
		tb_error_if(nil == checkboxControl, "tbInternalError: Expected mControl to be a type of NSButton.");
		tb_error_if(false == [mControl respondsToSelector:@selector(state)], "tbInternalError: Expected control to respond to selector.");

		NSInteger checkboxState = [checkboxControl state];
		bool isChecked = (NSOnState == checkboxState) ? true : false;
		tbCore::tbString controlName = tbImplementation::StringToCheckboxName(dialogControl->mValue);
		dialogControl->mValue = tbImplementation::CheckboxToString(isChecked, controlName);
	}

	tbImplementation::tbiApplicationHandler->OnDialogAction(mDialogIdentifier, mControlIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)comboBoxSelectionDidChange:(NSNotification *)notification
{
	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mDialogIdentifier, mControlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbInternalError: Expected to find a dialog control (%d) for dialog (%d)", mControlIdentifier, mDialogIdentifier);

	NSComboBox* dropdownControl = tbImplementation::objc_dynamic_cast<NSComboBox>(mControl);
	tb_error_if(nil == dropdownControl, "tbInternalError: Expected mControl to be a type of NSComboBox");
	tb_error_if(false == [mControl respondsToSelector:@selector(indexOfSelectedItem)], "tbInternalError: Expected control to respond to selector.");
	
	int selectedIndex = [dropdownControl indexOfSelectedItem];
	int throwAwayIndex = 0;
	std::vector<tbCore::tbString> values;
	tbImplementation::StringToDropdown(dialogControl->mValue, throwAwayIndex, values);
	dialogControl->mValue = tbImplementation::DropdownToString(selectedIndex, values);

	tbImplementation::tbiApplicationHandler->OnDialogAction(mDialogIdentifier, mControlIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)controlTextDidChange:(NSNotification *)notification
{
//	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mDialogIdentifier, mControlIdentifier);
//	tb_error_if(nullptr == dialogControl, "tbInternalError: Expected to find a dialog control (%d) for dialog (%d)", mControlIdentifier, mDialogIdentifier);
//
//	if (tbApplication::kTextControl == dialogControl->mType)
//	{
//		NSTextField *textField = [notification object];
//		dialogControl->mValue = tbCore::ToString(tbImplementation::ToWideString([textField stringValue]));
//	}
}

@end

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OpenDialog(const tbApplication::DialogIdentifier& dialogIdentifier,
	tbApplication::ApplicationHandlerInterface& applicationHandler)
{
	DialogTable::iterator dialogItr = tbiDialogTable.find(dialogIdentifier);
	tb_error_if(dialogItr != tbiDialogTable.end(), "tbExternalError: OpenDialog can't open existing dialog: %d", dialogIdentifier);

	NSPanel* dialogWindow = CreateDialogWindow(dialogIdentifier);
	tbiDialogTable[dialogIdentifier].mDialogWindow = dialogWindow;
	tbiDialogTable[dialogIdentifier].mDialogIdentifier = dialogIdentifier;

	const size_t numberOfControls = NumberOfControlsForDialog(dialogIdentifier);
	for (size_t controlIndex = 0; controlIndex < numberOfControls; ++controlIndex)
	{
		const DialogControl* dialogControl = GetDialogControlByIndex(dialogIdentifier, controlIndex);
		CreateDialogControl(*dialogControl, dialogIdentifier, dialogWindow);
	}
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CloseDialog(const tbApplication::DialogIdentifier& dialogIdentifier,
	tbApplication::ApplicationHandlerInterface& applicationHandler)
{
	DialogTable::iterator dialogItr = tbiDialogTable.find(dialogIdentifier);
	tb_error_if(dialogItr == tbiDialogTable.end(), "tbInternalError: CloseDialog found dialog did not exist: %d", dialogIdentifier);

	DialogWindowInformation& dialogInformation = dialogItr->second;
	[dialogInformation.mDialogWindow close];

	tbiDialogTable.erase(dialogItr);
}

//--------------------------------------------------------------------------------------------------------------------//

NSPanel* tbImplementation::CreateDialogWindow(const tbApplication::DialogIdentifier& dialogIdentifier)
{
	//tb_error_if(, "tbInternalError: OpenDialog cannot be called without a valid window.");

	AreaRectangle dialogArea;
	int filledHeight(0);
	GetDialogArea(dialogArea, filledHeight, dialogIdentifier);
	//const bool isScrolling((filledHeight > dialogArea.mHeight) ? true : false);

  NSPanel* dialogPanel = [[NSPanel alloc] init];
  NSRect frame = [dialogPanel frame];
  frame.size.width = dialogArea.mWidth;
  frame.size.height = dialogArea.mHeight;
  [dialogPanel setFrame:frame display:YES];
	[dialogPanel setTitle:@"TurtleBrains Dialog"];
	[dialogPanel setStyleMask:NSTitledWindowMask];

//	NSRect contentFrame = [dialogPanel frame];
//	contentFrame.size.height = dialogFilledHeight;
//
//	NSScrollView *scrollView = [[NSScrollView alloc] initWithFrame:[[dialogPanel contentView] frame]];
//	[scrollView setBorderType:NSNoBorder];
//	[scrollView setHasVerticalScroller:YES];
////	[[scrollView contentView] setCopiesOnScroll:false];
//
//	[scrollView setDocumentView:[[NSView alloc] initWithFrame:contentFrame]];
//	//	[scrollView setDocumentView:outletToCustomViewLoadedFromNib];
//
////	[scrollView setAutoresizingMask:NSViewHeightSizable];
//
//	[dialogPanel setContentView:scrollView];
	
  [[NSApplication sharedApplication] setNextResponder:dialogPanel];
	[dialogPanel retain];

	[dialogPanel makeKeyAndOrderFront:nil];

	return dialogPanel;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CreateDialogControl(const tbImplementation::DialogControl& dialogControl,
	const tbApplication::DialogIdentifier& dialogIdentifier, NSPanel* dialogWindow)
{
	static NSControl* hack = nil;		//Attempting to allow user to tab between controls.
	
	AreaRectangle controlArea;
	GetDialogControlArea(controlArea, dialogControl);

	NSRect dialogFrame = [dialogWindow frame];	//TODO: TIM: Figure out how not to leave 42 in here...
	NSRect controlFrame = NSMakeRect(controlArea.mLeft, dialogFrame.size.height - 42 - controlArea.mTop,
																	 controlArea.mWidth, controlArea.mHeight);
//	NSRect controlFrame = NSMakeRect(controlLeft, controlTop, controlWidth, kControlHeight);
	
	if (kButtonControl == dialogControl.mType)
	{
		NSButton *buttonControl = [[[NSButton alloc] initWithFrame:controlFrame] autorelease];
		[buttonControl setTitle:ToCocoaString(dialogControl.mValue)];
		[buttonControl setEnabled:dialogControl.mIsEnabled];
		[buttonControl setButtonType:NSMomentaryLightButton]; //Set what type button You want
		[buttonControl setBezelStyle:NSRoundedBezelStyle]; //Set what style You want

		ControlActionDelegator* actionDelegator = [[[ControlActionDelegator alloc] init] autorelease];
		[actionDelegator retain];
		[actionDelegator SetIdentifier:dialogIdentifier :dialogControl.mIdentifier];
		[actionDelegator SetControl:buttonControl];
		[buttonControl setTarget:actionDelegator];
		[buttonControl setAction:@selector(PerformButtonAction:)];

		[buttonControl setNextKeyView:hack];
		hack = buttonControl;
		[[dialogWindow contentView] addSubview: buttonControl];
	}
	else if (kCheckboxControl == dialogControl.mType)
	{
		bool isChecked = false;
		tbCore::tbString controlName;
		tbImplementation::StringToCheckbox(dialogControl.mValue, isChecked, controlName);

		NSButton *checkboxControl = [[[NSButton alloc] initWithFrame:controlFrame] autorelease];
		[checkboxControl setTitle:ToCocoaString(controlName)];
		[checkboxControl setEnabled:dialogControl.mIsEnabled];
		[checkboxControl setButtonType:NSSwitchButton];
		[checkboxControl setButtonType:NSMomentaryLightButton]; //Set what type button You want
		[checkboxControl setBezelStyle:NSRoundedBezelStyle]; //Set what style You want
		[checkboxControl setState:(true == isChecked) ? NSOnState : NSOffState];

		ControlActionDelegator* actionDelegator = [[[ControlActionDelegator alloc] init] autorelease];
		[actionDelegator retain];
		[actionDelegator SetIdentifier:dialogIdentifier :dialogControl.mIdentifier];
		[actionDelegator SetControl:checkboxControl];
		[checkboxControl setTarget:actionDelegator];
		[checkboxControl setAction:@selector(PerformButtonAction:)];

		[checkboxControl setNextKeyView:hack];
		hack = checkboxControl;
		[[dialogWindow contentView] addSubview: checkboxControl];
	}
	else if (kLabelControl == dialogControl.mType)
	{
		NSTextField* labelControl = [[[NSTextField alloc] initWithFrame:controlFrame] autorelease];
		[labelControl setStringValue:ToCocoaString(dialogControl.mValue)];
		[labelControl setEnabled:dialogControl.mIsEnabled];
		[labelControl setBezeled:NO];
		[labelControl setDrawsBackground:NO];
		[labelControl setEditable:NO];
		[labelControl setSelectable:NO];
		[[dialogWindow contentView] addSubview: labelControl];
	}
	else if (kTextControl == dialogControl.mType)
	{
		NSTextField* textControl = [[[NSTextField alloc] initWithFrame:controlFrame] autorelease];
		[textControl setEditable:true];
		[textControl setEnabled:true];
		[textControl setSelectable:true];
		[textControl setAllowsEditingTextAttributes:true];
		[textControl setStringValue:tbImplementation::ToCocoaString(dialogControl.mValue)];
		[textControl setEnabled:dialogControl.mIsEnabled];

		ControlActionDelegator* actionDelegator = [[[ControlActionDelegator alloc] init] autorelease];
		[actionDelegator retain];
		[actionDelegator SetIdentifier:dialogIdentifier :dialogControl.mIdentifier];
		[actionDelegator SetControl:textControl];
		[textControl setDelegate:actionDelegator];

		[textControl setNextKeyView:hack];
		hack = textControl;
		[[dialogWindow contentView] addSubview:textControl];
	}
	else if (kDropdownControl == dialogControl.mType)
	{
		NSComboBox* dropdownControl = [[NSComboBox alloc] initWithFrame:controlFrame];

		int selectedIndex = 0;
		std::vector<tbCore::tbString> values;
		tbImplementation::StringToDropdown(dialogControl.mValue, selectedIndex, values);
		for (size_t valueIndex = 0; valueIndex < values.size(); ++valueIndex)
		{
			[dropdownControl addItemWithObjectValue:ToCocoaString(values[valueIndex])];
		}

		[dropdownControl setNumberOfVisibleItems:3];

		[dropdownControl selectItemAtIndex:selectedIndex];
		[dropdownControl setEnabled:dialogControl.mIsEnabled];
		[dropdownControl setEditable:NO];

		ControlActionDelegator* actionDelegator = [[[ControlActionDelegator alloc] init] autorelease];
		[actionDelegator retain];
		[actionDelegator SetIdentifier:dialogIdentifier :dialogControl.mIdentifier];
		[actionDelegator SetControl:dropdownControl];
		[dropdownControl setDelegate:actionDelegator];

		[dropdownControl setNextKeyView:hack];
		hack = dropdownControl;
		[[dialogWindow contentView] addSubview:dropdownControl];
	}
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_macosx */
