
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_macosx

#include "../tbi_system_application_menu.h"
#include "../../../core/tb_defines.h"
#include "../../../core/tb_error.h"
#include "../../tb_application_menu.h"
#include "../../tb_application_handler_interface.h"

#include <map>

#import <AppKit/NSMenu.h>
#import <AppKit/NSButton.h>

namespace tbImplementation
{

	typedef std::map<tbApplication::MenuItemIdentifier, NSObject*> ItemActionContainer;
	typedef std::map<tbApplication::ApplicationHandlerInterface*, ItemActionContainer > MenuItemContainer;
	MenuItemContainer tbiActionsForWindowMenu;
	MenuItemContainer tbiActionsForContextMenu;
  NSMenu* CreateOsxMenuFromItems(const tbImplementation::MenuItem& parentItem, tbApplication::ApplicationHandlerInterface& menuHandler,
																 NSMenu* parentMenu, MenuItemContainer& itemActions);

	NSMenu* tbiContextMenu = nil;

	//Defined in tbi_mac_application_dialog.cpp for now...
	NSString* ToCocoaString(const std::wstring& input);
	NSString* ToCocoaString(const std::string& input);

};  /* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

@interface MenuActionDelegator : NSObject
{
  tbApplication::MenuItemIdentifier mIdentifier;
	tbApplication::ApplicationHandlerInterface* mApplicationHandler;
}
- (void)SetIdentifier:(tbApplication::MenuItemIdentifier)identifier;
- (void)SetApplicationHandler:(tbApplication::ApplicationHandlerInterface*)applicationHandler;
- (void)PerformMenuAction:(id)sender;
@end

@implementation MenuActionDelegator

//--------------------------------------------------------------------------------------------------------------------//

- (void) SetIdentifier:(tbApplication::MenuItemIdentifier)identifier
{
  mIdentifier = identifier;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) SetApplicationHandler:(tbApplication::ApplicationHandlerInterface *)applicationHandler
{
	mApplicationHandler = applicationHandler;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) PerformMenuAction:(id)sender
{
  tb_unused(sender);
	//TODO: TIM:Should pass the menu identifier for parameter 0
	mApplicationHandler->OnMenuAction(0, mIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//

@end


//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowMenu(const tbApplication::ApplicationMenu &menu, tbApplication::ApplicationHandlerInterface& menuHandler)
{
	//  NSMenuItem* menuItem;
	//  menuItem = [[NSMenuItem alloc] initWithTitle:@"Turtles" action:@selector(performMiniaturize:) keyEquivalent:@""];
	//
	//  [menuItem setSubmenu:TurtleBrainsPrivate::CreateOsxMenuFromItems(menu, NULL)];
	//  [[NSApp windowsMenu] addItem:menuItem];


	//  [NSApp setWindowsMenu:TurtleBrainsPrivate::CreateOsxMenuFromItems(menu, NULL)];


  //Create and set the application menu.
//  [[NSApplication sharedApplication] setMenu:TurtleBrainsPrivate::CreateOsxMenuFromItems(menu, menuHandler, NULL)];

	WindowInformation windowInformation;
	windowInformation.mApplication = NSApp;
	tbImplementation::SetWindowMenu(windowInformation, menu, menuHandler);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowMenu(WindowInformation& windowInformation, const tbApplication::ApplicationMenu& menu, tbApplication::ApplicationHandlerInterface& menuHandler)
{
	const MenuItem& primaryMenuItem(GetMenuFor(menu.GetIdentifier()));
	[windowInformation.mApplication setMenu:tbImplementation::CreateOsxMenuFromItems(primaryMenuItem, menuHandler, NULL, tbiActionsForWindowMenu)];
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CleanupWindowMenu(tbApplication::ApplicationHandlerInterface& menuHandler)
{
	MenuItemContainer::iterator menuIterator = tbiActionsForWindowMenu.find(&menuHandler);
	tb_error_if(tbiActionsForWindowMenu.end() == menuIterator, "tbError: Cannot cleanup a menu window that does not exist.");

	//All action delegates are auto released, so this shouldn't be leaking any memory.
	tbiActionsForWindowMenu.erase(menuIterator);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetContextMenu(const TurtleBrains::Application::ApplicationMenu &menu,
																			TurtleBrains::Application::ApplicationHandlerInterface &menuHandler)
{
	WindowInformation windowInformation;
	windowInformation.mApplication = NSApp;
	tbImplementation::SetContextMenu(windowInformation, menu, menuHandler);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetContextMenu(WindowInformation& windowInformation, const TurtleBrains::Application::ApplicationMenu &menu,
																			TurtleBrains::Application::ApplicationHandlerInterface &menuHandler)
{
	const MenuItem& primaryMenuItem(GetMenuFor(menu.GetIdentifier()));
	tbiContextMenu = tbImplementation::CreateOsxMenuFromItems(primaryMenuItem, menuHandler, nullptr, tbiActionsForContextMenu);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CleanupContextMenu(TurtleBrains::Application::ApplicationHandlerInterface &menuHandler)
{
	if (nil == tbImplementation::tbiContextMenu)
	{
		[tbImplementation::tbiContextMenu release];
		tbImplementation::tbiContextMenu = nil;
	}
}

//--------------------------------------------------------------------------------------------------------------------//

NSMenu* tbImplementation::CreateOsxMenuFromItems(const tbImplementation::MenuItem& parentItem, tbApplication::ApplicationHandlerInterface& menuHandler,
																								 NSMenu* parentMenu, MenuItemContainer& itemActions)
{
  tb_unused(parentMenu);

	//const size_t menuSize(menu.Size());
	const size_t menuSize(parentItem.mSubItems.size());

	if (0 == menuSize)
	{
		return NULL;
	}

  NSMenu* osxMenu = [[NSMenu alloc] init];
  NSMenuItem* osxMenuItem = NULL;

	for (size_t itemIndex = 0; itemIndex < menuSize; ++itemIndex)
		//	for (size_t itemIndex = menuSize - 1; itemIndex < menuSize; --itemIndex)
	{
		//const tbApplication::ApplicationMenu& menuItem = menu.GetItemAtIndex(itemIndex);
		const MenuItem& menuItem = parentItem.mSubItems[itemIndex];

		if (false == menuItem.mIsVisible)
		{	//Skip this item, it is invisible.
			continue;
		}

    const std::wstring& displayName(tbCore::ToWideString(menuItem.mDisplayName));
    if (true == displayName.empty())
    {
      [osxMenu addItem:[NSMenuItem separatorItem]];
      continue;
    }

    osxMenuItem = [[NSMenuItem alloc] init];

		//Find, or build an action for this Menu/Item combination.
		MenuItemContainer::iterator menuIterator = itemActions.find(&menuHandler);
		if (itemActions.end() == menuIterator)
		{
			menuIterator = itemActions.insert(MenuItemContainer::value_type(&menuHandler, ItemActionContainer())).first;
		}

		tb_error_if(itemActions.end() == menuIterator, "tbInternalError: Expected menuIterator to be valid.");

		ItemActionContainer::iterator actionIterator = menuIterator->second.find(menuItem.mItemIdentifier);
		if (menuIterator->second.end() == actionIterator)
		{
			MenuActionDelegator* itemAction = [[MenuActionDelegator alloc] init];
			[itemAction SetIdentifier:menuItem.mItemIdentifier];
			[itemAction SetApplicationHandler:&menuHandler];
			
			actionIterator = menuIterator->second.insert(ItemActionContainer::value_type(menuItem.mItemIdentifier, itemAction)).first;
		}

		tb_error_if(menuIterator->second.end() == actionIterator, "tbInternalError: Expected actionIterator to be valid.");

    [osxMenuItem setTarget:actionIterator->second];
    [osxMenuItem setAction:@selector(PerformMenuAction:)];
    [osxMenuItem setTitle:ToCocoaString(displayName)];
    [osxMenuItem setEnabled:menuItem.mIsEnabled];  //Doesn't actually work as expected. -TIM 11/25/13
    [osxMenuItem setState:(true == menuItem.mIsChecked) ? NSOnState : NSOffState];

		if (false == menuItem.mSubItems.empty())
		{
      [osxMenuItem setSubmenu:CreateOsxMenuFromItems(menuItem, menuHandler, osxMenu, itemActions)];
      [[osxMenuItem submenu] setTitle:[osxMenuItem title]];
		}
    [osxMenu addItem:osxMenuItem];
	}

	return osxMenu;
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_macosx */
