
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_macosx

#include "../tbi_realtime_system_application.h"
#include "../../tb_realtime_application.h"
#include "../../../core/tb_defines.h"
#include "../../../core/tb_error.h"
#include "../../../core/tb_opengl.h"
#include "../../tb_application_handler_interface.h"
#include "../../tb_application_window.h"
#include "../tbi_system_application_input.h"

#include "../../../graphics/implementation/tbi_renderer.h"

#include <map>

#import <AppKit/NSWindow.h>
#import <AppKit/NSColor.h>
#import <AppKit/NSOpenGLView.h>
#import <AppKit/NSOpenGL.h>
#import <Carbon/Carbon.h>

//--------------------------------------------------------------------------------------------------------------------//

@interface TurtleBrainsMain : NSObject<NSApplicationDelegate, NSWindowDelegate>
{
	NSWindow* mWindow;
}
@end

//--------------------------------------------------------------------------------------------------------------------//

@interface TurtleBrainsOpenGLView : NSOpenGLView
{
	tbApplication::ApplicationHandlerInterface* mApplicationHandler;
	NSTimer* mRealTimer;
}
@end

//--------------------------------------------------------------------------------------------------------------------//

namespace tbImplementation
{
	extern NSMenu* tbiContextMenu;
	TurtleBrainsMain* tbiMainDelegate = nil;
	NSRect windowBounds;
	NSAutoreleasePool* tbiMemoryPool = nil;
	tbApplication::ApplicationHandlerInterface* tbiApplicationHandler(nullptr);

	void FixWorkingDirectoryAsNeeded(void);

	//Defined in tbi_mac_application_dialog.cpp for now...
	NSString* ToCocoaString(const std::wstring& input);
	NSString* ToCocoaString(const std::string& input);
	std::wstring ToWideString(NSString* input);

	typedef std::map<unsigned long, tbApplication::Key> KeyTable;
	KeyTable tbiVirtualCodesToKey;

	void SetupKeyTable(void);

};  /* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::FixWorkingDirectoryAsNeeded(void)
{
	char cwdBuffer[FILENAME_MAX];
	getcwd(cwdBuffer, FILENAME_MAX);
	const tbCore::tbString currentDirectory(cwdBuffer);

	NSAutoreleasePool *pool = [[NSAutoreleasePool alloc] init];
	const tbCore::tbString workingDirectory(tbCore::ToString(tbImplementation::ToWideString([[NSBundle mainBundle] bundlePath])));
	[pool release];

	if (currentDirectory != workingDirectory && currentDirectory == tb_string("/") && tbCore::tbString::npos != workingDirectory.find(".app"))
	{
		//In my experience when in app workingDirectory is "/Some/Path/To/TheApplication.app"
		//The TurtleBrains packager script will put the data folder in TheApplication.app/Contents/Resources/"
		chdir((workingDirectory + "/Contents/Resources/").c_str());
	}
};

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetupKeyTable(void)
{
	tbiVirtualCodesToKey.clear();

	tbiVirtualCodesToKey[kVK_ANSI_0] = tbApplication::tbKey0;
	tbiVirtualCodesToKey[kVK_ANSI_1] = tbApplication::tbKey1;
	tbiVirtualCodesToKey[kVK_ANSI_2] = tbApplication::tbKey2;
	tbiVirtualCodesToKey[kVK_ANSI_3] = tbApplication::tbKey3;
	tbiVirtualCodesToKey[kVK_ANSI_4] = tbApplication::tbKey4;
	tbiVirtualCodesToKey[kVK_ANSI_5] = tbApplication::tbKey5;
	tbiVirtualCodesToKey[kVK_ANSI_6] = tbApplication::tbKey6;
	tbiVirtualCodesToKey[kVK_ANSI_7] = tbApplication::tbKey7;
	tbiVirtualCodesToKey[kVK_ANSI_8] = tbApplication::tbKey8;
	tbiVirtualCodesToKey[kVK_ANSI_9] = tbApplication::tbKey9;

	tbiVirtualCodesToKey[kVK_ANSI_A] = tbApplication::tbKeyA;
	tbiVirtualCodesToKey[kVK_ANSI_B] = tbApplication::tbKeyB;
	tbiVirtualCodesToKey[kVK_ANSI_C] = tbApplication::tbKeyC;
	tbiVirtualCodesToKey[kVK_ANSI_D] = tbApplication::tbKeyD;
	tbiVirtualCodesToKey[kVK_ANSI_E] = tbApplication::tbKeyE;
	tbiVirtualCodesToKey[kVK_ANSI_F] = tbApplication::tbKeyF;
	tbiVirtualCodesToKey[kVK_ANSI_G] = tbApplication::tbKeyG;
	tbiVirtualCodesToKey[kVK_ANSI_H] = tbApplication::tbKeyH;
	tbiVirtualCodesToKey[kVK_ANSI_I] = tbApplication::tbKeyI;
	tbiVirtualCodesToKey[kVK_ANSI_J] = tbApplication::tbKeyJ;
	tbiVirtualCodesToKey[kVK_ANSI_K] = tbApplication::tbKeyK;
	tbiVirtualCodesToKey[kVK_ANSI_L] = tbApplication::tbKeyL;
	tbiVirtualCodesToKey[kVK_ANSI_M] = tbApplication::tbKeyM;
	tbiVirtualCodesToKey[kVK_ANSI_N] = tbApplication::tbKeyN;
	tbiVirtualCodesToKey[kVK_ANSI_O] = tbApplication::tbKeyO;
	tbiVirtualCodesToKey[kVK_ANSI_P] = tbApplication::tbKeyP;
	tbiVirtualCodesToKey[kVK_ANSI_Q] = tbApplication::tbKeyQ;
	tbiVirtualCodesToKey[kVK_ANSI_R] = tbApplication::tbKeyR;
	tbiVirtualCodesToKey[kVK_ANSI_S] = tbApplication::tbKeyS;
	tbiVirtualCodesToKey[kVK_ANSI_T] = tbApplication::tbKeyT;
	tbiVirtualCodesToKey[kVK_ANSI_U] = tbApplication::tbKeyU;
	tbiVirtualCodesToKey[kVK_ANSI_V] = tbApplication::tbKeyV;
	tbiVirtualCodesToKey[kVK_ANSI_W] = tbApplication::tbKeyW;
	tbiVirtualCodesToKey[kVK_ANSI_X] = tbApplication::tbKeyX;
	tbiVirtualCodesToKey[kVK_ANSI_Y] = tbApplication::tbKeyY;
	tbiVirtualCodesToKey[kVK_ANSI_Z] = tbApplication::tbKeyZ;

	tbiVirtualCodesToKey[kVK_Space] = tbApplication::tbKeySpace;
	tbiVirtualCodesToKey[kVK_Escape] = tbApplication::tbKeyEscape;
	tbiVirtualCodesToKey[kVK_Return] = tbApplication::tbKeyEnter;
	tbiVirtualCodesToKey[kVK_UpArrow] = tbApplication::tbKeyUp;
	tbiVirtualCodesToKey[kVK_DownArrow] = tbApplication::tbKeyDown;
	tbiVirtualCodesToKey[kVK_LeftArrow] = tbApplication::tbKeyLeft;
	tbiVirtualCodesToKey[kVK_RightArrow] = tbApplication::tbKeyRight;
}

void tbImplementation::SetMousePosition(int mouseX, int mouseY)
{
	CGEventSourceRef source = CGEventSourceCreate(kCGEventSourceStateCombinedSessionState);

	CGPoint mousePoint = CGPointMake(mouseX, mouseY);

//	NSPoint locationInView = [self convertPoint:[theEvent locationInWindow] fromView:nil];

	NSRect bo = [[NSApp mainWindow] frame];
	mousePoint.y = -mousePoint.y + windowBounds.size.height;

	mousePoint.x += bo.origin.x;
	mousePoint.y += bo.origin.y - 20;  //Why is this here?  Obviously a magic number of some sort...

  CGEventRef mouse = CGEventCreateMouseEvent (NULL, kCGEventMouseMoved, mousePoint, 0);
  CGEventPost(kCGHIDEventTap, mouse);
  CFRelease(mouse);
  CFRelease(source);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

@implementation TurtleBrainsOpenGLView

- (void) SetApplicationHandler:(tbApplication::ApplicationHandlerInterface*)applicationHandler
{
	mApplicationHandler = applicationHandler;
}

//--------------------------------------------------------------------------------------------------------------------//

- (NSMenu*) menuForEvent:(NSEvent*)theEvent
{
	return tbImplementation::tbiContextMenu;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) Redraw:(NSTimer*)timer
{
	tbImplementation::windowBounds = [self bounds];
	[self drawRect:[self bounds]];
	tbImplementation::StartNewFrame();
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) drawRect:(NSRect)rect
{
	if (nullptr == mApplicationHandler)
	{
		tb_check_gl_errors(glClearColor(0.0f, 0.0f, 0.0f, 1.0f));
		tb_check_gl_errors(glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT));
	}
	else	//Allow the developer to render the scene.
	{
		mApplicationHandler->OnRealtimeUpdate();
	}

	[[self openGLContext] flushBuffer];

	tb_check_recent_gl_errors("drawRect");
}

//--------------------------------------------------------------------------------------------------------------------//

+ (NSOpenGLPixelFormat*) basicPixelFormat
{
	NSOpenGLPixelFormatAttribute attributes [] = {
//		NSOpenGLPFAWindow,
		NSOpenGLPFADoubleBuffer,    // double buffered
		NSOpenGLPFAAccelerated,
		NSOpenGLPFADepthSize, (NSOpenGLPixelFormatAttribute)24, // 24 bit depth buffer
		NSOpenGLPFABackingStore, YES,

#if defined(tb_legacy_gl_support) && defined(tb_legacy_gl_forced)
		NSOpenGLPFAOpenGLProfile, (NSOpenGLPixelFormatAttribute)NSOpenGLProfileVersionLegacy,
#else
		NSOpenGLPFAOpenGLProfile, (NSOpenGLPixelFormatAttribute)NSOpenGLProfileVersion3_2Core,
#endif /* tb_legacy_gl_support && tb_legacy_gl_forced */
		0
	};

	return [[NSOpenGLPixelFormat alloc] initWithAttributes:attributes];
}

//--------------------------------------------------------------------------------------------------------------------//

- (id) initWithFrame: (NSRect) frameRect
{
	if (self = [super initWithFrame:frameRect pixelFormat:[TurtleBrainsOpenGLView basicPixelFormat]])
	{
		GLint swapInt = 0;		//NSOpenGLCPSwapInterval  0 = no vsync (default)   1 = vsync
		[[self openGLContext] setValues:&swapInt forParameter:NSOpenGLCPSwapInterval];

//		mRealTimer = [NSTimer scheduledTimerWithTimeInterval:0.0001 target:self
//			mRealTimer = [NSTimer scheduledTimerWithTimeInterval:0.001 target:self
					mRealTimer = [NSTimer scheduledTimerWithTimeInterval:0.01 target:self
			selector:@selector(Redraw:) userInfo:nil repeats:true];
		[mRealTimer retain];
	}

	tb_check_recent_gl_errors("initWithFrame");
	return self;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)prepareOpenGL
{
//	int contextAttributes[] = {
//
//	};

	[super prepareOpenGL];
}

//--------------------------------------------------------------------------------------------------------------------//

-(BOOL)acceptsFirstResponder { return YES; }
-(BOOL)canBecomeKeyWindow { return YES; }
-(BOOL)canBecomeMainWindow { return YES; }

//--------------------------------------------------------------------------------------------------------------------//

- (void)mouseDown:(NSEvent*)theEvent
{
	tbImplementation::OnPressKey(tbApplication::tbMouseLeft);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)mouseUp:(NSEvent*)theEvent
{
	tbImplementation::OnReleaseKey(tbApplication::tbMouseLeft);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)rightMouseDown:(NSEvent*)theEvent
{
	tbImplementation::OnPressKey(tbApplication::tbMouseRight);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)rightMouseUp:(NSEvent*)theEvent
{
	tbImplementation::OnReleaseKey(tbApplication::tbMouseRight);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)otherMouseDown:(NSEvent*)theEvent
{
	tbImplementation::OnPressKey(tbApplication::tbMouseMiddle);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)otherMouseUp:(NSEvent*)theEvent
{
	tbImplementation::OnReleaseKey(tbApplication::tbMouseMiddle);
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)keyDown:(NSEvent *)theEvent
{
	if (false == [theEvent isARepeat])
	{
		tbImplementation::KeyTable::const_iterator itr = tbImplementation::tbiVirtualCodesToKey.find([theEvent keyCode]);
		if (itr != tbImplementation::tbiVirtualCodesToKey.end())
		{
			tbImplementation::OnPressKey(itr->second);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)keyUp:(NSEvent *)theEvent
{
	if (false == [theEvent isARepeat])
	{
		tbImplementation::KeyTable::const_iterator itr = tbImplementation::tbiVirtualCodesToKey.find([theEvent keyCode]);
		if (itr != tbImplementation::tbiVirtualCodesToKey.end())
		{
			tbImplementation::OnReleaseKey(itr->second);
		}
	}
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)mouseMoved:(NSEvent *)theEvent
{
	NSPoint locationInView = [self convertPoint:[theEvent locationInWindow] fromView:nil];
	tbImplementation::UpdateMousePosition(static_cast<int>(locationInView.x), static_cast<int>(-locationInView.y + [self bounds].size.height));
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)mouseDragged:(NSEvent *)theEvent
{
	NSPoint locationInView = [self convertPoint:[theEvent locationInWindow] fromView:nil];
	tbImplementation::UpdateMousePosition(static_cast<int>(locationInView.x), static_cast<int>(-locationInView.y + [self bounds].size.height));
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)rightMouseDragged:(NSEvent *)theEvent
{
	NSPoint locationInView = [self convertPoint:[theEvent locationInWindow] fromView:nil];
	tbImplementation::UpdateMousePosition(static_cast<int>(locationInView.x), static_cast<int>(-locationInView.y + [self bounds].size.height));
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)otherMouseDragged:(NSEvent *)theEvent
{
	NSPoint locationInView = [self convertPoint:[theEvent locationInWindow] fromView:nil];
	tbImplementation::UpdateMousePosition(static_cast<int>(locationInView.x), static_cast<int>(-locationInView.y + [self bounds].size.height));
}

//--------------------------------------------------------------------------------------------------------------------//

@end

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

@implementation TurtleBrainsMain

//--------------------------------------------------------------------------------------------------------------------//

- (id)init
{
	self = [super init];
	return self;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)dealloc
{
//	[mWindow release];
	[super dealloc];
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)OpenApplication
{
	tb_error_if(tbImplementation::tbiApplicationHandler == nullptr, "tbInternalError: Expected a valid applicationHandler at this point.");

	//Create and setup default window properties but allow the programmer to change them if desired.
	tbApplication::WindowProperties properties;
	tbImplementation::tbiApplicationHandler->CollectWindowProperties(properties);

	const NSRect frame = NSMakeRect(properties.mWindowPositionX, properties.mWindowPositionY, properties.mWindowWidth, properties.mWindowHeight);

	//TODO: TIM: If we are going to support resizable windows this needs to be changed...
	//const unsigned int windowStyle = NSTitledWindowMask | NSClosableWindowMask | NSMiniaturizableWindowMask | NSResizableWindowMask;
	const unsigned int windowStyle = NSTitledWindowMask | NSClosableWindowMask | NSMiniaturizableWindowMask;
	mWindow = [[NSWindow alloc] initWithContentRect:frame styleMask:windowStyle backing:NSBackingStoreBuffered defer:NO];

	[mWindow setTitle:@"TurtleBrains Project"];
	[mWindow setBackgroundColor:[NSColor grayColor]];
	[mWindow setDelegate:tbImplementation::tbiMainDelegate];
	[mWindow setAcceptsMouseMovedEvents:YES];

	tbImplementation::SetupKeyTable();

	TurtleBrainsOpenGLView* openglView = [[TurtleBrainsOpenGLView alloc] initWithFrame:frame];
	[openglView SetApplicationHandler:nullptr];
	[mWindow setContentView:openglView];

	glewExperimental = GL_TRUE;
	GLenum errorCode = glewInit();
	if (GLEW_OK != errorCode)
	{
		/* Problem: glewInit failed, something is seriously wrong. */
		tb_error("glewError: %s\n", glewGetErrorString(errorCode));
	}

	while (GL_NO_ERROR != (errorCode = glGetError())) {
		/* remove all errorCodes from gl before we start. */
	}

#if defined(tb_legacy_gl_support) && defined(tb_legacy_gl_forced)
	tbImplementation::Renderer::SetLegacyRenderer(true);
#endif /* tb_legacy_gl_support && tb_legacy_gl_forced */

	tbImplementation::Renderer::InitializeBasicRenderer();

	//An attempt at a status bar, being abandoned for now.
	//		[mWindow setContentBorderThickness:24.0 forEdge:NSMinYEdge];
	//		[mWindow setAutorecalculatesContentBorderThickness:false forEdge:NSMinYEdge];

	//Just before showing the window we let the end-programmer initialize things, possibly window menus.
	tbImplementation::tbiApplicationHandler->OnWindowOpen();

	[NSApp activateIgnoringOtherApps:YES];

	//Only after the first frame has been drawn allow the Realtime draw to start up.
	[openglView SetApplicationHandler:tbImplementation::tbiApplicationHandler];
}

//--------------------------------------------------------------------------------------------------------------------//

- (BOOL)applicationShouldTerminateAfterLastWindowClosed:(NSApplication *)theApplication
{
	return YES;
//	return NO;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)applicationWillFinishLaunching:(NSNotification *)notification
{
	[mWindow makeKeyAndOrderFront:self];
}

//--------------------------------------------------------------------------------------------------------------------//

- (void) applicationDidFinishLaunching: (NSNotification *) note
{
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)applicationWillTerminate:(NSNotification *)notification
{
//	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
//	tbImplementation::Renderer::CleanupBasicRenderer();
//	tbImplementation::tbiApplicationHandler->OnWindowClose();
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)windowWillClose:(NSNotification *)notification
{
	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
	tbImplementation::Renderer::CleanupBasicRenderer();
	tbImplementation::tbiApplicationHandler->OnWindowClose();

	mWindow = nil;
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)applicationDidBecomeActive:(NSNotification *)notification
//- (void)windowDidBecomeKey:(NSNotification *)notification
{
	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
	tbImplementation::tbiApplicationHandler->OnBecomeActive();
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)applicationDidResignActive:(NSNotification *)notification
//- (void)windowDidResignKey:(NSNotification *)notification
{
	tb_error_if(nullptr == tbImplementation::tbiApplicationHandler, "tbInternalError: Application Handler should be valid here.");
	tbImplementation::tbiApplicationHandler->OnBecomeInactive();
}

//--------------------------------------------------------------------------------------------------------------------//

- (void)SetWindowTitle:(NSString*)title
{
	tb_error_if(nullptr == mWindow, "tbInternalError: This cannot be called while the window is nil.");
	[mWindow setTitle:title];
}

//--------------------------------------------------------------------------------------------------------------------//


@end

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OpenRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(nullptr != tbiApplicationHandler, "tbExternalError: TurtleBrains currently supports only a single application at a time.");

	tbiApplicationHandler = &applicationHandler;

	FixWorkingDirectoryAsNeeded();

	tbiMemoryPool = [[NSAutoreleasePool alloc] init];
	[NSApplication sharedApplication];
	[NSApp setActivationPolicy:NSApplicationActivationPolicyRegular];

	tbiMainDelegate = [[TurtleBrainsMain alloc] init];
	[tbiMainDelegate retain];
	[tbiMainDelegate OpenApplication];
	[NSApp setDelegate:(id<NSFileManagerDelegate>)tbiMainDelegate];

	//The following was moved to run
//	[NSApp run];

	//The following was moved to close for cleanup.
//	[tbiMainDelegate release];
//	tbiMainDelegate = nil;
//
//	[tbiMemoryPool drain];
//	[tbiMemoryPool release]; //new from local to static tbi
//  tbiMemoryPool = nil;     //new
//
//	tbiApplicationHandler = nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::RunRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: ApplicationHandler attempted to change between Open and Run.");

	[NSApp run];
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CloseRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface &applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: ApplicationHandler attempted to change between Open and Close.");

	[NSApp terminate:nil];
	
	[tbiMainDelegate release];
	tbiMainDelegate = nil;
	tbiApplicationHandler = nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowTitle(const tbCore::tbString& windowTitle)
{
	tb_error_if(nullptr == tbiApplicationHandler, "tbExternalError: This cannot be called until after the Window/Application is opened.");
	tb_error_if(nullptr == tbiMainDelegate, "tbExternalError: This cannot be called until after the Window/Application is opened.");

	[tbiMainDelegate SetWindowTitle:ToCocoaString(windowTitle)];
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_macosx */
