///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved --> 
///-----------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_windows

#include "../tbi_realtime_system_application.h"
#include "../../tb_application_handler_interface.h"
#include "../tbi_system_application_input.h"
#include "../../tb_application_input.h"
#include "../../../core/tb_error.h"
#include "../../../core/tb_opengl.h"
#include "../../../core/implementation/tbi_diagnostics.h"
#include "../../../graphics/implementation/tbi_renderer.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <windowsx.h>  //For GET_X_LAPARAM

#include <map>

namespace tbImplementation
{
	LRESULT CALLBACK WindowProcedure(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam);

	bool tbiIsRunning(false);
	bool tbiHasApplication(false);
	HWND tbiWindowHandle(NULL);

	tbApplication::ApplicationHandlerInterface* tbiApplicationHandler(nullptr);
	
	bool HandleAsDialogMessage(MSG& message);	//Found in tbi_windows_application_dialog.cpp

	void SetupKeyTable(void);
	void FixWindowSize(HWND windowHandle, const int windowWidth, const int windowHeight);

	///
	/// 
	///
	void CheckAndFixWindowPosition(HWND windowHandle);

	//TODO: TIM: This is now shared on Windows and Linux, it should be refactored to a more common base.
	enum GLContextType { kContextCore, kContextCompatible, kContextES, };
	struct GLVersion
	{
		GLContextType mType;
		int mMajor;
		int mMinor;

		GLVersion(int major, int minor, const GLContextType& type) :
			mType(type),
			mMajor(major),
			mMinor(minor)
		{
		}
	};

	bool TryGLContext(const GLVersion& glVersion, HGLRC parentContext, HDC& hDC, HGLRC& hRC);
	void SetupOpenGL(const tbApplication::WindowProperties& windowProperties);
	void ClearOpenGL(void);

	typedef std::map<unsigned long, tbApplication::Key> KeyTable;
	KeyTable tbiVirtualCodesToKey;

	//tbApplication::WindowsApplication* hackeryApplication = nullptr;

};	/* namespace tbImplementation */

//--------------------------------------------------------------------------------------------------------------------//

namespace tbiWindows
{	//Definition for this is actually in tbi_windows_application_dialog.cpp currently.
	typedef std::basic_string<TCHAR> WindowsString;
	WindowsString ToWindowsString(const tbCore::tbString& input);
};

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetupKeyTable(void)
{
	for (int number = 0; number < 10; ++number)
	{
		tbiVirtualCodesToKey[0x30 + number] = static_cast<tbApplication::Key>(tbApplication::tbKey0 + number);
	}

	for (int letter = 0; letter < 26; ++letter)
	{
		tbiVirtualCodesToKey[0x41 + letter] = static_cast<tbApplication::Key>(tbApplication::tbKeyA + letter);
	}

	tbiVirtualCodesToKey[VK_SPACE] = tbApplication::tbKeySpace;
	tbiVirtualCodesToKey[VK_ESCAPE] = tbApplication::tbKeyEscape;
	tbiVirtualCodesToKey[VK_RETURN] = tbApplication::tbKeyEnter;
	tbiVirtualCodesToKey[VK_UP] = tbApplication::tbKeyUp;
	tbiVirtualCodesToKey[VK_DOWN] = tbApplication::tbKeyDown;
	tbiVirtualCodesToKey[VK_LEFT] = tbApplication::tbKeyLeft;
	tbiVirtualCodesToKey[VK_RIGHT] = tbApplication::tbKeyRight;

	//This doesn't work because WM_KEYDOWN doesn't process the mouse keys the same.
	//tbiVirtualCodesToKey[VK_LBUTTON] = tbApplication::tbMouseLeft;
	//tbiVirtualCodesToKey[VK_RBUTTON] = tbApplication::tbMouseRight;
	//tbiVirtualCodesToKey[VK_MBUTTON] = tbApplication::tbMouseMiddle;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::FixWindowSize(HWND windowHandle, const int windowWidth, const int windowHeight)
{
	RECT windowRect;
	RECT clientRect;
	GetClientRect(windowHandle, &clientRect);
	GetWindowRect(windowHandle, &windowRect);
	const long moreWidth = windowWidth - (clientRect.right - clientRect.left);
	const long moreHeight = windowHeight - (clientRect.bottom - clientRect.top);
	const long newWidth = (windowRect.right - windowRect.left) + moreWidth;
	const long newHeight = (windowRect.bottom - windowRect.top) + moreHeight;

	SetWindowPos(windowHandle, NULL, 0, 0, newWidth, newHeight, SWP_NOMOVE | SWP_NOZORDER);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CheckAndFixWindowPosition(HWND windowHandle)
{
	RECT primaryMonitor;
	primaryMonitor.left = 0;
	primaryMonitor.top = 0;
	primaryMonitor.right = primaryMonitor.left + GetSystemMetrics(SM_CXSCREEN);
	primaryMonitor.bottom = primaryMonitor.top + GetSystemMetrics(SM_CYSCREEN);

	RECT windowRect;
	GetWindowRect(windowHandle, &windowRect);

	int width = windowRect.right - windowRect.left;
	int height = windowRect.bottom - windowRect.top;
	int newPositionX = primaryMonitor.left + (primaryMonitor.right - primaryMonitor.left - width) / 2;
	int newPositionY = primaryMonitor.top + (primaryMonitor.bottom - primaryMonitor.top - height) / 2;

//	MonitorFromWindow(windowHandle, MONITOR_DEFAULTTOPRIMARY);

	SetWindowPos(windowHandle, NULL, newPositionX, newPositionY, 0, 0, SWP_NOSIZE | SWP_NOZORDER);
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbImplementation::TryGLContext(const GLVersion& glVersion, HGLRC parentContext, HDC& hDC, HGLRC& hRC)
{
	int contextProfileBit(WGL_CONTEXT_CORE_PROFILE_BIT_ARB);
	if (kContextCompatible == glVersion.mType)
	{
		contextProfileBit = WGL_CONTEXT_COMPATIBILITY_PROFILE_BIT_ARB;
	}
	else if (kContextES == glVersion.mType)
	{
		tb_error("tbInternalError: This context is unsupported at the preset time.");
	}

	int contextAttributes[] = { 
		WGL_CONTEXT_PROFILE_MASK_ARB, contextProfileBit,
		WGL_CONTEXT_FLAGS_ARB, WGL_CONTEXT_FORWARD_COMPATIBLE_BIT_ARB,
		WGL_CONTEXT_MAJOR_VERSION_ARB, glVersion.mMajor,
		WGL_CONTEXT_MINOR_VERSION_ARB, glVersion.mMinor,
		NULL
	};

	tb_error_if(!WGL_ARB_create_context, "tbInternalError: Was unable to confirm WGL_ARB_create_context extension for OpenGL 3.2 context.");
	tb_error_if(!WGL_ARB_create_context_profile, "tbInternalError: Was unable to confirm WGL_ARB_create_context_profile extension for OpenGL 3.2 context.");
	
	if (nullptr != wglCreateContextAttribsARB)
	{
		HGLRC coreProfile = wglCreateContextAttribsARB(hDC, NULL, contextAttributes);
		if (NULL != coreProfile)
		{	//Success condition, swap first context with new context with setup attributes.
			wglMakeCurrent(hDC, NULL);
			wglDeleteContext(hRC);

			wglMakeCurrent(hDC, coreProfile);
			hRC = coreProfile;

			return true;
		}
	}
	else
	{
		tb_error_if(nullptr == wglCreateContextAttribsARB, "tbInternalError: Was unable to find/use wglCreateContextAttribsARB function.");
	}

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetupOpenGL(const tbApplication::WindowProperties& windowProperties)
{
	HDC hDC = GetDC(tbiWindowHandle);		//Get Device Context...

	PIXELFORMATDESCRIPTOR pixelFormat;
	memset(&pixelFormat, 0, sizeof(PIXELFORMATDESCRIPTOR));
	pixelFormat.nSize = sizeof(PIXELFORMATDESCRIPTOR);
	pixelFormat.nVersion = 1;
	pixelFormat.dwFlags = PFD_DRAW_TO_WINDOW | PFD_SUPPORT_OPENGL | PFD_DOUBLEBUFFER;
	pixelFormat.iPixelType = PFD_TYPE_RGBA;
	pixelFormat.cColorBits = 32;
	pixelFormat.cDepthBits = 24;
	pixelFormat.cStencilBits = 0;
	pixelFormat.iLayerType = PFD_MAIN_PLANE;
	
	int pixelFormatId = ChoosePixelFormat(hDC, &pixelFormat);		//Does not set the PixelFormat, just checks for what we really got.
	if (0 == pixelFormatId)
	{
		tbi_log_warning("tbWarning: ChoosePixelFormat() failed, this may cause create context to fail.\n");
	}

	if (false == SetPixelFormat(hDC, pixelFormatId, &pixelFormat))
	{	//This returns false on error, also fills in pfd with what was set.
		//TODO: TIM: Warning: Perhaps do some logging here that this wasn't set.
		tbi_log_warning("tbWarning: SetPixelFormat() failed, this may cause create context to fail.\n");
	}

	//NeilGD gave the following for reference for setting up OpenGL 3.2 context.
	// https://github.com/Psybrus/Psybrus/blob/master/Engine/Source/Shared/System/Renderer/GL/RsContextGL.cpp#L550
	// https://github.com/Psybrus/Psybrus/blob/master/Engine/Source/Shared/System/Renderer/GL/RsContextGL.cpp#L746

	HGLRC hRC = wglCreateContext(hDC);			//Create the OpenGL Rendering Context (IF NULL QUIT!)
	if (NULL == hRC)
	{
		tb_error("tbInternalError: Was unable to create OpenGL Context.");
	}

	wglMakeCurrent(hDC, hRC);
	glewExperimental = 0;
	GLenum glewCode = glewInit();
	if (GLEW_OK != glewCode)
	{
		tb_error("tbInternalError: Could not initialize glew. Error code: %s", tb_string(glewCode).c_str());
		return;
	}

	bool setupContext(false);

#if !defined(tb_legacy_gl_forced)
	std::vector<GLVersion> versions;
	versions.push_back(GLVersion(3, 2, kContextCore));
	versions.push_back(GLVersion(3, 2, kContextCompatible));

	for (auto itr = versions.begin(), itrEnd = versions.end(); itr != itrEnd && false == setupContext; ++itr)
	{
		if (true == TryGLContext(*itr, NULL, hDC, hRC))
		{
			setupContext = true;
		}
	}
#endif /* tb_legacy_gl_forced */

	if (false == setupContext)
	{
#if defined(tb_legacy_gl_support)
		tbi_log_warning("tbWarning: Unable to setup an OpenGL 3.2 Context, using Legacy OpenGL which may not be fully supported.\n");
		tbImplementation::Renderer::SetLegacyRenderer(true);
		setupContext = true;
#else
		tb_error_if(false == setupContext, "tbInternalError: Failed to CreateContextAttributes.");
#endif
	}

	glewCode = glewInit();
	if (GLEW_OK != glewCode)
	{
		tb_error("tbInternalError: Could not reinitialize glew. Error code: %s", tb_string(glewCode).c_str());
		return;
	}

	const int verticalSync((true == windowProperties.mVerticalSync) ? 1 : 0);
	if (nullptr != wglSwapIntervalEXT)
	{
		wglSwapIntervalEXT(verticalSync);
	}
	else
	{ //Because GL defaults to vsync, we only need to trigger an error if vsync was false.
		tb_error_if(false == windowProperties.mVerticalSync, "tbInternalError: Could not set GL to be free from vertical sync");
	}

	glClearColor(0.0f, 0.0f, 0.0f, 0.0f);
	glClear(GL_COLOR_BUFFER_BIT | GL_DEPTH_BUFFER_BIT);

	SwapBuffers(hDC);
	SwapBuffers(hDC);

	ReleaseDC(tbiWindowHandle, hDC);

	//tb_error_if(TRUE != _CrtCheckMemory(), "ERROR: MEMORY ISSUE");
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::ClearOpenGL(void)
{
	HDC hDC = GetDC(tbiWindowHandle);		//Get Device Context...
	HGLRC hRC = wglGetCurrentContext();
	wglMakeCurrent(hDC, NULL);
	wglDeleteContext(hRC);
	ReleaseDC(tbiWindowHandle, hDC);	
	//ChangeDisplaySettingsEx(NULL, NULL,	NULL, 0, NULL);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetMousePosition(int mouseX, int mouseY)
{
	POINT mousePoint;
	mousePoint.x = mouseX;
	mousePoint.y = mouseY;

	HMENU menu = GetMenu(tbiWindowHandle);
	if (NULL != menu)
	{
		mousePoint.y -= GetSystemMetrics(SM_CYMENU);
	}

	ClientToScreen(tbiWindowHandle, &mousePoint);
	SetCursorPos(mousePoint.x, mousePoint.y);
}

//--------------------------------------------------------------------------------------------------------------------//

LRESULT CALLBACK tbImplementation::WindowProcedure(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	tb_error_if(nullptr == tbiApplicationHandler, "tbInternalError: Expected to have a valid ApplicationHandler at this moment.");

	switch (message)
	{
	case WM_CREATE: {
		tb_error_if(NULL != tbiWindowHandle, "tbInternalError: Expected window handle to be unused at this moment.");
		tbiWindowHandle = windowHandle;

		//This was called here, but since the OpenGL context is not yet ready, we will delay ever so slightly.
		//tbiApplicationHandler->OnWindowOpen();
		break;  }
	case WM_DESTROY: {
		tbiApplicationHandler->OnWindowClose();
		PostQuitMessage(0);
		break;  }

	//	case WM_CONTEXTMENU: { /* This would be handled in tbi_windows_application_menu.cpp if a context menu is applied. */ } break;

	//The user has pressed the X to close the window.
	case WM_CLOSE: {
		CloseRealtimeApplication(*tbiApplicationHandler);
		break; }

	case WM_ACTIVATE: {
		if (WA_INACTIVE == wParam)
		{
			tbiApplicationHandler->OnBecomeInactive();
		}
		else
		{
			tbiApplicationHandler->OnBecomeActive();
		}
		break; }

	case WM_KEYDOWN: {
		KeyTable::const_iterator itr = tbiVirtualCodesToKey.find(wParam);
		if (itr != tbiVirtualCodesToKey.end())
		{
			if (0 == (lParam & (1 << 30)))
			{ 
				tbImplementation::OnPressKey(itr->second);
			}
		}
		break; }
	case WM_KEYUP: {
		KeyTable::const_iterator itr = tbiVirtualCodesToKey.find(wParam);
		if (itr != tbiVirtualCodesToKey.end())
		{
			tbImplementation::OnReleaseKey(itr->second);
		}
		break; }

	case WM_MOUSEMOVE: {
		//int mouseX = GET_X_LPARAM(lParam);
		//int mouseY = GET_Y_LPARAM(lParam);
		//tbImplementation::UpdateMousePosition(mouseX, mouseY);
		break; }
	case WM_LBUTTONDOWN: { tbImplementation::OnPressKey(tbApplication::tbMouseLeft); break; }
	case WM_LBUTTONUP: { tbImplementation::OnReleaseKey(tbApplication::tbMouseLeft); break; }
	case WM_RBUTTONDOWN: { tbImplementation::OnPressKey(tbApplication::tbMouseRight); break; }
	case WM_RBUTTONUP: { tbImplementation::OnReleaseKey(tbApplication::tbMouseRight); break; }
	case WM_MBUTTONDOWN: { tbImplementation::OnPressKey(tbApplication::tbMouseMiddle); break; }
	case WM_MBUTTONUP: { tbImplementation::OnReleaseKey(tbApplication::tbMouseMiddle); break; }

	case WM_PAINT: {
		HDC hDC = GetDC(tbiWindowHandle);
		SwapBuffers(hDC);
		ReleaseDC(tbiWindowHandle, hDC);
		break; }
	};

	return DefWindowProc(windowHandle, message, wParam, lParam);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::OpenRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(true == tbiHasApplication, "tbExternalError: TurtleBrains currently supports only a single application at a time.");
	tb_error_if(nullptr != tbiApplicationHandler, "tbInternalError: Expected application handler to by null at this moment.");
	tb_error_if(true == tbiIsRunning, "tbInternalError: Expected application not to be running at this moment.");
	
	///////////////////////  Initialization of the Realtime Application
	tbiHasApplication = true;
	tbiApplicationHandler = &applicationHandler;

	//Create and setup default window properties but allow the programmer to change them if desired.
	tbApplication::WindowProperties windowProperties;
	tbImplementation::tbiApplicationHandler->CollectWindowProperties(windowProperties);

	const HINSTANCE processInstance(GetModuleHandle(nullptr));

	const TCHAR kDefaultWindowClass[] = TEXT("TurtleBrainsApplication");
	const TCHAR kDefaultWindowTitle[] = TEXT("TurtleBrains Project");

	const long windowPositionX = windowProperties.mWindowPositionX;
	const long windowPositionY = windowProperties.mWindowPositionY;
	const long windowWidth = windowProperties.mWindowWidth;
	const long windowHeight = windowProperties.mWindowHeight;

	WNDCLASS windowClass;
	windowClass.style = CS_VREDRAW | CS_HREDRAW | CS_OWNDC;
	windowClass.lpfnWndProc = tbImplementation::WindowProcedure;
	windowClass.cbClsExtra = 0;
	windowClass.cbWndExtra = 0;
	windowClass.hInstance = processInstance;
	windowClass.hIcon = LoadIcon(NULL, IDI_APPLICATION);		
	windowClass.hCursor = LoadCursor(NULL, IDC_ARROW);
	windowClass.hbrBackground = (HBRUSH)COLOR_WINDOW;
	windowClass.lpszMenuName = NULL;
	windowClass.lpszClassName = kDefaultWindowClass;
	RegisterClass(&windowClass);

	//Was using WS_OVERLAPPEDWINDOW, removed for current usage until Window Resizing is decidedly a supported feature.
	const long overlappedWindowStyle = WS_CAPTION | WS_SYSMENU | WS_MINIMIZEBOX;
	const long windowStyle = overlappedWindowStyle | WS_CLIPCHILDREN | WS_CLIPSIBLINGS;

	HWND newWindow(CreateWindow(kDefaultWindowClass, kDefaultWindowTitle, windowStyle,
		windowPositionX, windowPositionY, windowWidth, windowHeight, NULL, NULL, processInstance, 0));

	tb_error_if(NULL == tbiWindowHandle, "tbInternalError, Expected window handle to be valid.");
	tb_error_if(tbiWindowHandle != newWindow, "tbInternalError, Expected window handle to match.");
	//tbiWindowHandle = newWindow;

	FixWindowSize(newWindow, windowWidth, windowHeight);
	CheckAndFixWindowPosition(newWindow);
	SetupKeyTable();
	SetupOpenGL(windowProperties);
	tbImplementation::Renderer::InitializeBasicRenderer();

	tbiApplicationHandler->OnWindowOpen();

	UpdateWindow(newWindow);
	ShowWindow(newWindow, SW_SHOW);

//	tbiApplicationHandler->OnInitialization();

	///////////////////////  Run message pump for a Realtime Application

	//Moved to Run
	//HDC hDC = GetDC(tbiWindowHandle);
	//MSG message;
	//tbiIsRunning = true;
	//while (true == tbiIsRunning)
	//{
	//	tbImplementation::StartNewFrame();

	//	while (TRUE == PeekMessage(&message, NULL, 0, 0, PM_REMOVE))
	//	{
	//		if (WM_QUIT == message.message)
	//		{
	//			tbiIsRunning = false;
	//		}

	//		if (false == tbImplementation::HandleAsDialogMessage(message))
	//		{
	//			TranslateMessage(&message);
	//			DispatchMessage(&message);
	//		}
	//	}

	//	if (true == tbiIsRunning)
	//	{
	//		tbiApplicationHandler->OnRealtimeUpdate();
	//		SwapBuffers(hDC);
	//	}
	//}

	///////////////////////  Realtime Application has run it's course, cleanup time.

	//Moved to close
	//ReleaseDC(tbiWindowHandle, hDC);
	//ClearOpenGL();
	//DestroyWindow(tbiWindowHandle);
	//tbiHasApplication = false;
	//tbiApplicationHandler = nullptr;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::RunRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: applicationHandler changed from Open() to Run().");

	HDC hDC = GetDC(tbiWindowHandle);
	MSG message;
	tbiIsRunning = true;
	while (true == tbiIsRunning)
	{
		tbImplementation::StartNewFrame();
		
		POINT mousePoint;
		GetCursorPos(&mousePoint);
		ScreenToClient(tbiWindowHandle, &mousePoint);
		HMENU menu = GetMenu(tbiWindowHandle);
		if (NULL != menu)
		{
			mousePoint.y += GetSystemMetrics(SM_CYMENU);
		}

		tbImplementation::UpdateMousePosition(mousePoint.x, mousePoint.y);

		while (TRUE == PeekMessage(&message, NULL, 0, 0, PM_REMOVE))
		{
			if (WM_QUIT == message.message)
			{
				tbiIsRunning = false;
			}

			if (false == tbImplementation::HandleAsDialogMessage(message))
			{
				TranslateMessage(&message);
				DispatchMessage(&message);
			}
		}

		if (true == tbiIsRunning)
		{
			tbiApplicationHandler->OnRealtimeUpdate();

			if (true == tbImplementation::Renderer::tbiIsRendererAvailable)
			{
				SwapBuffers(hDC);
			}
		}
	}

	ReleaseDC(tbiWindowHandle, hDC);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::StopRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tbiIsRunning = false;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CloseRealtimeApplication(TurtleBrains::Application::ApplicationHandlerInterface& applicationHandler)
{
	tb_error_if(&applicationHandler != tbiApplicationHandler, "tbInternalError: applicationHandler changed from Open() to Close().");

	//Note: There may be an edge case here in the future, the change of Open() -> Close() to Open(), Run(), Close() may
	//  have introduced a way for the resources to be destroyed before the Run() call has been completed since the call
	//  to Close() triggers Run() to stop running.  -TIM 20150317

	tbImplementation::Renderer::CleanupBasicRenderer();
	ClearOpenGL();
	DestroyWindow(tbiWindowHandle);
	tbiIsRunning = false;
	tbiHasApplication = false;
	tbiApplicationHandler = nullptr;
} 

//--------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowTitle(const tbCore::tbString& windowTitle)
{
	tbiWindows::WindowsString title(tbiWindows::ToWindowsString(windowTitle));
	SetWindowText(tbiWindowHandle, title.c_str());
}

//--------------------------------------------------------------------------------------------------------------------//

#endif /* tb_windows */
