///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved --> 
///-----------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_windows

#include "../tbi_system_application_menu.h"
#include "../../tb_application_menu.h"
#include "../../tb_application_handler_interface.h"
#include "../../../core/tb_error.h"

#include <map>
#include <vector>

#define WIN32_LEAN_AND_MEAN
#include <windows.h>

namespace tbiWindows
{	//Definition for this is actually in tbi_windows_application_dialog.cpp currently.
	typedef std::basic_string<TCHAR> WindowsString;
	WindowsString ToWindowsString(const tbCore::tbString& input);
};

namespace tbImplementation
{
	//struct WindowToMenu
	//{
	//	WNDPROC mRealWindowProcedure;
	//	tbApplication::ApplicationHandlerInterface* mMenuHandler;
	//	std::vector<tbApplication::MenuItemIdentifier> mMenuItems;
	//};

	//typedef std::map<HWND, WindowToMenu> WindowToMenuContainer;
	//WindowToMenuContainer mWindowsToMenus;

	extern HWND tbiWindowHandle;	//defined in tbi_realtime_windows_application.

	struct MenuInformation
	{
		WNDPROC mRealWindowProcedure;
		HWND mWindowHandle;
		HMENU mWindowMenuHandle;
		HMENU mContextMenuHandle;

		typedef std::vector<tbApplication::MenuItemIdentifier> MenuItemContainer;
		MenuItemContainer mWindowMenuItems;
		MenuItemContainer mContextMenuItems;
	};

	typedef std::map<tbApplication::ApplicationHandlerInterface*, MenuInformation> HandlerToMenusContainer;
	HandlerToMenusContainer mMenuInformation;

	typedef std::map<HWND, HandlerToMenusContainer::iterator> WindowToMenuContainer;
	WindowToMenuContainer mWindowsToMenus;

	//HMENU CreateWindowsMenuFromItems(const tbApplication::ApplicationMenu& menu, HMENU parentMenu, MenuInformation::MenuItemContainer& menuItems);
	HMENU CreateWindowsMenuFromItems(const MenuItem& menuItem, HMENU parentMenu, MenuInformation::MenuItemContainer& menuItems);
	LRESULT CALLBACK WindowMenuProcedure(HWND hwnd, UINT uMsg, WPARAM wParam, LPARAM lParam);

}; /* namespace tbImplementation */

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowMenu(const tbApplication::ApplicationMenu& menu, tbApplication::ApplicationHandlerInterface& menuHandler)
{
	WindowInformation windowInformation;
	//windowInformation.mWindowHandle = GetActiveWindow();
	windowInformation.mWindowHandle = tbiWindowHandle;
	tbImplementation::SetWindowMenu(windowInformation, menu, menuHandler);
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowMenu(WindowInformation& windowInformation, const tbApplication::ApplicationMenu& menu, tbApplication::ApplicationHandlerInterface& menuHandler)
{
	//HWND* window(reinterpret_cast<HWND*>(windowInfo));
	//tb_error_if(nullptr == windowInfo, "Invalid parameter; windowHandle, expected non-null value to address of HWND object.\n");
	//tb_error_if(nullptr == *window, "Invalid parameter; windowHandle, expected address of a non-NULL HWND object.\n");
	//HWND windowHandle(*window);

	tb_error_if(NULL == windowInformation.mWindowHandle, "tbInternalError: Invalid parameter; windowHandle, expected a non-NULL HWND object.\n");
	HWND windowHandle(windowInformation.mWindowHandle);

	HandlerToMenusContainer::iterator itr = mMenuInformation.find(&menuHandler);
	if (itr == mMenuInformation.end())
	{	//This section is creating a new MenuInformation for the Window/Application Handler (duplicated in SetWindowMenu and SetContextMenu)
		//Steal the Window Procedure from the window, we will call it after checking for menu items.
		MenuInformation menuInformation;
	
		menuInformation.mWindowMenuHandle = NULL;
		menuInformation.mContextMenuHandle = NULL;
		menuInformation.mWindowHandle = windowHandle;
		menuInformation.mRealWindowProcedure = reinterpret_cast<WNDPROC>(GetWindowLongPtr(windowHandle, GWLP_WNDPROC));
		tb_error_if(NULL == menuInformation.mRealWindowProcedure, "tbInternalError: Invalid parameter windowHandle has no window with a procedure.");
	

		///
		/// The following is a comment block because it does not work as expected on Windows XP.  The code is only used
		///   for making sure the real window procedure is responsive, so it may be unnecessary. -TIM 05/31/2014
		///
/*
		DWORD result;
		tb_error_if(0 == SendMessageTimeout(windowHandle, WM_NULL, 0, 0, SMTO_BLOCK | SMTO_ERRORONEXIT, 250, &result), "tbInternalError: Expected window not to timeout with WM_NULL message.");
		tb_error_if(0 != result, "tbInternalError: Expected the provided window to handle the WM_NULL message by returning 0.");
*/
		SetWindowLongPtr(windowHandle, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WindowMenuProcedure));

		itr = mMenuInformation.insert(HandlerToMenusContainer::value_type(&menuHandler, menuInformation)).first;
		mWindowsToMenus[windowHandle] = itr;
	}
	else if (NULL != itr->second.mWindowMenuHandle)
	{
		MenuInformation& menuInformation(itr->second);

		SetMenu(menuInformation.mWindowHandle, NULL);
		//If I understand this correctly, then windows just called DestroyMenu on the menu in the Window:
		//http://msdn.microsoft.com/en-us/library/windows/desktop/ms647624(v=vs.85).aspx
		menuInformation.mWindowMenuHandle = NULL;
		menuInformation.mWindowMenuItems.clear();
	}

	const MenuItem& primaryMenuItem(GetMenuFor(menu.GetIdentifier()));
	itr->second.mWindowMenuHandle = CreateWindowsMenuFromItems(primaryMenuItem, CreateMenu(), itr->second.mWindowMenuItems);
	SetMenu(itr->second.mWindowHandle, itr->second.mWindowMenuHandle);
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CleanupWindowMenu(tbApplication::ApplicationHandlerInterface& menuHandler)
{
	HandlerToMenusContainer::iterator itr = mMenuInformation.find(&menuHandler);
	if (itr != mMenuInformation.end())
	{
		MenuInformation& menuInformation = itr->second;
		
		SetMenu(menuInformation.mWindowHandle, NULL);
		//If I understand this correctly, then windows just called DestroyMenu on the menu in the Window:
		//http://msdn.microsoft.com/en-us/library/windows/desktop/ms647624(v=vs.85).aspx
		menuInformation.mWindowMenuHandle = NULL;
		menuInformation.mWindowMenuItems.clear();

		if (NULL == menuInformation.mWindowMenuHandle && NULL == menuInformation.mContextMenuHandle)
		{	//Remove the stolen Window Procedure and clean up completely, no more menu for this window. (Code duplicated in CleanupWindowMenu and CleanupContextMenu())
			SetWindowLongPtr(menuInformation.mWindowHandle, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(menuInformation.mRealWindowProcedure));
			mWindowsToMenus.erase(menuInformation.mWindowHandle);
			itr = mMenuInformation.erase(itr);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetContextMenu(const TurtleBrains::Application::ApplicationMenu& menu, 
	TurtleBrains::Application::ApplicationHandlerInterface& menuHandler)
{
	WindowInformation windowInformation;
	//windowInformation.mWindowHandle = GetActiveWindow();
	windowInformation.mWindowHandle = tbiWindowHandle;
	tbImplementation::SetContextMenu(windowInformation, menu, menuHandler);
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetContextMenu(WindowInformation& windowInformation, const TurtleBrains::Application::ApplicationMenu& menu,
	TurtleBrains::Application::ApplicationHandlerInterface& menuHandler)
{
	//HWND* window(reinterpret_cast<HWND*>(windowInfo));
	//tb_error_if(nullptr == windowInfo, "Invalid parameter; windowHandle, expected non-null value to address of HWND object.\n");
	//tb_error_if(nullptr == *window, "Invalid parameter; windowHandle, expected address of a non-NULL HWND object.\n");
	//HWND windowHandle(*window);

	tb_error_if(NULL == windowInformation.mWindowHandle, "tbInternalError: Invalid parameter; windowHandle, expected a non-NULL HWND object.\n");
	HWND windowHandle(windowInformation.mWindowHandle);

	HandlerToMenusContainer::iterator itr = mMenuInformation.find(&menuHandler);
	if (itr == mMenuInformation.end())
	{	//This section is creating a new MenuInformation for the Window/Application Handler (duplicated in SetWindowMenu and SetContextMenu)
		//Steal the Window Procedure from the window, we will call it after checking for menu items.
		MenuInformation menuInformation;
	
		menuInformation.mWindowMenuHandle = NULL;
		menuInformation.mContextMenuHandle = NULL;
		menuInformation.mWindowHandle = windowHandle;
		menuInformation.mRealWindowProcedure = reinterpret_cast<WNDPROC>(GetWindowLongPtr(windowHandle, GWLP_WNDPROC));
		tb_error_if(NULL == menuInformation.mRealWindowProcedure, "tbInternalError: Invalid parameter windowHandle has no window with a procedure.");
	
		DWORD result;
		tb_error_if(0 == SendMessageTimeout(windowHandle, WM_NULL, 0, 0, SMTO_BLOCK | SMTO_ERRORONEXIT, 250, &result), "tbInternalError: Expected window not to timeout with WM_NULL message.");
		tb_error_if(0 != result, "tbInternalError: Expected the provided window to handle the WM_NULL message by returning 0.");

		SetWindowLongPtr(windowHandle, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(WindowMenuProcedure));

		itr = mMenuInformation.insert(HandlerToMenusContainer::value_type(&menuHandler, menuInformation)).first;
		mWindowsToMenus[windowHandle] = itr;
	}
	else if (NULL != itr->second.mContextMenuHandle)
	{
		DestroyMenu(itr->second.mContextMenuHandle);
		itr->second.mContextMenuHandle = NULL;
		itr->second.mContextMenuItems.clear();
	}

	const MenuItem& primaryMenuItem(GetMenuFor(menu.GetIdentifier()));
	itr->second.mContextMenuHandle = CreateWindowsMenuFromItems(primaryMenuItem, CreatePopupMenu(), itr->second.mContextMenuItems);
	//Context menu pops up when invoked in the stolen Window Procedure.
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CleanupContextMenu(TurtleBrains::Application::ApplicationHandlerInterface& menuHandler)
{
	HandlerToMenusContainer::iterator itr = mMenuInformation.find(&menuHandler);
	if (itr != mMenuInformation.end())
	{
		MenuInformation& menuInformation = itr->second;
		
		DestroyMenu(menuInformation.mContextMenuHandle);
		menuInformation.mContextMenuHandle = NULL;
		menuInformation.mContextMenuItems.clear();

		if (NULL == menuInformation.mWindowMenuHandle && NULL == menuInformation.mContextMenuHandle)
		{	//Remove the stolen Window Procedure and clean up completely, no more menu for this window. (Code duplicated in CleanupWindowMenu and CleanupContextMenu())
			SetWindowLongPtr(menuInformation.mWindowHandle, GWLP_WNDPROC, reinterpret_cast<LONG_PTR>(menuInformation.mRealWindowProcedure));
			mWindowsToMenus.erase(menuInformation.mWindowHandle);
			itr = mMenuInformation.erase(itr);
		}
	}
}

//-------------------------------------------------------------------------------------------------------------------//
//-------------------------------------------------------------------------------------------------------------------//
//-------------------------------------------------------------------------------------------------------------------//

HMENU tbImplementation::CreateWindowsMenuFromItems(const MenuItem& parentItem, HMENU parentMenu,
	MenuInformation::MenuItemContainer& menuItems)
{
	//const size_t menuSize(tbImplementation::NumberOfItemsForMenu(menu.GetIdentifier()));
	const size_t menuSize(parentItem.mSubItems.size());

	if (0 == menuSize)
	{
		return NULL;
	}

	HMENU windowsMenu = parentMenu;

	tbiWindows::WindowsString menuItemTitle;

	for (size_t itemIndex = menuSize - 1; itemIndex < menuSize; --itemIndex)
	{
		//const tbApplication::ApplicationMenu& menuItem = menu.GetItemAtIndex(itemIndex);
		const MenuItem& menuItem = parentItem.mSubItems[itemIndex];

		if (false == menuItem.mIsVisible)
		{	//Skip this item, and the subitems, if it is invisible.
			continue;
		}

		MENUITEMINFO itemInfo;		memset(&itemInfo, 0, sizeof(MENUITEMINFO));			itemInfo.cbSize = sizeof(MENUITEMINFO);
		menuItems.push_back(menuItem.mItemIdentifier);
		itemInfo.wID = menuItem.mItemIdentifier;
		itemInfo.fMask = MIIM_ID | MIIM_STATE;
		itemInfo.fState |= (menuItem.mIsEnabled) ? MFS_ENABLED : MFS_DISABLED;
		itemInfo.fState |= (menuItem.mIsChecked) ? MFS_CHECKED : MFS_UNCHECKED;

		if (false == menuItem.mDisplayName.empty())
		{
			itemInfo.fMask |= MIIM_STRING;

			menuItemTitle = tbiWindows::ToWindowsString(menuItem.mDisplayName);

			//This should be okay, dwTypeData is not a const because of GetMenuItemInfo can change it.
			//It isn't very amazing though, so maybe a dynamic buffer, and string copy is worth removing this.
			itemInfo.dwTypeData = itemInfo.dwTypeData = const_cast<TCHAR*>(menuItemTitle.c_str());
		}
		else
		{
			itemInfo.fMask |= MIIM_TYPE;
			itemInfo.fType = MFT_SEPARATOR;
		}

		if (false == menuItem.mSubItems.empty())
		{
			itemInfo.hSubMenu = CreateWindowsMenuFromItems(menuItem, CreateMenu(), menuItems);
			if (NULL != itemInfo.hSubMenu)
			{
				itemInfo.fMask |= MIIM_SUBMENU;
			}
		}
		InsertMenuItem(windowsMenu, 0, true, &itemInfo);
	}

	return windowsMenu;
}

//-------------------------------------------------------------------------------------------------------------------//

LRESULT CALLBACK tbImplementation::WindowMenuProcedure(HWND windowHandle, UINT message, WPARAM wParam, LPARAM lParam)
{
	WindowToMenuContainer::const_iterator itr = mWindowsToMenus.find(windowHandle);
	tb_error_if(itr == mWindowsToMenus.end(), "tbInternalError: Expected to find window in the WindowsToMenus container.");

	tbApplication::ApplicationHandlerInterface* applicationHandler(itr->second->first);
	const MenuInformation& menuInformation(itr->second->second);

	if (WM_COMMAND == message)
	{
		//Make sure we actually should handle this command here... for Window Menu Items
		for (size_t index = 0; index < menuInformation.mWindowMenuItems.size(); ++index)
		{
			if (menuInformation.mWindowMenuItems[index] == LOWORD(wParam))
			{
				//TODO: TIM: Implement the menuIdentifier and pass it where 0 is.
				applicationHandler->OnMenuAction(0, menuInformation.mWindowMenuItems[index]);
				break;
			}
		}

		//Make sure we actually should handle this command here... for Context Menu Items
		for (size_t index = 0; index < menuInformation.mContextMenuItems.size(); ++index)
		{
			if (menuInformation.mContextMenuItems[index] == LOWORD(wParam))
			{
				//TODO: TIM: Implement the menuIdentifier and pass it where 0 is.
				applicationHandler->OnMenuAction(0, menuInformation.mContextMenuItems[index]);
				break;
			}
		}
	}

	if (WM_CONTEXTMENU == message)
	{
		int contextPositionX = LOWORD(lParam);
		int contextPositionY = HIWORD(lParam);

		BOOL result = TrackPopupMenu(menuInformation.mContextMenuHandle, TPM_NONOTIFY | TPM_RETURNCMD | TPM_LEFTALIGN | TPM_TOPALIGN,
			contextPositionX, contextPositionY, 0, menuInformation.mWindowHandle, NULL);

		if (0 != result)
		{
			//TODO: TIM: Implement the menuIdentifier and pass it where 0 is.
			itr->second->first->OnMenuAction(0, static_cast<tbApplication::MenuItemIdentifier>(result));
		}
	}

	return CallWindowProc(menuInformation.mRealWindowProcedure, windowHandle, message, wParam, lParam);
}

//-------------------------------------------------------------------------------------------------------------------//

#endif /* tb_windows */
