///
/// @file
///
/// @details This file contains implementation details specific to TurtleBrains and may be modified without warning.
/// @note using any classes, functions or definitions found within TurtleBrains::Implementation / tbImplementation is
///   not recommended as they can be changed or removed completely without warning.  This is your final warning.
///
/// <!-- Copyright (c) Tim Beaudet 2015 - All Rights Reserved --> 
///-----------------------------------------------------------------------------------------------------------------///

#include "../../../core/tb_configuration.h"
#ifdef tb_windows

#include "../tbi_system_application_status.h"
#include "../../tb_application_status.h"
#include "../../../core/tb_error.h"

#define WIN32_LEAN_AND_MEAN
#include <windows.h>
#include <CommCtrl.h>

//The following system specific implementation details can be found in tb_system_application_menu.cpp

namespace tbiWindows
{	//Definition for this is actually in tbi_windows_application_dialog.cpp currently.
	typedef std::basic_string<TCHAR> WindowsString;
	WindowsString ToWindowsString(const tbCore::tbString& input);
};

namespace tbImplementation
{

	extern HWND tbiWindowHandle;	//defined in tbi_realtime_windows_application.

	typedef std::map<tbApplication::ApplicationHandlerInterface*, HWND> HandlerToStatusContainer;
	HandlerToStatusContainer mHandlersToStatus;

};	/* namespace tbImplementation */

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowStatus(const TurtleBrains::Application::ApplicationStatus& status,
	TurtleBrains::Application::ApplicationHandlerInterface& statusHandler)
{
	WindowInformation windowInformation;
	//windowInformation.mWindowHandle = GetActiveWindow();
	windowInformation.mWindowHandle = tbiWindowHandle;
	SetWindowStatus(windowInformation, status, statusHandler);
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::SetWindowStatus(WindowInformation& windowInformation, const TurtleBrains::Application::ApplicationStatus& status,
	TurtleBrains::Application::ApplicationHandlerInterface& statusHandler)
{
	//HWND* window(reinterpret_cast<HWND*>(windowInformation));
	//tb_error_if(nullptr == windowInfo, "Invalid parameter; windowHandle, expected non-null value to address of HWND object.\n");
	//tb_error_if(nullptr == *window, "Invalid parameter; windowHandle, expected address of a non-NULL HWND object.\n");
	//HWND windowHandle(*window);

	tb_error_if(NULL == windowInformation.mWindowHandle, "tbInternalError: Invalid parameter; windowHandle, expected a non-NULL HWND object.\n");
	HWND windowHandle(windowInformation.mWindowHandle);

	HandlerToStatusContainer::iterator itr = mHandlersToStatus.find(&statusHandler);
	if (itr == mHandlersToStatus.end())
	{
		RECT clientArea;
		GetClientRect(windowHandle, &clientArea);
		HINSTANCE hInstance = reinterpret_cast<HINSTANCE>(GetWindowLongPtr(windowHandle, GWLP_HINSTANCE));
		HWND statusHandle = CreateWindowEx(0, STATUSCLASSNAME, NULL, WS_CHILD | WS_VISIBLE, clientArea.left, 0, clientArea.right - clientArea.left, 0,
			windowHandle, NULL, hInstance, NULL);

		itr = mHandlersToStatus.insert(HandlerToStatusContainer::value_type(&statusHandler, statusHandle)).first;
	}
	else
	{
		//TODO: TIM: Do we need to clean up after the previous time here??
	}

	std::vector<tbImplementation::StatusItem> statusItems;
	tbImplementation::GetStatusItems(status.GetIdentifier(), statusItems);

	RECT statusArea;
	GetClientRect(itr->second, &statusArea);
	const float windowWidth(static_cast<float>(statusArea.right - statusArea.left));

	int partitionRightEdges[100];
	tb_error_if(statusItems.size() >= 100, "tbInternalError: Allowed too many status items.");
	for (size_t index = 1; index < statusItems.size(); ++index)
	{	//Windows expects the right edge of each partition, not the left like we store in the item info.
		int approximatePosition = static_cast<int>((static_cast<float>(statusItems[index].second) / 100.0f) * windowWidth);
		partitionRightEdges[index - 1] = approximatePosition;
	}

	partitionRightEdges[statusItems.size() - 1] = -1;	//Extend the final bit to the border.

	SendMessage(itr->second, SB_SETPARTS, statusItems.size(), reinterpret_cast<LPARAM>(partitionRightEdges));

	tbiWindows::WindowsString windowsStatusMessage;
	for (size_t index = 0; index < statusItems.size(); ++index)
	{
		windowsStatusMessage = tbiWindows::ToWindowsString(statusItems[index].first);
		SendMessage(itr->second, SB_SETTEXT, LOWORD(index) | HIWORD(SBT_POPOUT), reinterpret_cast<LPARAM>(windowsStatusMessage.c_str()));
	}
}

//-------------------------------------------------------------------------------------------------------------------//

void tbImplementation::CleanupWindowStatus(TurtleBrains::Application::ApplicationHandlerInterface& menuHandler)
{
}

//-------------------------------------------------------------------------------------------------------------------//

#endif /* tb_windows */
