///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_application_dialog.h"
#include "implementation/tbi_system_application_dialog.h"
#include "../core/tb_error.h"

#include <set>

namespace tbImplementation
{

	const tbCore::tbString kEmptyString(tbCore::ToString(""));

};	/* namespace tbImplementation */

using tbImplementation::kLabelControl;
using tbImplementation::kButtonControl;
using tbImplementation::kCheckboxControl;
using tbImplementation::kTextControl;
using tbImplementation::kDropdownControl;

//--------------------------------------------------------------------------------------------------------------------//

const tbApplication::DialogControlSlot tbApplication::kDefaultSlot(-1);
//const tbApplication::DialogControlSlot tbApplication::kTopSlot(0);
//const tbApplication::DialogControlSlot tbApplication::kBottomSlot(-2);	//heh... ??

void tbApplication::ApplicationDialog::SetControlSafeForDuplication(const tbApplication::DialogControlIdentifier& controlIdentifier)
{
	tbImplementation::SetControlSafeForDuplication(controlIdentifier);
}

//--------------------------------------------------------------------------------------------------------------------//

tbApplication::ApplicationDialog::ApplicationDialog(const DialogIdentifier& dialogIdentifier) :
	mIdentifier(dialogIdentifier)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbApplication::ApplicationDialog::~ApplicationDialog(void)
{
}

//--------------------------------------------------------------------------------------------------------------------//

const tbApplication::DialogIdentifier& tbApplication::ApplicationDialog::GetIdentifier(void) const
{
	return mIdentifier;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::ResizeDialog(const tbApplication::DialogBucketType& bucketType,
	const int visibleSlots, const bool allowVerticalScrolling)
{
	tbImplementation::ResizeDialog(mIdentifier, bucketType, visibleSlots, allowVerticalScrolling);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::AddLabelControl(const DialogControlIdentifier& controlIdentifier,
	const tbCore::tbString& controlValue, const DialogControlGuide& guide)
{
	tbImplementation::CreateDialogControl(mIdentifier, controlIdentifier, kLabelControl, controlValue, guide);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::AddButtonControl(const DialogControlIdentifier& controlIdentifier,
	const tbCore::tbString& controlValue, const DialogControlGuide& guide)
{
	tbImplementation::CreateDialogControl(mIdentifier, controlIdentifier, kButtonControl, controlValue, guide);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::AddCheckboxControl(const DialogControlIdentifier& controlIdentifier, 
	const tbCore::tbString& controlName, const bool isChecked, const DialogControlGuide& guide)
{
	const tbCore::tbString fixedValueString(tbImplementation::CheckboxToString(isChecked, controlName));
	tbImplementation::CreateDialogControl(mIdentifier, controlIdentifier, kCheckboxControl, fixedValueString, guide);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::AddTextControl(const DialogControlIdentifier& controlIdentifier,
	const tbCore::tbString& controlValue, const DialogControlGuide& guide)
{
	tbImplementation::CreateDialogControl(mIdentifier, controlIdentifier, kTextControl, controlValue, guide);
}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::AddDropdownControl(const DialogControlIdentifier& controlIdentifier,
	const std::vector<tbCore::tbString>& controlValues, const DialogControlGuide& guide)
{
	tb_error_if(controlValues.size() < 2, "tbExternalError: Expected at least 2 items to create a dropdown control.");
	tb_error_if(controlValues.size() > tbImplementation::kMaximumAvailableDropdownItems, "tbExternalError: Expected less than %d items to create a dropdown control.", tbImplementation::kMaximumAvailableDropdownItems);

	const tbCore::tbString fixedValueString(tbImplementation::DropdownToString(0, controlValues));
	tbImplementation::CreateDialogControl(mIdentifier, controlIdentifier, kDropdownControl, fixedValueString, guide);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbApplication::ApplicationDialog::GetValueAsString(const DialogControlIdentifier& controlIdentifier) const
{
	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
	const tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);

	switch (dialogControl->mType)
	{
	case kLabelControl:
	case kButtonControl:
	case kTextControl:
		return dialogControl->mValue;
	case kCheckboxControl: {
		bool isChecked = false;
		tbCore::tbString checkboxName;
		tbImplementation::StringToCheckbox(dialogControl->mValue, isChecked, checkboxName);
		return checkboxName;
	}
	case kDropdownControl: {
		int selectedIndex = 0;
		std::vector<tbCore::tbString> dropdownValues;
		tbImplementation::StringToDropdown(dialogControl->mValue, selectedIndex, dropdownValues);
		return dropdownValues[selectedIndex];
	}
	default:
		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
	};
	return tbImplementation::kEmptyString;
}

//--------------------------------------------------------------------------------------------------------------------//

float tbApplication::ApplicationDialog::GetValueAsFloat(const DialogControlIdentifier& controlIdentifier) const
{
	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
	const tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);

	switch (dialogControl->mType)
	{
	case kLabelControl:
	case kButtonControl:
	case kCheckboxControl:
	case kDropdownControl:
		return 0.0f;
	case kTextControl:
		if (true == dialogControl->mValue.empty()) { return 0.0f; }
		return tbCore::FromString<float>(dialogControl->mValue);
	default:
		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
	};
	return 0.0;
}

//--------------------------------------------------------------------------------------------------------------------//

int tbApplication::ApplicationDialog::GetValueAsInteger(const DialogControlIdentifier& controlIdentifier) const
{
	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
	const tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);

	switch (dialogControl->mType)
	{
	case kButtonControl:
	case kLabelControl:
		return 0;
	case kCheckboxControl: {
		bool isChecked = false;
		tbCore::tbString checkboxName;
		tbImplementation::StringToCheckbox(dialogControl->mValue, isChecked, checkboxName);
		return (true == isChecked) ? 1 : 0;
	}
	case kTextControl:
		if (true == dialogControl->mValue.empty()) { return 0; }
		return tbCore::FromString<int>(dialogControl->mValue);
	case kDropdownControl: {
		int selectedIndex = 0;
		std::vector<tbCore::tbString> dropdownValues;
		tbImplementation::StringToDropdown(dialogControl->mValue, selectedIndex, dropdownValues);
		return selectedIndex;
	}
	default:
		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
	};
	return 0;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

//void tbApplication::ApplicationDialog::SetValueAsString(const DialogControlIdentifier& controlIdentifier, const tbCore::tbString& controlValue)
//{
//	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
//	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
//	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);
//
//	switch (dialogControl->mType)
//	{
//	case kButtonControl:
//	case kLabelControl:
//	case kTextControl:
//		dialogControl->mValue = controlValue;
//		return;
//	case kCheckboxControl: {
//		const bool isChecked = (1 == GetValueAsInteger(controlIdentifier)) ? true : false;
//		dialogControl->mValue = tbCore::ToString((true == isChecked) ? "1" : "0") + controlValue;
//		return; }
//
//	case kDropdownControl:
//		tb_error_if(true, "tbExternalError: Value cannot be set as string for type: (%d)", dialogControl->mType);
//		tb_error_if(true, "tbInternalError: This is not yet implemented!");
//		return;
//
//	default:
//		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
//	};
//}

//--------------------------------------------------------------------------------------------------------------------//

//void tbApplication::ApplicationDialog::SetValueAsFloat(const DialogControlIdentifier& controlIdentifier, const float controlValue)
//{
//	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
//	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
//	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);
//
//	switch (dialogControl->mType)
//	{
//	case kTextControl:
//		dialogControl->mValue = tbCore::ToString(controlValue);
//		return;
//
//	case kButtonControl:
//	case kLabelControl:
//	case kCheckboxControl:
//	case kDropdownControl:
//		tb_error_if(true, "tbExternalError: Value cannot be set as float for type: (%d)", dialogControl->mType);
//		return;
//
//	default:
//		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
//	};
//}

//--------------------------------------------------------------------------------------------------------------------//

//void tbApplication::ApplicationDialog::SetValueAsInteger(const DialogControlIdentifier& controlIdentifier, const int controlValue)
//{
//	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
//	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
//	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);
//
//	switch (dialogControl->mType)
//	{
//	case kTextControl:
//		dialogControl->mValue = tbCore::ToString(controlValue);
//		return;
//
//	case kCheckboxControl: {
//		tb_error_if(0 != controlValue && 1 != controlValue, "tbExternalError: Invalid input for paramater controlValue(%d), expected 1 (checked) or 0 (unchecked).", controlValue);
//		tbCore::tbString name = GetValueAsString(controlIdentifier);
//		dialogControl->mValue = tbCore::ToString((0 == controlValue) ? "0" : "1") + name;
//		return; }
//
//	case kDropdownControl: {
//		tb_error_if(true, "tbInternalError: This is not yet implemented!");
//		return; }
//
//	case kButtonControl:
//	case kLabelControl:
//		tb_error_if(true, "tbExternalError: Value cannot be set as integer for type: (%d)", dialogControl->mType);
//		return;
//
//	default:
//		tb_error_if(true, "tbInternalError: Unhandled DialogControlType: (%d)", dialogControl->mType);
//	};
//}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::SetEnabled(const DialogControlIdentifier& controlIdentifier, const bool isEnabled)
{
	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);

	dialogControl->mIsEnabled = isEnabled;
}

//--------------------------------------------------------------------------------------------------------------------//

//void tbApplication::ApplicationDialog::SetVisible(const DialogControlIdentifier& controlIdentifier, const bool isVisible)
//{
//	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
//	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
//	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);
//
//	dialogControl->mIsVisible = isVisible;
//}

//--------------------------------------------------------------------------------------------------------------------//

void tbApplication::ApplicationDialog::RepositionControl(const tbApplication::DialogControlIdentifier& controlIdentifier,
	const tbApplication::DialogControlGuide& guide, const int leftEdge, const int width, const tbApplication::DialogControlSlot& slot)
{
	//tb_error_if(mWidthType == kSingleBucket && (guide < 0 || guide > 0), "tbExternalError: Expected guide to remain 0 for a single bucket dialog box.");
	//tb_error_if(mWidthType == kDoubleBucket && (guide < 0 || guide > 2), "tbExternalError: Expected guide to be 0, 1, 2 for a double bucket dialog box.");
	//tb_error_if(mWidthType == kTripleBucket && (guide < 0 || guide > 4), "tbExternalError: Expected guide to be in range for a triple bucket dialog box.");

	tb_error_if(true == tbImplementation::IsControlSafeForDuplication(controlIdentifier), "tbExternalError: Cannot access the dialog control with id: (%d) as it is safe to duplicate.", controlIdentifier);
	tbImplementation::DialogControl* dialogControl = tbImplementation::GetDialogControl(mIdentifier, controlIdentifier);
	tb_error_if(nullptr == dialogControl, "tbExternalError: Dialog Control id (%d) was not found.", controlIdentifier);

	dialogControl->mLeftEdgeOffset = leftEdge;
	dialogControl->mWidthPercentage = width;
	dialogControl->mGuide = guide;

	if (kDefaultSlot != slot)
	{
		dialogControl->mSlotInBucket = slot;
	}
}

//--------------------------------------------------------------------------------------------------------------------//
