///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_ApplicationMenu_h_
#define _TurtleBrains_ApplicationMenu_h_

#include <vector>

#include "../core/tb_types.h"
#include "implementation/tbi_application_information.h"

namespace TurtleBrains
{
	namespace Application
	{

		class ApplicationHandlerInterface;

		///
		/// @details The key to an application menu lies within this identifier, the value must be unique for different
		///   menus in your project.  The ApplicationMenu object takes the identifier and uses it to hide the messy
		///   implementation details behind the scenes.  Multiple ApplicationMenu objects with the same identifier will
		///   access the settings for the same menu even though they are different ApplicationMenu objects.
		///
		typedef unsigned short MenuIdentifier;

		///
		/// @details Each item added to a menu must have a unique identifier that is used to access and modify the item
		///   at any point after it is added.  The MenuItemIdentifier is also used to handle the action of the menu item
		///   being clicked or interacted with by the user.  It is possible to use the same identifier for multiple items
		///   by using ApplicationMenu::SetItemSafeForDuplication().  However identifiers that are safe for duplication
		///   cannot be accessed or modified, and the action may not be reported.
		///
		typedef unsigned short MenuItemIdentifier;

		///
		/// @details No valid MenuItem should have this identifier.
		///
		extern const MenuItemIdentifier kInvalidMenuItemIdentifier;

//--------------------------------------------------------------------------------------------------------------------//

		///
		/// @brief To create simple but powerful menu to be used in an applications or game development tools.
		///
		/// @details Creates an application menu that multiple menu items can be added to.  Each menu item has a name, and
		///   a handful of basic states; enabled, visible, checked, and can hold subitems to create a useful menu system
		///   that can be added to an application for Window or Context menus.
		///
		/// @snippet tb_menu_item_example.cpp exampleApplicationMenu
		///
		class ApplicationMenu
		{
		public:
			///
			/// @details When you create an ApplicationMenu object with a MenuIdentifier, you can later create another
			///   ApplicationMenu object with that same MenuIdentifier value and BOTH of those menu objects will refer to
			///   and edit the same menu settings.
			///
			explicit ApplicationMenu(const MenuIdentifier& menuIdentifier);

			///
			/// @details This will destruct the ApplicationMenu object, but will not destroy resources or remove the menu
			///   settings from memory or the Window or Context areas.  Creating another ApplicationMenu object with the
			///   same MenuIdentifier value will give access to these settings once again.
			///
			/// @note this does not cause the menu to be destroyed, or even cause any settings to be removed from memory, as
			///   creating another ApplicationMenu object with the same MenuIdentifier will provide access again. To remove
			///   the menu from the Window or Context areas you will need to use the primary RealtimeApplication object.
			///
			~ApplicationMenu(void);

			///
			/// @details Retrieves the MenuIdentifier as specified when the object was created, this identifier cannot change
			///   for an ApplicationMenu object.
			///
			const MenuIdentifier& GetIdentifier(void) const;

			///
			/// @details This will add a menu item to the menu, and it is possible to a sub-menu within by supplying a
			///   parentIdentifier.
			///
			/// @param itemIdentifier    The identifier that will be used to signal that the item has been clicked/activated.
			///   Every menu item must have a unique identifier value, even if the items are in different menu systems.
			/// @param displayName       The text of the new menu item that will be displayed, empty string represents a 
			///   seperator.
			/// @param parentIdentifier  To add a menu item as a subitem of another menu item, the parentIdentifier can
			///   can be set to kInvalidMenuItemIdentifier to cause the item to be added directly to the menu instead of
			///   as a submenu to another item.
			/// @param isEnabled         Disabled menu items are grayed out and unselectable, defaults to enabled.
			/// @param isChecked         When true a checkmark will be placed next to the menu item, defaults to false.
			/// @param isVisible         If this is false, the menu item will not appear in the menu, defaults to true.
			///
			void AddMenuItem(const MenuItemIdentifier& itemIdentifier, const tbCore::tbString& displayName,
					const MenuItemIdentifier& parentIdentifier = kInvalidMenuItemIdentifier, const bool isEnabled = true,
					const bool isChecked = false, const bool isVisible = true);

			///
			/// @details Retrieve the current display name setting on the specified menu item.  The value returned may not
			///   be the value that is actually displayed in the menu if SetWindowMenu() / SetContextMenu() has not been
			///   called recently to update the menu with the latest settings.
			///
			/// @param itemIdentifier    The identifier of the menu item that should be used when retrieving the name.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			const tbCore::tbString& GetDisplayName(const MenuItemIdentifier& itemIdentifier) const;

			///
			/// @details Retrieve the current enabled / disabled state of the specified menu item.  The state returned
			///   may not be the state that is actually displayed in the menu if SetWindowMenu() / SetContextMenu() has
			///   not been called recently to update the menu with the latest settings.
			///
			/// @param itemIdentifier    The identifier of the menu item that should be used when retrieving the state.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			bool IsEnabled(const MenuItemIdentifier& itemIdentifier) const;

			///
			/// @details Retrieve the current enabled / disabled state of the specified menu item.  The state returned
			///   may not be the state that is actually displayed in the menu if SetWindowMenu() / SetContextMenu() has
			///   not been called recently to update the menu with the latest settings.
			///
			/// @param itemIdentifier    The identifier of the menu item that should be used when retrieving the state.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			bool IsChecked(const MenuItemIdentifier& itemIdentifier) const;

			///
			/// @details Retrieve the current enabled / disabled state of the specified menu item.  The state returned
			///   may not be the state that is actually displayed in the menu if SetWindowMenu() / SetContextMenu() has
			///   not been called recently to update the menu with the latest settings.
			///
			/// @param itemIdentifier    The identifier of the menu item that should be used when retrieving the state.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			bool IsVisible(const MenuItemIdentifier& itemIdentifier) const;

			///
			/// @details Change the name setting of the specified menu item.  The item's name will not be changed until the
			///   next SetWindowMenu() / SetContextMenu call.
			///
			/// @param itemIdentifier    The identifier of the menu item that should have the display name setting changed.
			/// @param displayName       The new display name setting of the menu item.  An empty string will cause the
			///   menu item to become a seperator item.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			void SetDisplayName(const MenuItemIdentifier& itemIdentifier, const tbCore::tbString& displayName);

			///
			/// @details Change the enabled settings for the specified menu item.  A disabled menu item is visible, but the
			///   menu item cannot be interacted with unless it is enabled.  Typically the operating system shows disabled
			///   menu items as grayed out.  The item will not be changed until the next SetWindowMenu / SetContextMenu call.
			///
			/// @param itemIdentifier    The identifier of the menu item that should become enabled or disabled.
			/// @param isEnabled         The new enabled/disabled setting of the menu item, a value of false will cause the
			///   menu item to remain visible, usually grayed by the operating system, but not allow user interaction.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			void SetEnabled(const MenuItemIdentifier& itemIdentifier, bool isEnabled);

			///
			/// @details Add or remove a checkmark on a menu item which is useful for menu items that have an on/off state
			///   that can be toggled when interacted with.  To toggle the setting, call SetChecked(id, !IsChecked(id))
			///   where id is the itemIdentifier of the menu item.
			///
			/// @param itemIdentifier    The identifier of the menu item that should have the checkmark added or removed.
			/// @param isChecked         The new checked/unchecked setting of the menu item, a value of false will remove
			///   any checkmark previously added, and a value of true will make the menu item display a checkmark like it
			///   is selected or in an active state.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			void SetChecked(const MenuItemIdentifier& itemIdentifier, bool isChecked);

			///
			/// @details Change the visibility settings for the specified menu item.  When changing the settings for a
			///   a menu item, the menu will not be updated until calling SetWindowMenu / SetContextMenu again.
			///
			/// @param itemIdentifier    The identifier of the menu item that should have the visibility setting changed.
			/// @param isVisible         The new visibility setting of the menu item, a value of false will cause the
			///   menu item, and any possible sub items, to not be added to the system menu during the next update.
			///
			/// @note an error will be triggered if itemIdentifier is invalid, or the menu does not have an item with
			///   the specified identifier.
			///
			void SetVisible(const MenuItemIdentifier& itemIdentifier, bool isVisible);

			///
			/// @details Sometimes you don't care to modify or retrieve information from a menu item, mainly separators.
			///   Since each item must have an identifier value attached to it, this allows you to create an identifier
			///   that is safe for duplicating.  Once an identifier is made safe for duplication it can not be made unsafe
			///   and access to any menu items with the identifier will be lost.  Calling accessor / modifier functions
			///   like IsEnabled() or SetEnabled() for an identifier safe for duplication will trigger errors.
			///
			/// @param itemIdentifier    The identifier value that is safe for duplication, no menu items created with
			///   this identifier can be modified or accessed after creation.
			///
			/// @note Once an identifier is set safe for duplication, that value can not be used to access or modifiy any
			///   menu items with that identifier, and the value cannot be later removed from safe for duplication.  This
			///   effects any menu using an item with the identifier.
			///
			static void SetItemSafeForDuplication(const MenuItemIdentifier& itemIdentifier);

		private:
			MenuIdentifier mIdentifier;
		};

	}; /* namespace Application */
}; /* namespace TurtleBrains */

namespace tbApplication = TurtleBrains::Application;

#endif /* _TurtleBrains_ApplicationMenu_h_ */
