///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
/// @note This is a very early version of the API, expect to make constant changes until locked in at v1.0.0.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#ifndef _TurtleBrains_AudioManager_h_
#define _TurtleBrains_AudioManager_h_

#include "../core/tb_noncopyable.h"
#include "../core/tb_string.h"
#include "../core/tb_types.h"

namespace TurtleBrains
{
	namespace Audio
	{

		///
		///	@details An AudioHandle is a unique value for each unique sound created with the AudioManager.  Use the
		///   handle to access the sound properties and to destroy any resources associated.
		///
		/// TODO: TIM: Implementation: Maybe this should be a tiny class/object that automatically cleans up resources?
		/// This may be removed, or changed, or other
		///
		typedef tbCore::uint32 AudioHandle;

		///
		/// @details This will be returned or set for audio handles that are invalid, not loaded, unset...
		///
		extern const AudioHandle kInvalidAudio;

		///
		/// @details This represents an audio channel, although the concept may become hidden behind a deeper wall as
		///   this is really an implementation detail of TurtleBrains.
		///
		typedef tbCore::uint32 AudioChannel;

		///
		/// @details Represents an unused or invalid channel.  The concept may become hidden as this is really an
		///   implementation detail of TurtleBrains.
		///
		extern const AudioChannel kInvalidChannel;

		///
		/// @details The AudioController may become an AudioEvent or Audio class that allows the controlling of an
		///   effect as the AudioChannel and possibly even AudioHandle data may become more hidden.  At a minimum there 
		///   will be some object that gives the ability to Stop, SetVolume and SetPitch if this object changes.
		///
		class AudioController
		{
		public:
			///
			/// @details Constructs an AudioController, this may change and should be considered an implementation
			///   detail of TurtleBrains.  Note this class may change drastically in future iterations.
			///
			explicit AudioController(const AudioHandle& audioData = kInvalidAudio);

			///
			/// @details Constructs an AudioController from another AudioController as a copy constructor would.
			///
			AudioController(const AudioController& other);

			///
			/// @details Cleans up and destroys the AudioController, once destroyed the audio channel/sound effect it
			///   had been controlling cannot be stopped or modified.
			///
			~AudioController(void);

			///
			/// @details Abruptly stops the sound or music from playing and to play again call to AudioManager::PlayEvent.
			///
			void Stop(void);

			///
			/// @details Removes a looping flag from the sound so it will continue to play through until it finishes and then
			///   stop playing.  Does not abruptly stop the sound.
			///
			void StopLooping(void);

			///
			/// @details Changes the pitch, 'speed / frequency' of the sound and how it plays.  1.0 plays at normal pitch
			///   while 0.5 plays at what seems to be half speed, much slower.  2.0f should be twice as fast as normal.
			///
			void SetPitch(const float pitch);

			///
			/// @details Changes the volume of the sound where 0.0f is silent and 1.0f is full volume.  If a value is
			///   out of that range it will be clamped to remain in range, so a value of 1.2 will become 1.0.
			///
			void SetVolume(const float volume);

			///
			/// TODO: TIM: Documentation: Teach the user how to use this.
			///
			float GetVolume(void) const;

			///
			/// TODO: TIM: Documentation: Teach the user how to use this.
			///
			bool IsComplete(void) const;

			friend class AudioManager;
		private:
			AudioHandle mAudioData;
			AudioChannel mAudioChannel;
		};


		///
		/// @details The AudioManager is the interface for loading and playing sound effects.  Like the SpriteManager
		///   the AudioManager is a global singleton like object accessed with tbAudio::theAudioManager.  It can create
		///   a table of audio events that can then be played with PlayEvent().
		///
		class AudioManager : public tbCore::Noncopyable
		{
		public:
			///
			///	@details The AudioManager should be treated like a singleton object and should only ever be accessed through
			///   the tbAudio::theAudioManager 'global' variable.
			///
			AudioManager(void);

			///
			///	@details The AudioManager should be treated like a singleton object and should only ever be accessed through
			///   the tbAudio::theAudioManager 'global' variable.
			///
			~AudioManager(void);

			///
			///	@details Loads a table of audio events and the audio data files required for playback.  Once loaded an audio
			///   event can be played by calling PlayEvent().  If the tablefile or any audio files could not be loaded an
			///   error condition will be triggered.
			///
			/// @note currently an event table cannot be unloaded until shutdown.
			///
			/// Ex. EventTable:
			///
			/// { "audio_events" : [
			///   { "name" : "event_name1", "file" : "data/sound.wav", "looping" : false },
			///   { "name" : "event_name2", "file" : "data/sound.wav", "pitch" : 1.0, "volume" : 1.0 },
			///   { "name" : "event_name3", "file" : [ "data/sound.wav, "data/sound2.wav" ... ] },
			///   { "name" : "event_name3", "file" : [ { "file" : "data/sound.wav, "pitch" : 1.0, "volume" : 1.0 },
			///     { "file" : "data/sound2.wav", "min_pitch" : 0.8, "max_pitch" : 1.2, "min_volume" : 0.5, "max_volume" : 0.6 } ... ] },
			///   ]
			/// }
			///
			/// The json data does not need to contain the optional parameters shown below with their default values.
			///   looping(false), pitch(1.0), volume(1.0), min_pitch(1.0), max_pitch(1.0), min_volume(1.0), max_volume(1.0)
			///
			bool LoadEventTable(const tbCore::tbString& eventTableName, const tbCore::tbString& eventTableFile);

			///
			///	@details Consider this an implementation detail, it will indeed load a sound from the file provided, however
			///   it may become an implementation detail and be stripped away from the AudioManager interface.
			///
			/// @note Use at your own risk, this may be pulled out of the AudioManager to be replaced by LoadEventTable().
			///
			AudioHandle CreateSoundFromFile(const tbCore::tbString& filename);

			///
			///	@details Consider this an implementation detail, it will indeed destroy a sound previously created, however
			///   it will do so even if it is being played, or shouldn't.  Also this may become an implementation detail and
			///   be stripped away from the AudioManager interface.
			///
			/// @note Use at your own risk, this may be pulled out of the AudioManager to be replaced by LoadEventTable().
			///
			void DestroySound(const AudioHandle& audioHandle);

			///
			///	@details Triggers an audio event to begin playing in the background, an audio event can have multiple sound
			///   files to randomly choose from.  The audio event must have been loaded by LoadEventTable prior to being the
			///   call or an error condition will be triggered.
			///
			/// @note Currently this is still under development and not completely set in stone, the eventTableName and
			///   eventName may be swapped to provide an overload that searches all tables for eventName.
			///
			AudioController PlayEvent(const tbCore::tbString& eventTableName, const tbCore::tbString& eventName);

			///
			///	@details Consider this an implementation detail, it will indeed play a sound by an AudioHandle loaded from
			///   CreateSoundFromFile, however it may be pulled out of the AudioManager interface in favor of PlayEvent().
			///
			/// @note Use at your own risk, this may be pulled out of the AudioManager in favor for only playing by event or
			///   other means.  It is provided for quickly testing and developing the AudioManager api.
			///
			AudioController PlaySound(const AudioHandle& audioHandle, const bool isLooping = false);

			///
			///	@details Consider this an implementation detail, it will indeed load a sound from file and play it however
			///   it may be pulled out of the AudioManager interface in favor of PlayEvent().
			///
			/// @note Use at your own risk, this may be pulled out of the AudioManager in favor for only playing by event or
			///   other means.  It is provided for quickly testing and developing the AudioManager api.
			///
			//AudioController PlaySound(const tbCore::tbString& filename, const bool isLooping = false);

			///
			///	@details This must be called once per frame, every frame, to keep the sounds playing, it is likely already
			///   called by the GameScene::PerformFrame, however if for some reason that is not being used, call manually.
			///
			void Update(void);

		private:
		};

		///
		/// @details Instead of creating your own AudioManager object, this is a way to access it as a singleton.  This
		///   _may_ turn into a pointer to the object, although the SoundManager api will remain the same.
		///
		extern AudioManager theAudioManager;

	}; /* namespace Audio */
}; /* namespace TurtleBrains */

namespace tbAudio = TurtleBrains::Audio;

#endif /* _TurtleBrains_AudioManager_h_ */
