///
/// @file
/// @details This is currently in early development and will be properly documented at a later date once
///   the details are more concrete.  TODO: TIM: DocFinal: Check over interface and documentation for first public release.
///
/// <!-- Copyright (c) Tim Beaudet 2016 - All Rights Reserved -->
///------------------------------------------------------------------------------------------------------------------///

#include "tb_dynamic_structure.h"
#include "tb_error.h"

#include <cmath>
#include <sstream>

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure tbCore::DynamicStructure::kNullValue;
const unsigned int tbCore::DynamicStructure::kInvalidSize(0);
const bool tbCore::DynamicStructure::kTypeSafeArrays(true);
const bool tbCore::DynamicStructure::kImplicitConversions(true);
const bool tbCore::DynamicStructure::kImplicitTypeChange(true);

const tbCore::tbString tbCore::DynamicStructure::kNullAsString("");
const tbCore::tbString tbCore::DynamicStructure::kTrueAsString("true");
const tbCore::tbString tbCore::DynamicStructure::kFalseAsString("false");

const float tbCore::DynamicStructure::kFloatElipson(0.00001f);

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(void) :
	mValueType(kNilType),
	mInteger(0)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(const int& integerValue) :
	mValueType(kIntegerType),
	mInteger(integerValue)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(const float& floatValue) :
	mValueType(kFloatType),
	mFloat(floatValue)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(const bool& booleanValue) :
	mValueType(kBooleanType),
	mBoolean(booleanValue)
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(const tbCore::tbString& stringValue) :
	mValueType(kStringType),
	mString(new tbCore::tbString(stringValue))
{
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::DynamicStructure(const DynamicStructure& other) :
	mValueType(other.mValueType)
{
	switch (other.mValueType)
	{
		case kNilType: /* already a kNilType */ break;
		case kBooleanType: mBoolean = other.mBoolean; break;
		case kIntegerType: mInteger = other.mInteger; break;
		case kFloatType: mFloat = other.mFloat; break;
		case kStringType: mString = new tbCore::tbString(*other.mString); break;

		case kArrayType: mArray = new ArrayContainer(*other.mArray); break;
		case kStructureType: mStructure = new StructureContainer(*other.mStructure); break;
		default: tb_error_if(true, "tbInternalError: Unexpected value for mValueType.");
	};
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::operator=(const DynamicStructure& other)
{
	if (this == &other) { return *this; }

	SetToNil();
	mValueType = other.mValueType;
	switch (other.mValueType)
	{
		case kNilType: /* already a kNilType */ break;
		case kBooleanType: mBoolean = other.mBoolean; break;
		case kIntegerType: mInteger = other.mInteger; break;
		case kFloatType: mFloat = other.mFloat; break;
		case kStringType: mString = new tbCore::tbString(*other.mString); break;

		case kArrayType: mArray = new ArrayContainer(*other.mArray); break;
		case kStructureType: mStructure = new StructureContainer(*other.mStructure); break;
		default: tb_error_if(true, "tbInternalError: Unexpected value for mValueType.");
	};

	return *this;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::~DynamicStructure(void)
{
	SetToNil();
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsNil(void) const
{
	return (kNilType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsArray(void) const
{
	return (kArrayType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsStructure(void) const
{
	return (kStructureType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsInteger(void) const
{
	return (kIntegerType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsFloat(void) const
{
	return (kFloatType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsBoolean(void) const
{
	return (kBooleanType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::IsString(void) const
{
	return (kStringType == mValueType) ? true : false;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

int tbCore::DynamicStructure::AsInteger(bool implicitConversion) const
{
	if (true == IsInteger())
	{
		return mInteger;
	}

	tb_error_if(false == implicitConversion, "This is not an IntegerType, type-safety required without implicit conversion.\n");
	return ConvertToInteger();
}

//--------------------------------------------------------------------------------------------------------------------//

float tbCore::DynamicStructure::AsFloat(bool implicitConversion) const
{
	if (true == IsFloat())
	{
		return mFloat;
	}

	tb_error_if(false == implicitConversion, "This is not an FloatType, type-safety required without implicit conversion.\n");
	return ConvertToFloat();
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::AsBoolean(bool implicitConversion) const
{
	if (true == IsBoolean())
	{
		return mBoolean;
	}

	tb_error_if(false == implicitConversion, "This is not an BooleanType, type-safety required without implicit conversion.\n");
	return ConvertToBoolean();
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbCore::DynamicStructure::AsString(bool implicitConversion) const
{
	if (true == IsString())
	{
		return *mString;
	}

	tb_error_if(false == implicitConversion, "This is not an StringType, type-safety required without implicit conversion.\n");
	return ConvertToString();
}

//--------------------------------------------------------------------------------------------------------------------//

int tbCore::DynamicStructure::AsIntegerWithDefault(const int defaultValue) const
{
	return (true == IsInteger()) ? AsInteger() : defaultValue;
}

//--------------------------------------------------------------------------------------------------------------------//

float tbCore::DynamicStructure::AsFloatWithDefault(const float defaultValue) const
{
	return (true == IsFloat()) ? AsFloat() : defaultValue;
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::AsBooleanWithDefault(const bool defaultValue) const
{
	return (true == IsBoolean()) ? AsBoolean() : defaultValue;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbCore::DynamicStructure::AsStringWithDefault(const tbCore::tbString& defaultValue) const
{
	return (true == IsString()) ? AsString() : defaultValue;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbCore::DynamicStructure::SetValue(const int& value, bool implicitTypeChange)
{
	tb_error_if(false == implicitTypeChange && false == IsInteger() && false == IsNil(), "Expected NilType or IntegerType types in order to set a integer value.\n");
	SetToNil();
	mValueType = kIntegerType;
	mInteger = value;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::DynamicStructure::SetValue(const float& value, bool implicitTypeChange)
{
	tb_error_if(false == implicitTypeChange && false == IsFloat() && false == IsNil(), "Expected NilType or FloatType types in order to set a float value.\n");
	SetToNil();
	mValueType = kFloatType;
	mFloat = value;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::DynamicStructure::SetValue(const bool& value, bool implicitTypeChange)
{
	tb_error_if(false == implicitTypeChange && false == IsBoolean() && false == IsNil(), "Expected NilType or BooleanType types in order to set a booelean value.\n");
	SetToNil();
	mValueType = kBooleanType;
	mBoolean = value;
}

//--------------------------------------------------------------------------------------------------------------------//

void tbCore::DynamicStructure::SetValue(const tbCore::tbString& value, bool implicitTypeChange)
{
	tb_error_if(false == implicitTypeChange && false == IsString() && false == IsNil(), "Expected NilType or StringType types in order to set a string value.\n");
	SetToNil();
	mValueType = kStringType;
	mString = new tbCore::tbString(value);
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::PushValue(const DynamicStructure& value)
{
	tb_error_if(false == IsArray() && false == IsNil(), "tbExternalError: Expected this DynamicStructure to be NilType or ArrayType.");
	tb_error_if(true == value.IsNil(), "Cannot push a Nil value into an array.\n");	//Really?  How come?

	if (true == IsNil())
	{
		mValueType = kArrayType;
		mArray = new ArrayContainer();
		mArray->push_back(value);
	}
	else
	{
		tb_error_if(false == IsArray(), "tbExternalError: Expected this DynamicStructure to be an ArrayType.\n");
		tb_error_if(true == kTypeSafeArrays && false == mArray->empty() && (*mArray)[0].mValueType != value.mValueType, "ValueType does not match array value type.\n");
		mArray->push_back(value);
	}

	return mArray->back();
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure& tbCore::DynamicStructure::GetValue(const size_t& arrayIndex) const
{
	tb_error_if(false == IsArray(), "tbExternalError: Expected this MagicValue to be an ArrayType.\n");
	tb_error_if(arrayIndex >= mArray->size(), "tbExternalError: Expected index to be in bounds of array size.\n");
	return (*mArray)[arrayIndex];
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::GetValue(const size_t& arrayIndex)
{
	tb_error_if(false == IsArray(), "tbExternalError: Expected this MagicValue to be an ArrayType.\n");
	tb_error_if(arrayIndex >= mArray->size(), "tbExternalError: Expected index to be in bounds of array size.\n");
	return (*mArray)[arrayIndex];
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const size_t& arrayIndex) const { return GetValue(arrayIndex); }
tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const size_t& arrayIndex) { return GetValue(arrayIndex); }
//const tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const int& arrayIndex) const { return GetValue(arrayIndex); }
//tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const int& arrayIndex) { return GetValue(arrayIndex); }

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::AddMember(const tbCore::tbString& memberName, const DynamicStructure& memberValue)
{
	tb_error_if(false == IsStructure() && false == IsNil(), "tbExternalError: Expected this DynamicStructure to be NilType or StructureType.");

	if (true == IsNil())
	{
		mValueType = kStructureType;
		mStructure = new StructureContainer();
		////(*mStructure)[memberName] = memberValue;
	}
	////else
	////{
	////	(*mStructure)[memberName] = memberValue;
	////}

	////Not the most efficient way to return the reference to the added member.
	////return (*mStructure)[memberName];

	//All lines above with //// were commented 20150425 to follow the current logic:

	tb_error_if(mStructure->end() != mStructure->find(memberName), "tbExternalError: Expected not to find a member with name \"%s\" already existing.", memberName.c_str());

	auto insertReturn = mStructure->insert(StructureContainer::value_type(memberName, memberValue));
	return insertReturn.first->second;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::SetMember(const tbCore::tbString& memberName, const tbCore::DynamicStructure& memberValue)
{
	tb_error_if(false == IsStructure() && false == IsNil(), "tbExternalError: Expected this DynamicStructure to be NilType or StructureType.");
	if (true == IsNil())
	{
		mValueType = kStructureType;
		mStructure = new StructureContainer();
	}

	(*mStructure)[memberName] = memberValue;
	return (*mStructure)[memberName];
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure& tbCore::DynamicStructure::GetMember(const tbCore::tbString& memberName) const
{
	//TODO: Would be awesome if we could do something like the following:
	//	carRoot["tire[2].airPressure.psi"].GetAsInteger()
	//	where we would return the psi member of the airPressure object that belongs
	//	to the tire at index 2 in the carRoot object.

	if (false == IsNil())
	{
		tb_error_if(false == IsStructure(), "Expected this MagicValue to be an StructureType or a NilType.\n");
		auto memberIterator = mStructure->find(memberName);
		if (memberIterator != mStructure->end())
		{
			return memberIterator->second;
		}
	}
	return DynamicStructure::kNullValue;
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure& tbCore::DynamicStructure::GetMember(const tbCore::tbString& memberName)
{
	//TODO: Would be awesome if we could do something like the following:
	//	carRoot["tire[2].airPressure.psi"].GetAsInteger()
	//	where we would return the psi member of the airPressure object that belongs
	//	to the tire at index 2 in the carRoot object.

	tb_error_if(false == IsStructure(), "Expected this DynamicStructure to be a StructureType.\n");
	auto memberIterator = mStructure->find(memberName);
	if (memberIterator == mStructure->end())
	{
		tb_error_if(true, "The member: %s was not found within the DynamicStructure.\n", memberName.c_str());
		//The following code shouldn't actually run due to the error condition...
		static tbCore::DynamicStructure nonConstNullObject;
		nonConstNullObject.SetToNil();
		return nonConstNullObject;
	}
	return memberIterator->second;
}

//--------------------------------------------------------------------------------------------------------------------//

const tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const tbCore::tbString& memberName) const { return GetMember(memberName); }
tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const tbCore::tbString& memberName) { return GetMember(memberName); }
const tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const char* const memberName) const { return GetMember(memberName); }
tbCore::DynamicStructure& tbCore::DynamicStructure::operator[](const char* const memberName) { return GetMember(memberName); }

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::StructureContainer::const_iterator tbCore::DynamicStructure::BeginStructure(void) const
{
	tb_error_if(false == IsStructure(), "tbExternalError: Expected this DynamicStructure to be a StructureType.\n");
	return mStructure->begin();
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::StructureContainer::const_iterator tbCore::DynamicStructure::EndStructure(void) const
{
	tb_error_if(false == IsStructure(), "tbExternalError: Expected this DynamicStructure to be a StructureType.\n");
	return mStructure->end();
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::StructureContainer::iterator tbCore::DynamicStructure::BeginStructure(void)
{
	tb_error_if(false == IsStructure(), "tbExternalError: Expected this DynamicStructure to be a StructureType.\n");
	return mStructure->begin();
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::DynamicStructure::StructureContainer::iterator tbCore::DynamicStructure::EndStructure(void)
{
	tb_error_if(false == IsStructure(), "tbExternalError: Expected this DynamicStructure to be a StructureType.\n");
	return mStructure->begin();
}

//--------------------------------------------------------------------------------------------------------------------//

size_t tbCore::DynamicStructure::Size(void) const
{
	if (true == IsNil())
	{
		return 0;
	}

	if (true == IsArray())
	{
		return mArray->size();
	}

	if (true == IsStructure())
	{
		return mStructure->size();
	}

	return kInvalidSize;
}

//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//
//--------------------------------------------------------------------------------------------------------------------//

void tbCore::DynamicStructure::SetToNil(void)
{
	if (kStringType == mValueType)
	{
		delete mString;
		mString = NULL;
	}
	else if (kArrayType == mValueType)
	{
		delete mArray;
		mArray = NULL;
	}
	else if (kStructureType == mValueType)
	{
		delete mStructure;
		mStructure = NULL;
	}

	mValueType = kNilType;
}

//--------------------------------------------------------------------------------------------------------------------//

int tbCore::DynamicStructure::ConvertToInteger(void) const
{
	switch(mValueType)
	{
		case kIntegerType:		return mInteger;
		case kFloatType:		return static_cast<int>(mFloat);
		case kBooleanType:		return (true == mBoolean) ? 1 : 0;
		case kStringType:		return FromString<int>(*mString);
		case kNilType:			return 0;

		case kArrayType:		/* fall to default error case */
		case kStructureType:	/* fall to default error case */
		default:	tb_error("Cannot convert this type(%d) to an integer.\n", mValueType);
	};

	return 0;		//Statement never reached, avoids warnings.
}

//--------------------------------------------------------------------------------------------------------------------//

float tbCore::DynamicStructure::ConvertToFloat(void) const
{
	switch (mValueType)
	{
		case kFloatType:		return mFloat;
		case kIntegerType:		return static_cast<float>(mInteger);
		case kBooleanType:		return (true == mBoolean) ? 1.0f : 0.0f;
		case kNilType:			return 0.0f;

		case kArrayType:		/* fall to default error case */
		case kStructureType:	/* fall to default error case */
		default:	tb_error_if(true, "Cannot convert this type(%d) to a float.\n", mValueType);
	}

	return 0.0f;	//Statement never reached, avoids warnings.
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::ConvertToBoolean(void) const
{
	switch (mValueType)
	{
		case kBooleanType:		return mBoolean;
		case kIntegerType:		return (0 == mInteger) ? false : true;
		case kFloatType:		return (mFloat > -0.00001f && mFloat < 0.00001f) ? false : true;
		case kNilType:			return false;
		case kStringType:		return (tbCore::tbString("1") == *mString || tbCore::tbString("true") == *mString) ? true : false;

		case kArrayType:		/* fall to default error case */
		case kStructureType:	/* fall to default error case */
		default:	tb_error_if(true, "Cannot convert this type(%d) to a boolean.\n", mValueType);
	}

	return false;	//Statement never reached, avoids warnings.
}

//--------------------------------------------------------------------------------------------------------------------//

tbCore::tbString tbCore::DynamicStructure::ConvertToString(void) const
{
	switch (mValueType)
	{
		case kStringType:   return *mString;
		case kIntegerType:  { std::stringstream stream; stream << mInteger; return stream.str(); }
		case kFloatType:    { std::stringstream stream; stream << mFloat; return stream.str(); }
		case kBooleanType:  return (true == mBoolean) ? DynamicStructure::kTrueAsString : DynamicStructure::kFalseAsString;
		case kNilType:      return DynamicStructure::kNullAsString;

		case kArrayType:		/* fall to default error case */
		case kStructureType:	/* fall to default error case */
		default:	tb_error_if(true, "Cannot convert this type(%d) to a string.\n", mValueType);
	}

	return DynamicStructure::kNullAsString;	//Statement never reached, avoids warnings.
}

//--------------------------------------------------------------------------------------------------------------------//

bool tbCore::DynamicStructure::operator==(const tbCore::DynamicStructure& rhs) const
{
	if (mValueType != rhs.mValueType)
	{
		return false;
	}

	switch (mValueType)
	{
		case kNilType:			return (kNilType == rhs.mValueType) ? true : false;
		case kIntegerType:		return (AsInteger() == rhs.AsInteger()) ? true : false;
		case kFloatType:		return (fabs(AsFloat() - rhs.AsFloat()) <= kFloatElipson) ? true : false;
		case kBooleanType:		return (AsBoolean() == rhs.AsBoolean()) ? true : false;
		case kStringType:		return (AsString() == rhs.AsString()) ? true : false;

		case kArrayType:		/* fall to default error case */
		case kStructureType:	/* fall to default error case */
		default:	tb_error_if(true, "Cannot test equality for this type(%d).\n", mValueType);
	};

	return false;
}

//--------------------------------------------------------------------------------------------------------------------//
